﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Diagnostics;
using NWCore.DataModel;

namespace App.Data
{

    //-------------------------------------------------------------------------
    /// ControlWidthType
    ///
    /// <summary>Command animation edit types</summary>
    //-------------------------------------------------------------------------
    public enum CommandType
    {
        //-------------------------------------------------------------------------
        /// <summary>Unknown CommandType</summary>
        //-------------------------------------------------------------------------
        COMMAND_UNKNOWN,

        //-------------------------------------------------------------------------
        /// <summary>add graph item</summary>
        //-------------------------------------------------------------------------
        COMMAND_GRAPHITEM_ADD,

        //-------------------------------------------------------------------------
        /// <summary>remove graph item</summary>
        //-------------------------------------------------------------------------
        COMMAND_GRAPHITEM_REMOVE,

        //-------------------------------------------------------------------------
        /// <summary>paste animation keys</summary>
        //-------------------------------------------------------------------------
        COMMAND_ANIMATIONKEY_PASTE,

        //-------------------------------------------------------------------------
        /// <summary>add animation key</summary>
        //-------------------------------------------------------------------------
        COMMAND_ANIMATIONKEY_ADD,

        //-------------------------------------------------------------------------
        /// <summary>remove animation key</summary>
        //-------------------------------------------------------------------------
        COMMAND_ANIMATIONKEY_REMOVE,

        //-------------------------------------------------------------------------
        /// <summary>move animation key</summary>
        //-------------------------------------------------------------------------
        COMMAND_ANIMATIONKEY_MOVE,

        //-------------------------------------------------------------------------
        /// <summary>change the curve slope around the animation key</summary>
        //-------------------------------------------------------------------------
        COMMAND_ANIMATIONKEY_SLOPE_CHANGE,

        //-------------------------------------------------------------------------
        /// <summary>change animation key type</summary>
        //-------------------------------------------------------------------------
        COMMAND_ANIMATIONKEY_TYPE_CHANGE
    }

    /// <summary>
    /// ドキュメントプロパティ編集イベントハンドラデリゲート。
    /// </summary>
    public delegate void DocumentPropertyChangedEventHandler(object sender, DocumentPropertyChangedEventArgs args);

    /// <summary>
    /// ドキュメントプロパティ編集ハンドラデリゲート。
    /// </summary>
    public delegate void DocumentPropertyChangedHandler(object sender, DocumentPropertyChangedArgs args);

    /// <summary>
    /// ドキュメントプロパティ編集イベント引数クラス。
    /// </summary>
    public sealed class DocumentPropertyChangedEventArgs : EventArgs
    {
        // 引数リスト
        private ArrayList _argsList = new ArrayList();

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DocumentPropertyChangedEventArgs() {}

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DocumentPropertyChangedEventArgs(DocumentPropertyChangedArgs args)
        {
            AddArgs(args);
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DocumentPropertyChangedEventArgs(DocumentPropertyChangedEventArgs eventArgs)
        {
            Set(eventArgs);
        }

        /// <summary>
        /// クリア
        /// </summary>
        public void Clear()
        {
            _argsList.Clear();
        }

        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(DocumentPropertyChangedEventArgs eventArgs)
        {
            Clear();
            Merge(eventArgs);
        }

        /// <summary>
        /// マージ。
        /// </summary>
        public void Merge(DocumentPropertyChangedEventArgs eventArgs)
        {
            foreach (DocumentPropertyChangedArgs args in eventArgs.GetArgs())
            {
                AddArgs(args);
            }
        }

        //---------------------------------------------------------------------
        #region 引数
        /// <summary>
        /// 引数の追加。
        /// </summary>
        public void AddArgs(DocumentPropertyChangedArgs additionalArgs)
        {
            foreach (DocumentPropertyChangedArgs args in _argsList)
            {
                // 同値の引数は無視する
                if (args.Equals(additionalArgs)) { return; }
            }
            _argsList.Add(additionalArgs);
        }

        /// <summary>
        /// 引数の取得。
        /// </summary>
        public ArrayList GetArgs()
        {
            return (ArrayList)_argsList.Clone();
        }

        /// <summary>
        /// 引数の取得。
        /// </summary>
        public ArrayList GetArgs(Type argsType)
        {
            Debug.Assert(argsType.IsSubclassOf(typeof(DocumentPropertyChangedArgs)));
            ArrayList result = new ArrayList();
            foreach (DocumentPropertyChangedArgs args in _argsList)
            {
                if (args.GetType().Equals(argsType)) { result.Add(args); }
            }
            return result;
        }

        /// <summary>
        /// 引数を持っているか。
        /// </summary>
        public bool HasArgs() { return (_argsList.Count > 0); }

        /// <summary>
        /// 引数を持っているか。
        /// </summary>
        public bool HasArgs(Type argsType)
        {
            Debug.Assert(argsType.IsSubclassOf(typeof(DocumentPropertyChangedArgs)));
            foreach (DocumentPropertyChangedArgs args in _argsList)
            {
                if (args.GetType().Equals(argsType)) { return true; }
            }
            return false;
        }
        #endregion

        //---------------------------------------------------------------------
        #region オブジェクト
        /// <summary>
        /// オブジェクトの取得。
        /// </summary>
        public ArrayList GetObjects()
        {
            ArrayList result = new ArrayList();
            foreach (DocumentPropertyChangedArgs args in _argsList)
            {
                if (result.Contains(args.Target)) { continue; }
                result.Add(args.Target);
            }
            return result;
        }

        /// <summary>
        /// オブジェクトの取得。
        /// </summary>
        public ArrayList GetObjects(Type argsType)
        {
            Debug.Assert(argsType.IsSubclassOf(typeof(DocumentPropertyChangedArgs)));
            ArrayList result = new ArrayList();
            foreach (DocumentPropertyChangedArgs args in _argsList)
            {
                if (result.Contains(args.Target)) { continue; }
                if (args.GetType().Equals(argsType) || args.GetType().IsSubclassOf(argsType))
                {
                    result.Add(args.Target);
                }
            }
            return result;
        }

        /// <summary>
        /// オブジェクトの取得。
        /// </summary>
        public ArrayList GetObjects(GuiObjectID objectID)
        {
            ArrayList result = new ArrayList();
            foreach (DocumentPropertyChangedArgs args in _argsList)
            {
                if (result.Contains(args.Target)) { continue; }
                if (args.Target.ObjectID == objectID) { result.Add(args.Target); }
            }
            return result;
        }

        /// <summary>
        /// オブジェクトの取得。
        /// </summary>
        public ArrayList GetObjects(Type argsType, GuiObjectID objectID)
        {
            Debug.Assert(argsType.IsSubclassOf(typeof(DocumentPropertyChangedArgs)));
            ArrayList result = new ArrayList();
            foreach (DocumentPropertyChangedArgs args in _argsList)
            {
                if (result.Contains(args.Target)) { continue; }
                if (args.GetType().Equals(argsType) && (args.Target.ObjectID == objectID))
                {
                    result.Add(args.Target);
                }
            }
            return result;
        }
        #endregion
    }

    //-------------------------------------------------------------------------
    /// <summary>
    /// ドキュメントプロパティ編集引数クラス。
    /// </summary>
    public class DocumentPropertyChangedArgs
    {
        // ターゲット
        private readonly IDocument _target;


        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DocumentPropertyChangedArgs()
        {
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DocumentPropertyChangedArgs(IDocument target)
        {
            Debug.Assert(target != null);
            _target = target;
        }

        /// <summary>
        /// ターゲット。
        /// </summary>
        public IDocument Target
        {
            get { return _target; }
        }

        /// <summary>
        /// 等値比較。
        /// </summary>
        protected bool Equals(DocumentPropertyChangedArgs src)
        {
            // 必ずサブクラスからas演算子を使って呼び出す為にprotected
            if (src == null) return false;
            if (src == this) return true;
            return (Target == src.Target);
        }
    }

    //-------------------------------------------------------------------------
    /// <summary>
    /// ドキュメント変更無し編集引数クラス。
    /// </summary>
    public sealed class DocumentNothingChangedArgs : DocumentPropertyChangedArgs
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DocumentNothingChangedArgs(IDocument target) : base(target) { }

        #region 比較
        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override bool Equals(object obj)
        {
            return Equals(obj as DocumentNothingChangedArgs);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override int GetHashCode() { return base.GetHashCode(); }
        #endregion
    }

    //-------------------------------------------------------------------------
    /// <summary>
    /// ドキュメント名前編集引数クラス。
    /// </summary>
    public sealed class DocumentNameChangedArgs : DocumentPropertyChangedArgs
    {
        // 古い名前
        private readonly string _oldName;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DocumentNameChangedArgs(IDocument target, string oldName) :
            base(target)
        {
            Debug.Assert(oldName != null);
            _oldName = oldName;
        }

        /// <summary>
        /// 古い名前。
        /// </summary>
        public string OldName
        {
            get { return _oldName; }
        }

        #region 比較
        /// <summary>
        /// 等値比較。
        /// </summary>
        public bool Equals(DocumentNameChangedArgs src)
        {
            return base.Equals(src) && (OldName.Equals(src.OldName));
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override bool Equals(object obj)
        {
            return Equals(obj as DocumentNameChangedArgs);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override int GetHashCode()
        {
            return base.GetHashCode() ^ OldName.GetHashCode();
        }
        #endregion
    }

    //-------------------------------------------------------------------------
    /// <summary>
    /// ドキュメントヘッダ編集引数クラス。
    /// </summary>
    public sealed class DocumentHeaderChangedArgs : DocumentPropertyChangedArgs
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DocumentHeaderChangedArgs(IDocument target) : base(target) { }

        #region 比較
        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override bool Equals(object obj)
        {
            return Equals(obj as DocumentHeaderChangedArgs);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override int GetHashCode() { return base.GetHashCode(); }
        #endregion
    }

   	//-------------------------------------------------------------------------
    /// <summary>
    /// ドキュメント内容編集引数クラス。
    /// </summary>
    public class DocumentContentsChangedArgs : DocumentPropertyChangedArgs
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DocumentContentsChangedArgs(IDocument target) : base(target) { }

        #region 比較
        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override bool Equals(object obj)
        {
            return Equals(obj as DocumentContentsChangedArgs);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override int GetHashCode() { return base.GetHashCode(); }
        #endregion
    }

    //-------------------------------------------------------------------------
    /// <summary>
    /// ドキュメント内容編集引数クラス。
    /// </summary>
    //-------------------------------------------------------------------------
    public sealed class DocumentGraphPropertyChangedArgs : DocumentPropertyChangedArgs
    {
        //---------------------------------------------------------------------
        /// <summary>変更対象のオブジェクトです。</summary>
        //---------------------------------------------------------------------
        public object m_target;

        //---------------------------------------------------------------------
        /// <summary>変更対象のオブジェクトです。</summary>
        //---------------------------------------------------------------------
        public CommandType m_commandType;

        //-------------------------------------------------------------------------
        /// <summary>
        /// Constructor
        /// </summary>
        //-------------------------------------------------------------------------
        public DocumentGraphPropertyChangedArgs( object target, CommandType type ) : base()
        {
            this.m_target = target;
            this.m_commandType = type;
        }
    }
}
