﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Windows.Forms;
using App.IO;
using NWCore.DataModel;

namespace App.Data
{
    /// <summary>
    /// ドキュメントクラス。
    /// </summary>
    public abstract class Document : IDocument
    {
        #region Member variables

        public const string ComplexEmitterKey       = "Deluxe";
        public const string UnreachableAssetKey     = "NoTex";
        public const string ActiveIconKey           = "Active";
        public const string NonactiveIconKey        = "NonActive";
        public const string TextureTypeMismatchKey  = "TexTypeMismatch";

        /// <summary>ドキュメント構成変更イベント。</summary>
        public static event DocumentComponentChangedEventHandler ComponentChanged = null;
        /// <summary>ドキュメントプロパティ変更イベント。</summary>
        public static event DocumentPropertyChangedEventHandler PropertyChanged = null;

        // Unique document ID.
        private int _iDocumentID = 0;
        // ファイルロケーション
        private string _fileLocation = string.Empty;
        // 編集状態かどうか
        private int _iSelfModifyCount  = 0;
        private int _iChildModifyCount = 0;
        private bool _bRemoved = true;

        private int _iSelfUnreachableAssetCount = 0;
        private int _iChildUnreachableAssetCount = 0;

        private int _iSelfTypeMismatchTextureCount = 0;
        private int _iChildTypeMismatchTextureCount = 0;

        #endregion

        //---------------------------------------------------------------------

        /// <summary>
        /// タイプ・コンストラクタ。
        /// </summary>
        static Document()
        {
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public Document()
        {
            _iDocumentID = DocumentManager.GetUniqueDocumentID();
            this.Comment = string.Empty;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public Document(string name)
        {
            _iDocumentID = DocumentManager.GetUniqueDocumentID();

            this.Name    = name;
            this.Enable  = true;
            this.Comment = string.Empty;
        }

        /// <summary>
        /// Get the unique ID for the document.
        /// </summary>
        public int DocumentID
        {
            get { return _iDocumentID; }
        }

        /// <summary>
        /// Get the remove flag.
        /// </summary>
        public bool IsRemoved
        {
            get { return _bRemoved; }
        }

        /// <summary>
        /// ドキュメント更新。
        /// </summary>
        public virtual void UpdateDocument()
        {
        }

        #region Notifications

        /// <summary>
        /// ドキュメント構成変更通知。
        /// </summary>
        public static void NotifyComponentChanged(object sender, DocumentComponentChangedEventArgs e)
        {
            if (!e.HasChanged())
            {
                return;
            }

            DebugConsole.WriteLine("event:DocumentComponentChanged opened={0} closed={1}",
                e.OpenedDocuments.Count, e.ClosedDocuments.Count);

            if (ComponentChanged != null)
            {
                ComponentChanged(sender, e);
            }
        }

        /// <summary>
        /// ドキュメントプロパティ変更通知。
        /// </summary>
        public static void NotifyPropertyChanged(object sender, DocumentPropertyChangedArgs args)
        {
            NotifyPropertyChanged(sender, new DocumentPropertyChangedEventArgs(args));
        }

        /// <summary>
        /// ドキュメントプロパティ変更通知。
        /// </summary>
        public static void NotifyPropertyChanged(object sender, DocumentPropertyChangedEventArgs args)
        {
            if (!args.HasArgs())
            {
                return;
            }

            // 抑制処理
            if (PropertyChangedSuppressBlock.Notify(args))
            {
                // ＧＵＩ更新を抑制します。
                return;
            }

            DebugConsole.WriteLine("event:DocumentPropertyChanged args={0} objects={1}",
                args.GetArgs().Count, args.GetObjects().Count);

            if (PropertyChanged != null)
            {
                PropertyChanged(sender, args);
            }
        }

        /// <summary>
        /// Notify this document that it was added to its parent.
        /// </summary>
        public virtual void NotifyAddedToParent()
        {
            _bRemoved = false;
        }

        /// <summary>
        /// Notify this document that it was removed from its parent.
        /// </summary>
        public virtual void NotifyRemovedFromParent()
        {
            DocumentManager.NotifyDocumentRemoved( this );
            _bRemoved = true;
        }

        public void NotifyDocumentSaveFinished(bool isError)
        {
            OnDocumentSaveFinished(isError);
        }

        protected virtual void OnDocumentSaveFinished(bool isError)
        {
        }

        #endregion

        #region シリアライズ関係

        /// <summary>
        /// ストリームにドキュメントの書き込みします。
        /// </summary>
        /// <param name="stream">保存先のストリーム。</param>
        /// <param name="bClearModifyFlag">
        /// Trueを指定したら、ドキュメントの変更のフラグを削除します。
        /// </param>
        /// <param name="bNormalizePath">
        /// Trueを指定場合、ドキュメントは記載するファイルパスを相対パスに転換する。
        /// </param>
        /// <returns>成功するとtrueを戻します。</returns>
        public virtual bool SaveDocumentToStream( Stream stream,
                                                  bool bClearModifyFlag = true,
                                                  bool bNormalizePath = true )
        {
            return false;
        }

        /// <summary>
        /// ドキュメントの書き込みします。
        /// </summary>
        /// <param name="filePath">保存先のファイルパス</param>
        /// <param name="bClearModifyFlag">
        /// Trueを指定したら、ドキュメントの変更のフラグを削除します。
        /// </param>
        /// <param name="bNormalizePath">
        /// Trueを指定場合、ドキュメントは記載するファイルパスを相対パスに転換する。
        /// </param>
        /// <returns>成功するとtrueを戻します。</returns>
        public virtual bool SaveDocument( string filePath,
                                          bool bClearModifyFlag = true,
                                          bool bNormalizePath = true )
        {
            try
            {
                using ( FileStream stream = new FileStream(filePath, FileMode.Create) )
                {
                    if ( SaveDocumentToStream(stream,
                                              bClearModifyFlag,
                                              bNormalizePath)==false )
                    {
                        return false;
                    }
                }
            }
            catch ( Exception ex )
            {
                TheApp.OutputLogMsg( NWCore.LogLevels.Error,
                                     ex.Message );
                return false;
            }

            return true;
        }

        /// <summary>
        /// ドキュメントをバイナリー保存します。
        /// </summary>
        /// <returns>=true..成功</returns>
        public virtual bool SaveBinary(FileStream writer, bool isOptimize)
        {
            return true;
        }

        #endregion

        #region アトリビュート

        /// <summary>
        /// 名前
        /// </summary>
        public virtual string Name
        {
            get;
            set;
        }

        /// <summary>
        /// ToString
        /// </summary>
        public override string ToString()
        {
            return this.Name;
        }

        /// <summary>
        /// オブジェクトＩＤ
        /// </summary>
        public abstract GuiObjectID ObjectID
        {
            get;
        }

        /// <summary>
        /// アイコンのイメージキー
        /// </summary>
        public virtual string ImageKey
        {
            get { return this.ObjectID.ToString(); }
        }

        /// <summary>
        /// 有効
        /// </summary>
        public virtual bool Enable
        {
            get;
            set;
        }

        /// <summary>
        /// サブ・ドキュメント数を取得します。
        /// </summary>
        public virtual int Count
        {
            get { return 0; }
        }

        /// <summary>
        /// サブ・ドキュメントを取得します。
        /// </summary>
        public virtual IEnumerable<IDocument> SubDocuments
        {
            get { return null; }
        }

        /// <summary>
        /// オーバーライド：オーナードキュメントです。
        /// </summary>
        public abstract IDocument OwnerDocument
        {
            get;
        }

        /// <summary>
        /// オーバーライド：シンボルカラーを許可・禁止します。
        /// </summary>
        public virtual bool IsSymbolColorEnabled
        {
            get { return false; }
        }

        /// <summary>
        /// オーバーライド：コメントです。
        /// </summary>
        public virtual string Comment
        {
            get;
            set;
        }

        #region Modifications

        /// <summary>
        /// 編集状態かどうか。
        /// </summary>
        public virtual bool Modified
        {
            get { return ( this.ModifyCount!=0 ); }
        }

        /// <summary>
        /// The number of the modify flags that has been set.
        /// </summary>
        public int ModifyCount
        {
            get { return SelfModifyCount + ChildModifyCount; }
        }

        /// <summary>
        /// The number of the modify flags that has been set from the document itself.
        /// </summary>
        protected int SelfModifyCount
        {
            get { return _iSelfModifyCount; }
            set { _iSelfModifyCount = value; }
        }

        /// <summary>
        /// The number of the modify flags that has been set from the child documents.
        /// </summary>
        public int ChildModifyCount
        {
            get { return _iChildModifyCount; }
            protected set { _iChildModifyCount = value; }
        }

        /// <summary>
        /// Internal flag indicating if the modify flag should be notified to parent document.
        /// </summary>
        protected virtual bool ShouldNotifyParentDocModified
        {
            get { return true; }
        }

        #endregion

        #region Modification state

        /// <summary>
        /// Notify that a modification has been done to the document.
        /// </summary>
        /// <param name="bUpdateProjectTree">True to update the project tree.</param>
        public virtual void SetModified(bool bUpdateProjectTree = true)
        {
            bool bOrigModifyFlag = this.Modified;

            ++SelfModifyCount;

            bool bNeedUpdate = false;
            Document parent = this.OwnerDocument as Document;
            if (this.ShouldNotifyParentDocModified == true &&
                 parent != null)
            {
                bNeedUpdate = parent.NotifyChildModified(this);
            }

            bNeedUpdate |= (this.Modified != bOrigModifyFlag);

            if (bUpdateProjectTree == true &&
                 bNeedUpdate == true)
            {
                ////App.PropertyEdit.ObjectPropertyDialog.UpdateProjectTree();
            }
        }


        /// <summary>
        /// Notify that a modification has been undone to the document.
        /// </summary>
        /// <param name="bUpdateProjectTree">True to update the project tree.</param>
        public virtual void UndoModify(bool bUpdateProjectTree = true)
        {
            bool bOrigModifyFlag = this.Modified;

            --SelfModifyCount;

            bool bNeedUpdate = false;
            Document parent = this.OwnerDocument as Document;
            if (this.ShouldNotifyParentDocModified == true &&
                 parent != null)
            {
                bNeedUpdate = parent.NotifyChildUndoModify(this);
            }

            bNeedUpdate |= (this.Modified != bOrigModifyFlag);

            if (bUpdateProjectTree == true &&
                 bNeedUpdate == true)
            {
                ////TheApp.MainFrame.RequestRebuildNodes();
            }
        }


        /// <summary>
        /// Clear the modified flag.
        /// </summary>
        /// <param name="bUpdateProjectTree">True to update the project tree.</param>
        /// <returns>Number of cleared flag counts.</returns>
        public virtual int ClearModifiedFlag(bool bUpdateProjectTree)
        {
            bool bOrigModifyFlag = this.Modified;

            int iNumFlags = SelfModifyCount + ChildModifyCount;

            SelfModifyCount = 0;
            ChildModifyCount = 0;

            if (bUpdateProjectTree == true &&
                 this.Modified != bOrigModifyFlag)
            {
                ////TheApp.MainFrame.RequestRebuildNodes();
            }

            return iNumFlags;
        }


        /// <summary>
        /// Notify that a child document has been modified.
        /// </summary>
        /// <returns>True if project tree needs to be updated.</returns>
        protected virtual bool NotifyChildModified(Document child)
        {
            bool bOrigModifyFlag = this.Modified;

            ++ChildModifyCount;

            bool bNeedUpdate = (this.Modified != bOrigModifyFlag);

            Document parent = this.OwnerDocument as Document;
            if (parent != null)
                bNeedUpdate |= parent.NotifyChildModified(this);

            return bNeedUpdate;
        }


        /// <summary>
        /// Notify that a modification to a child document has been undone.
        /// </summary>
        /// <returns>True if project tree needs to be updated.</returns>
        protected virtual bool NotifyChildUndoModify(Document child)
        {
            bool bOrigModifyFlag = this.Modified;

            --ChildModifyCount;

            bool bNeedUpdate = (this.Modified != bOrigModifyFlag);

            Document parent = this.OwnerDocument as Document;
            if (parent != null)
                bNeedUpdate |= parent.NotifyChildUndoModify(this);

            return bNeedUpdate;
        }

        #endregion

        #region Unreachable assets

        /// <summary>
        /// Is at least one of the assets unreachable?
        /// (missing file, wrong path, etc.)
        /// </summary>
        public bool HasUnreachableAsset
        {
            get { return (this.UnreachableAssetCount > 0); }
        }

        /// <summary>
        /// The number of the unreachable asset flags that has been set.
        /// </summary>
        protected int UnreachableAssetCount
        {
            get { return _iSelfUnreachableAssetCount + _iChildUnreachableAssetCount; }
        }

        /// <summary>
        /// The number of the unreachable asset flags that has been set from the document itself.
        /// </summary>
        public int SelfUnreachableAssetCount
        {
            get { return _iSelfUnreachableAssetCount; }
            set { _iSelfUnreachableAssetCount = value; }
        }

        /// <summary>
        /// The number of the unreachable asset flags that has been set from the child documents.
        /// </summary>
        public int ChildUnreachableAssetCount
        {
            get { return _iChildUnreachableAssetCount; }
            set { _iChildUnreachableAssetCount = value; }
        }

        /// <summary>
        /// Internal flag indicating if the unreachable asset flag should be notified to parent document.
        /// </summary>
        protected virtual bool ShouldNotifyParentUnreachableAsset
        {
            get { return false; }
        }

        /// <summary>
        /// </summary>
        public bool HasTypeMismatchTexture
        {
            get { return (this.TypeMismatchTextureCount > 0); }
        }

        /// <summary>
        /// </summary>
        protected int TypeMismatchTextureCount
        {
            get { return _iSelfTypeMismatchTextureCount + _iChildTypeMismatchTextureCount; }
        }

        /// <summary>
        /// </summary>
        public int SelfTypeMismatchTextureCount
        {
            get { return _iSelfTypeMismatchTextureCount; }
            set { _iSelfTypeMismatchTextureCount = value; }
        }

        /// <summary>
        /// </summary>
        public int ChildTypeMismatchTextureCount
        {
            get { return _iChildTypeMismatchTextureCount; }
            set { _iChildTypeMismatchTextureCount = value; }
        }

        /// <summary>
        /// </summary>
        protected virtual bool ShouldNotifyParentTypeMismatchTextureCount
        {
            get { return false; }
        }

        #endregion

        #region Unreachable assets state

        /// <summary>
        /// Checks if the assets are reachable, and report it otherwise
        /// </summary>
        public virtual void CheckAssetsReachability(bool bUpdateProjectTree = true)
        {
        }

        /// <summary>
        /// Notify that a asset reachability has changed.
        /// </summary>
        /// <param name="bUpdateProjectTree">True to update the project tree.</param>
        public void UpdateUnreachableAssetCount(int selfUnreachableAssetsCount, bool bUpdateProjectTree = true)
        {
            bool hadUnreachableAsset = this.HasUnreachableAsset;

            this.SelfUnreachableAssetCount = selfUnreachableAssetsCount;

            bool bNeedUpdate = false;
            Document parent = this.OwnerDocument as Document;
            if (this.ShouldNotifyParentUnreachableAsset == true && parent != null)
            {
                bNeedUpdate = parent.NotifyChildAssetReachability(this, UnreachableAssetCount);
            }

            bNeedUpdate |= (this.HasUnreachableAsset != hadUnreachableAsset);

            if (bUpdateProjectTree == true && bNeedUpdate == true)
            {
                ////App.PropertyEdit.ObjectPropertyDialog.UpdateProjectTree();
            }
        }

        /// <summary>
        /// Notify that a child asset reachability has changed.
        /// </summary>
        /// <returns>True if project tree needs to be updated.</returns>
        protected bool NotifyChildAssetReachability(Document child, int childUnreachableAssetsCount)
        {
            bool bOrigAssetReachabilityFlag = this.HasUnreachableAsset;

            this.ChildUnreachableAssetCount = childUnreachableAssetsCount;

            bool bNeedUpdate = (this.HasUnreachableAsset != bOrigAssetReachabilityFlag);

            Document parent = this.OwnerDocument as Document;
            if (parent != null)
                bNeedUpdate |= parent.NotifyChildAssetReachability(this, UnreachableAssetCount);

            return bNeedUpdate;
        }

        /// <summary>
        ///
        /// </summary>
        public void UpdateTypeMismatchTextureCount(int typeMismatchTextureCount, bool bUpdateProjectTree = true)
        {
            bool hadTypeMismatchTexture = this.HasTypeMismatchTexture;

            this.SelfTypeMismatchTextureCount = typeMismatchTextureCount;

            bool bNeedUpdate = false;
            Document parent = this.OwnerDocument as Document;
            if (this.ShouldNotifyParentUnreachableAsset == true && parent != null)
            {
                bNeedUpdate = parent.NotifyChildTextureTypeMismatch(this, TypeMismatchTextureCount);
            }

            bNeedUpdate |= (this.HasTypeMismatchTexture != hadTypeMismatchTexture);
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="child"></param>
        /// <param name="childTypeMismatchTextureCount"></param>
        /// <returns></returns>
        public bool NotifyChildTextureTypeMismatch(Document child, int childTypeMismatchTextureCount)
        {
            bool bOrigFlag = this.HasTypeMismatchTexture;

            this.ChildTypeMismatchTextureCount = childTypeMismatchTextureCount;

            bool bNeedUpdate = (this.HasTypeMismatchTexture != bOrigFlag);

            Document parent = this.OwnerDocument as Document;
            if (parent != null)
                bNeedUpdate |= parent.NotifyChildTextureTypeMismatch(this, TypeMismatchTextureCount);

            return bNeedUpdate;
        }

        #endregion

        #region File

        /// <summary>
        /// ファイルロケーション（ファイルパスのディレクトリー部分）です
        /// </summary>
        public virtual string FileLocation
        {
            get { return _fileLocation; }
        }

        /// <summary>
        /// ファイルロケーションを設定します。
        /// </summary>
        public virtual void SetFileLocation(string fileLocation)
        {
            Debug.Assert(fileLocation != null);
            _fileLocation = fileLocation;
        }

        /// <summary>
        /// ファイルロケーションをクリアします。
        /// </summary>
        public virtual void ClearFileLocation()
        {
            _fileLocation = String.Empty;
        }

        /// <summary>
        /// DataScrPath
        /// </summary>
        public virtual string DataScrPath
        {
            get
            {
                return "Document_uid" + this.DocumentID.ToString();
            }
        }

        /// <summary>
        /// Data path for displaying, not the actual data source path.
        /// </summary>
        public virtual string DisplayDataPath
        {
            get
            {
                if ( this.OwnerDocument==null )
                {
                    return this.Name;
                }
                else
                {
                    string parentDisplayPath = this.OwnerDocument.DisplayDataPath;
                    if ( parentDisplayPath!=null &&
                         parentDisplayPath.Length>0 )
                    {
                        return parentDisplayPath + "." + this.Name;
                    }
                    else
                    {
                        return this.Name;
                    }
                }
            }
        }

        /// <summary>
        /// data source path without data model
        /// </summary>
        public virtual string RelativeDataScrPath
        {
            get
            {
                return "Document_uid" + this.DocumentID.ToString();
            }
        }

        /// <summary>
        /// ファイル拡張子。
        /// </summary>
        public virtual string FileExt
        {
            get { return String.Empty; }
        }

        /// <summary>
        /// バイナリファイル拡張子。
        /// </summary>
        public virtual string BinaryFileExt
        {
            get { return String.Empty; }
        }

        /// <summary>
        /// ファイルフィルタ。
        /// </summary>
        public virtual string FileFilter
        {
            get { return String.Empty; }
        }

        /// <summary>
        /// バイナリファイルフィルタ。
        /// </summary>
        public virtual string BinaryFileFilter
        {
            get { return String.Empty; }
        }

        /// <summary>
        /// ファイル名にタイトル（ファイル名の拡張子の無い名前の部分です。）
        /// </summary>
        public virtual string FileTitle
        {
            get { return this.Name; }
            set { this.Name = value; }
        }

        /// <summary>
        /// ファイルパス。
        /// </summary>
        public virtual string FilePath
        {
            get
            {
                if (_fileLocation != string.Empty)
                {
                    return Path.Combine(_fileLocation, this.FileName);
                }
                else
                {
                    return this.FileName;
                }
            }
        }

        /// <summary>
        /// ファイル名。
        /// </summary>
        public virtual string FileName
        {
            get { return this.FileTitle + "." + this.FileExt; }
        }

        /// <summary>
        /// バイナリファイル名。
        /// </summary>
        public virtual string BinaryFileName
        {
            get { return this.FileTitle + "." + this.BinaryFileExt; }
        }

        /// <summary>
        /// ファイルが存在するか
        /// </summary>
        public bool FileExists
        {
            get
            {
                return (this.FileLocation != string.Empty) && File.Exists(this.FilePath);
            }
        }

        #endregion

        /// <summary>
        /// サブドキュメントを検索します。
        /// </summary>
        public virtual IDocument FindChild(string name)
        {
            return null;
        }

        #endregion

        #region プロパティ変更抑制ブロック

        /// <summary>
        /// プロパティ変更抑制ブロック
        /// </summary>
        public sealed class PropertyChangedSuppressBlock : IDisposable
        {
            private static readonly List<PropertyChangedSuppressBlock> m_instances =
                new List<PropertyChangedSuppressBlock>();
            private static readonly DocumentPropertyChangedEventArgs m_args =
                new DocumentPropertyChangedEventArgs();

            /// <summary>
            /// コンストラクタ。
            /// </summary>
            public PropertyChangedSuppressBlock()
            {
                m_instances.Add(this);
            }

            /// <summary>
            /// 開放処理。
            /// </summary>
            public void Dispose()
            {
                Debug.Assert(m_instances.Count > 0);
                m_instances.Remove(this);

                if (m_instances.Count == 0)
                {
                    // イベントの発行
                    Document.NotifyPropertyChanged(null, m_args);
                    m_args.Clear();
                }
            }

            /// <summary>
            /// 通知
            /// </summary>
            public static bool Notify(DocumentPropertyChangedEventArgs args)
            {
                if (m_instances.Count > 0)
                {
                    m_args.Merge(args);
                    return true;
                }
                return false;
            }
        }
        #endregion
    }
}
