﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;

using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.DataTypes;
using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelMaker.UILogic.ViewModels
{
    /// <summary>
    /// View model class for the binary conversion source data model property definition.
    /// </summary>
    public class SourcePropertyInstanceViewModel : ViewModelBase
    {
        /// <summary>The parent data model instance view model.</summary>
        private SourceDataModelInstanceViewModel parentDataModelInstanceVM;

        /// <summary>The binary conversion source property instance.</summary>
        private SourcePropertyInstanceDefinition propertyInstance = null;

        /// <summary>The data model property definition for the instance.</summary>
        private EditorDataModelPropertyDefinition propertyDefinition = null;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public SourcePropertyInstanceViewModel(
            SourceDataModelInstanceViewModel parent,
            SourcePropertyInstanceDefinition dataModel) :
            base(parent, dataModel)
        {
            this.parentDataModelInstanceVM = parent;
            this.propertyInstance = dataModel;

            // Find the data model property definition with the Guid.
            this.propertyDefinition =
                WorkspaceManager.FindDefinition(dataModel.PropertyDefinitionGuid) as EditorDataModelPropertyDefinition;

            // Create view model for the underlying data model instance.
            if (dataModel.DataModelInstance != null)
            {
                this.UnderlyingDataModelInstance =
                    new SourceDataModelInstanceViewModel(this, dataModel.DataModelInstance);
            }
            else
            {
                this.UnderlyingDataModelInstance = null;
            }
        }

        /// <summary>
        /// Get or set the underlying data model instance view model.
        /// </summary>
        public SourceDataModelInstanceViewModel UnderlyingDataModelInstance { get; set; }

        /// <summary>
        /// Get the description of the property.
        /// </summary>
        public string PropertyDescription
        {
            get
            {
                if (this.propertyDefinition == null)
                {
                    return string.Empty;
                }

                return this.propertyDefinition.Description;
            }
        }

        /// <summary>
        /// Get the type of the property without it's namespace.
        /// </summary>
        public string PropertyType
        {
            get
            {
                if (this.propertyDefinition == null)
                {
                    return string.Empty;
                }

                if (string.IsNullOrEmpty(this.propertyDefinition.Type) == true)
                {
                    return string.Empty;
                }

                // Find the information of the property value type.
                EditorTypeInfo primaryInfo = TypeManager.FindEditorType(
                    this.propertyDefinition.Namespace,
                    this.propertyDefinition.Type);
                if (primaryInfo == null)
                {
                    return this.propertyDefinition.Type;
                }

                // Compose the complete type string.
                string typeString = primaryInfo.TypeName;
                if (primaryInfo.IsGeneric == true)
                {
                    EditorTypeInfo elementInfo = TypeManager.FindEditorType(
                        this.propertyDefinition.ElementNamespace,
                        this.propertyDefinition.ElementType);
                    if (elementInfo != null)
                    {
                        typeString += "<" + elementInfo.TypeName + ">";
                    }
                    else
                    {
                        typeString += "<" + this.propertyDefinition.ElementType + ">";
                    }
                }

                return typeString;
            }
        }

        /// <summary>
        /// Get the name of the property.
        /// </summary>
        public string PropertyName
        {
            get
            {
                if (this.propertyDefinition == null)
                {
                    return string.Empty;
                }

                return this.propertyDefinition.Name;
            }
        }
    }
}
