﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;

using EffectMaker.DataModelMaker.Core.DataTypes;
using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelMaker.UILogic.ViewModels
{
    /// <summary>
    /// View model class for the Runtime data model property definition.
    /// </summary>
    public class RuntimeDataModelPropertyDefinitionViewModel : ViewModelBase
    {
        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public RuntimeDataModelPropertyDefinitionViewModel(
            ViewModelBase parent,
            RuntimeDataModelPropertyDefinition dataModel) :
            base(parent, dataModel)
        {
            this.OnCommitEditingExecutable = new AnonymousExecutable(this.OnCommitEditing);
            this.OnCancelEditingExecutable = new AnonymousExecutable(this.OnCancelEditing);
        }

        /// <summary>
        /// Static event triggered when the data model property is modified.
        /// </summary>
        public static event EventHandler DataModelPropertyModified = null;

        /// <summary>
        /// Get or set the type of the property without the namespace.
        /// </summary>
        public string PropertyTypeWithoutNamespace
        {
            get
            {
                string name = string.Empty;

                var myDataModel = this.DataModel as RuntimeDataModelPropertyDefinition;
                if (myDataModel != null)
                {
                    TypeManager.ComposeRuntimeTypeName(
                        myDataModel.TypeNamespace,
                        myDataModel.TypeName,
                        out name);

                    int arraySize;
                    if (int.TryParse(myDataModel.ArraySize, out arraySize) == true &&
                        arraySize > 1)
                    {
                        name += "[" + myDataModel.ArraySize + "]";
                    }
                    else if (string.IsNullOrEmpty(myDataModel.ArraySize) == false)
                    {
                        name += "[" + myDataModel.ArraySize + "]";
                    }
                }

                return name;
            }
        }

        /// <summary>
        /// Get or set the type of the property.
        /// </summary>
        public string PropertyType
        {
            get
            {
                string name = string.Empty;

                var myDataModel = this.DataModel as RuntimeDataModelPropertyDefinition;
                if (myDataModel != null)
                {
                    TypeManager.ComposeRuntimeTypeFullName(
                        myDataModel.TypeNamespace,
                        myDataModel.TypeName,
                        out name);
                }

                return name;
            }

            set
            {
                var myDataModel = this.DataModel as RuntimeDataModelPropertyDefinition;
                if (myDataModel == null)
                {
                    return;
                }

                string typeNamespace;
                string typeName;
                bool result = TypeManager.ParseRuntimeType(
                    value,
                    out typeNamespace,
                    out typeName);
                if (result == false)
                {
                    this.OnPropertyChanged("IsPropertyTypeInvalid");
                    return;
                }

                myDataModel.TypeNamespace = typeNamespace;
                myDataModel.TypeName = typeName;

                this.OnPropertyChanged("PropertyType");
                this.OnPropertyChanged("PropertyTypeWithoutNamespace");
                this.OnPropertyChanged("IsPropertyTypeInvalid");
            }
        }

        /// <summary>
        /// Get or set the type of the property.
        /// </summary>
        public string PropertyTypeWithArraySize
        {
            get
            {
                string name = this.PropertyType;

                var myDataModel = this.DataModel as RuntimeDataModelPropertyDefinition;
                if (myDataModel != null)
                {
                    int arraySize;
                    if (int.TryParse(myDataModel.ArraySize, out arraySize) == true &&
                        arraySize > 1)
                    {
                        name += "[" + myDataModel.ArraySize + "]";
                    }
                    else if (string.IsNullOrEmpty(myDataModel.ArraySize) == false)
                    {
                        name += "[" + myDataModel.ArraySize + "]";
                    }
                }

                return name;
            }
        }

        /// <summary>
        /// Get the flag indicating whether the property type is invalid or not.
        /// </summary>
        public bool IsPropertyTypeInvalid
        {
            get
            {
                var myDataModel = this.DataModel as RuntimeDataModelPropertyDefinition;
                if (myDataModel == null)
                {
                    return true;
                }

                if (string.IsNullOrEmpty(myDataModel.TypeName) == true)
                {
                    return true;
                }

                // Find the information of the property value type.
                RuntimeTypeInfo info =
                    TypeManager.FindRuntimeType(myDataModel.TypeNamespace, myDataModel.TypeName);
                if (info == null)
                {
                    return true;
                }

                return false;
            }
        }

        /// <summary>
        /// Get or set the executable for committing modification of the property.
        /// </summary>
        public IExecutable OnCommitEditingExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for canceling modification of the property.
        /// </summary>
        public IExecutable OnCancelEditingExecutable { get; set; }

        /// <summary>
        /// Handle execution when committing modification of the property.
        /// </summary>
        /// <param name="parameter">The execution parameters.</param>
        private void OnCommitEditing(object parameter)
        {
            var values = parameter as KeyValuePair<string, object>[];
            if (values == null)
            {
                return;
            }

            try
            {
                foreach (var pair in values)
                {
                    var binder = new EffectMakerSetMemberBinder(pair.Key, true);
                    this.TrySetMember(binder, pair.Value);
                }

                var myParent = this.Parent as RuntimeDataModelDefinitionViewModel;
                if (myParent != null)
                {
                    myParent.NotifyPropertyDefinitionChanged(this);
                }

                if (DataModelPropertyModified != null)
                {
                    DataModelPropertyModified(this, EventArgs.Empty);
                }
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "RuntimeDataModelPropertyDefinitionViewModel.OnCommitEditing => Failed setting values to the runtime data model property.");
                Logger.Log(LogLevels.Error, "Exception : {0}, message : {1}", ex.GetType().ToString(), ex.Message);
            }
        }

        /// <summary>
        /// Handle execution when canceling modification of the property.
        /// </summary>
        /// <param name="parameter">The execution parameters.</param>
        private void OnCancelEditing(object parameter)
        {
            var myParent = this.Parent as RuntimeDataModelDefinitionViewModel;
            if (myParent == null)
            {
                return;
            }

            myParent.DeleteProperty(this);
        }
    }
}
