﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;

using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.DataTypes;
using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelMaker.UILogic.ViewModels
{
    /// <summary>
    /// View model class for the runtime data model definition.
    /// </summary>
    public class RuntimeDataModelDefinitionViewModel : ViewModelBase
    {
        /// <summary>The runtime data model definition view models.</summary>
        private List<RuntimeDataModelPropertyDefinitionViewModel> propertyViewModels =
            new List<RuntimeDataModelPropertyDefinitionViewModel>();

        /// <summary>The data model include definition view models.</summary>
        private List<RuntimeDataModelIncludeDefinitionViewModel> includeDefViewModels =
            new List<RuntimeDataModelIncludeDefinitionViewModel>();

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public RuntimeDataModelDefinitionViewModel(
            ViewModelBase parent,
            RuntimeDataModelDefinition dataModel) :
            base(parent, dataModel)
        {
            this.OnCommitEditingExecutable = new AnonymousExecutable(this.OnCommitEditing);
            this.OnCancelEditingExecutable = new AnonymousExecutable(this.OnCancelEditing);
            this.OnCreatePropertyExecutable = new AnonymousExecutable(this.OnCreateProperty);
            this.OnDeletePropertyExecutable = new AnonymousExecutable(this.OnDeleteProperty);
            this.OnMovePropertyExecutable = new AnonymousExecutable(this.OnMoveProperty);
            this.OnCreateIncludeDefinitionExecutable = new AnonymousExecutable(this.OnCreateIncludeDefinition);
            this.OnDeleteIncludeDefinitionExecutable = new AnonymousExecutable(this.OnDeleteIncludeDefinition);

            foreach (RuntimeDataModelPropertyDefinition def in dataModel.Properties)
            {
                var child = new RuntimeDataModelPropertyDefinitionViewModel(this, def);
                this.Children.Add(child);
                this.propertyViewModels.Add(child);
            }

            foreach (RuntimeDataModelIncludeDefinition def in dataModel.Includes)
            {
                var child = new RuntimeDataModelIncludeDefinitionViewModel(this, def);
                this.Children.Add(child);
                this.includeDefViewModels.Add(child);
            }
        }

        /// <summary>
        /// Static event triggered when the data model is modified.
        /// </summary>
        public static event EventHandler DataModelModified = null;

        /// <summary>
        /// Static event triggered when the data model property list is changed.
        /// </summary>
        public static event EventHandler DataModelPropertyListChanged = null;

        /// <summary>
        /// Get the data model name.
        /// </summary>
        public string DataModelName
        {
            get { return ((RuntimeDataModelDefinition)this.DataModel).Name; }
        }

        /// <summary>
        /// Get the data model namespace.
        /// </summary>
        public string DataModelNamespace
        {
            get { return ((RuntimeDataModelDefinition)this.DataModel).Namespace; }
        }

        /// <summary>
        /// Enumerate the property definition view models.
        /// </summary>
        public IEnumerable<RuntimeDataModelPropertyDefinitionViewModel> PropertyDefinitionViewModels
        {
            get { return this.propertyViewModels; }
        }

        /// <summary>
        /// Enumerate the include definition view models.
        /// </summary>
        public IEnumerable<RuntimeDataModelIncludeDefinitionViewModel> IncludeDefinitionViewModels
        {
            get { return this.includeDefViewModels; }
        }

        /// <summary>
        /// Enumerate the the list of include definitions.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> IncludeDefinitionNameValuePairList
        {
            get
            {
                foreach (dynamic child in this.includeDefViewModels)
                {
                    yield return new KeyValuePair<string, object>(child.File, child);
                }
            }
        }

        /// <summary>
        /// Enumerate the the list of property the data model has.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> PropertyAndTypeNameValuePairList
        {
            get
            {
                foreach (dynamic child in this.propertyViewModels)
                {
                    yield return new KeyValuePair<string, object>(child.PropertyTypeWithoutNamespace + " " + child.Name, child);
                }
            }
        }

        /// <summary>
        /// Get the comma-separated string of the name (with namespace) of the inheriting data models.
        /// </summary>
        public string InheritingDataModels
        {
            get
            {
                // Get the data model.
                var myDataModel = this.DataModel as RuntimeDataModelDefinition;
                if (myDataModel == null)
                {
                    return string.Empty;
                }

                // Get the Guid of the inheriting data models.
                var guidList = myDataModel.SuperClassGuidList;
                if (guidList == null)
                {
                    return string.Empty;
                }

                string str = string.Empty;
                foreach (Guid guid in guidList)
                {
                    // Find the inheriting data model definition.
                    var def =
                        WorkspaceManager.FindDefinition(guid) as RuntimeDataModelDefinition;
                    if (def != null)
                    {
                        if (str.Length > 0)
                        {
                            str += ", ";
                        }

                        // Return the inheriting data model name.
                        if (string.IsNullOrEmpty(def.Namespace) == true)
                        {
                            str += def.Name;
                        }
                        else
                        {
                            str += def.Namespace + "::" + def.Name;
                        }
                    }
                }

                return str;
            }
        }

        /// <summary>
        /// Enumerate the name (without namespace) of the inheriting data models.
        /// </summary>
        public IEnumerable<string> InheritingDataModelsWithoutNamespace
        {
            get
            {
                // Get the data model.
                var myDataModel = this.DataModel as RuntimeDataModelDefinition;
                if (myDataModel == null)
                {
                    yield break;
                }

                // Get the Guid of the inheriting data models.
                var guidList = myDataModel.SuperClassGuidList;
                if (guidList == null)
                {
                    yield break;
                }

                foreach (Guid guid in guidList)
                {
                    // Find the inheriting data model definition.
                    var def =
                        WorkspaceManager.FindDefinition(guid) as RuntimeDataModelDefinition;
                    if (def != null)
                    {
                        // Return the inheriting data model name.
                        yield return def.Name;
                    }
                }
            }
        }

        /// <summary>
        /// Get the last created data model property view model.
        /// </summary>
        public RuntimeDataModelPropertyDefinitionViewModel LastCreatedProperty { get; private set; }

        /// <summary>
        /// Get the last created data model include definition view model.
        /// </summary>
        public RuntimeDataModelIncludeDefinitionViewModel LastCreatedIncludeDefinition { get; private set; }

        /// <summary>
        /// Get or set the executable for creating a new property.
        /// </summary>
        public IExecutable OnCreatePropertyExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for deleting a property.
        /// </summary>
        public IExecutable OnDeletePropertyExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for creating a new include definition.
        /// </summary>
        public IExecutable OnCreateIncludeDefinitionExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for deleting an include definition.
        /// </summary>
        public IExecutable OnDeleteIncludeDefinitionExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for moving a property up or down in the list.
        /// </summary>
        public IExecutable OnMovePropertyExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for committing modification of the data model.
        /// </summary>
        public IExecutable OnCommitEditingExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for canceling modification of the data model.
        /// </summary>
        public IExecutable OnCancelEditingExecutable { get; set; }

        /// <summary>
        /// Notify the specified data model property has been changed.
        /// </summary>
        /// <param name="viewModel">The view model that holds the data model property.</param>
        public void NotifyPropertyDefinitionChanged(
            RuntimeDataModelPropertyDefinitionViewModel viewModel)
        {
            this.OnPropertyChanged("PropertyDefinitionViewModels");
            this.OnPropertyChanged("PropertyAndTypeNameValuePairList");
        }

        /// <summary>
        /// Delete the specified data model property.
        /// </summary>
        /// <param name="viewModel">The view model of the property to delete.</param>
        public void DeleteProperty(
            RuntimeDataModelPropertyDefinitionViewModel viewModel)
        {
            int index = this.propertyViewModels.IndexOf(viewModel);
            if (index < 0)
            {
                return;
            }

            this.propertyViewModels.RemoveAt(index);

            var myDataModel = this.DataModel as RuntimeDataModelDefinition;
            if (myDataModel == null)
            {
                return;
            }

            var def = viewModel.DataModel as RuntimeDataModelPropertyDefinition;
            myDataModel.Properties.Remove(def);

            viewModel.Dispose();
            def.Dispose();

            this.OnPropertyChanged("PropertyDefinitionViewModels");
            this.OnPropertyChanged("PropertyAndTypeNameValuePairList");

            if (DataModelPropertyListChanged != null)
            {
                DataModelPropertyListChanged(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// Notify the specified data model include definition has been changed.
        /// </summary>
        /// <param name="viewModel">The view model that holds the include definition.</param>
        public void NotifyIncludeDefinitionChanged(
            RuntimeDataModelIncludeDefinitionViewModel viewModel)
        {
            this.OnPropertyChanged("IncludeDefinitionViewModels");
            this.OnPropertyChanged("IncludeDefinitionNameValuePairList");
        }

        /// <summary>
        /// Delete the specified include definition.
        /// </summary>
        /// <param name="viewModel">The view model of the include definition to delete.</param>
        public void DeleteIncludeDefinition(
            RuntimeDataModelIncludeDefinitionViewModel viewModel)
        {
            int index = this.includeDefViewModels.IndexOf(viewModel);
            if (index < 0)
            {
                return;
            }

            this.includeDefViewModels.RemoveAt(index);

            var myDataModel = this.DataModel as RuntimeDataModelDefinition;
            if (myDataModel == null)
            {
                return;
            }

            var def = viewModel.DataModel as RuntimeDataModelIncludeDefinition;
            myDataModel.Includes.Remove(def);

            viewModel.Dispose();
            def.Dispose();

            this.OnPropertyChanged("IncludeDefinitionViewModels");
            this.OnPropertyChanged("IncludeDefinitionNameValuePairList");
        }

        /// <summary>
        /// Handle execution when committing modification of the data model.
        /// </summary>
        /// <param name="parameter">The execution parameters.</param>
        private void OnCommitEditing(object parameter)
        {
            var values = parameter as KeyValuePair<string, object>[];
            if (values == null)
            {
                return;
            }

            try
            {
                foreach (var pair in values)
                {
                    var binder = new EffectMakerSetMemberBinder(pair.Key, true);
                    this.TrySetMember(binder, pair.Value);
                }

                // Dispatch the data model modified event.
                if (DataModelModified != null)
                {
                    DataModelModified(this, EventArgs.Empty);
                }
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "RuntimeDataModelDefinitionViewModel.OnCommitEditing => Failed setting values to the runtime data model.");
                Logger.Log(LogLevels.Error, "Exception : {0}, message : {1}", ex.GetType().ToString(), ex.Message);
            }
        }

        /// <summary>
        /// Handle execution when canceling modification of the data model.
        /// </summary>
        /// <param name="parameter">The execution parameters.</param>
        private void OnCancelEditing(object parameter)
        {
            var myParent = this.Parent as RootRuntimeDataModelDefinitionViewModel;
            if (myParent == null)
            {
                return;
            }

            myParent.DeleteDataModel(this);
        }

        /// <summary>
        /// Handle execution for creating a new property.
        /// </summary>
        /// <param name="parameter">The execution parameter.</param>
        private void OnCreateProperty(object parameter)
        {
            var myDataModel = this.DataModel as RuntimeDataModelDefinition;
            if (myDataModel == null)
            {
                return;
            }

            // Create a unique name for the property.
            string defaultName = "Field_";
            var regex = new Regex("^" + defaultName + "([0-9]+)$");

            List<int> existingSerialList = new List<int>();

            // Collect all the numbers that is used.
            foreach (RuntimeDataModelPropertyDefinition property in myDataModel.Properties)
            {
                Match match = regex.Match(property.Name);
                if (match.Groups.Count < 2)
                {
                    continue;
                }

                int currSerial;
                if (int.TryParse(match.Groups[1].Value, out currSerial) == true)
                {
                    existingSerialList.Add(currSerial);
                }
            }

            // Find an unused number for the default name.
            defaultName += this.FindUnusedSerialNumber(existingSerialList).ToString();

            // Create the new property and add it to the list.
            var newProperty = new RuntimeDataModelPropertyDefinition()
            {
                TypeName = "float",
                Description = string.Empty,
                Name = defaultName
            };

            myDataModel.Properties.Add(newProperty);

            // Create a view model for the new property definition.
            var child = new RuntimeDataModelPropertyDefinitionViewModel(this, newProperty);
            this.Children.Add(child);
            this.propertyViewModels.Add(child);

            // Issue property changed event.
            this.OnPropertyChanged("PropertyAndTypeNameValuePairList");
            this.OnPropertyChanged("PropertyDefinitionViewModels");

            // Remember the created property.
            var binder = new EffectMakerSetMemberBinder("LastCreatedProperty", true);
            this.TrySetMember(binder, child);

            if (DataModelPropertyListChanged != null)
            {
                DataModelPropertyListChanged(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// Handle execution for deleting a property.
        /// </summary>
        /// <param name="parameter">The execution parameter.</param>
        private void OnDeleteProperty(object parameter)
        {
            var viewModel = parameter as RuntimeDataModelPropertyDefinitionViewModel;
            if (viewModel == null)
            {
                return;
            }

            this.DeleteProperty(viewModel);
        }

        /// <summary>
        /// Handle execution for moving a property up or down in the list.
        /// </summary>
        /// <param name="parameter">The execution parameter.</param>
        private void OnMoveProperty(object parameter)
        {
            var dataModelDef = this.DataModel as RuntimeDataModelDefinition;
            if (dataModelDef == null)
            {
                return;
            }

            var myParam = parameter as Tuple<object, int>;
            if (myParam == null || myParam.Item2 == 0)
            {
                return;
            }

            var propertyVM = myParam.Item1 as RuntimeDataModelPropertyDefinitionViewModel;
            if (propertyVM == null)
            {
                return;
            }

            var propertyDef = propertyVM.DataModel as RuntimeDataModelPropertyDefinition;
            if (propertyDef == null)
            {
                return;
            }

            int currIndex = dataModelDef.Properties.IndexOf(propertyDef);
            int newIndex = currIndex + myParam.Item2;

            dataModelDef.Properties.RemoveAt(currIndex);

            newIndex = Math.Min(dataModelDef.Properties.Count, Math.Max(0, newIndex));

            dataModelDef.Properties.Insert(newIndex, propertyDef);

            // First remove all the property view models.
            for (int i = this.Children.Count - 1; i >= 0; --i)
            {
                if (this.Children[i] is RuntimeDataModelPropertyDefinitionViewModel)
                {
                    this.Children[i].Dispose();
                    this.Children.RemoveAt(i);
                }
            }

            this.propertyViewModels.Clear();

            // Update the view models for the properties.
            RuntimeDataModelPropertyDefinitionViewModel viewModelToSelect = null;
            foreach (RuntimeDataModelPropertyDefinition def in dataModelDef.Properties)
            {
                var child = new RuntimeDataModelPropertyDefinitionViewModel(this, def);
                this.Children.Add(child);
                this.propertyViewModels.Add(child);

                if (propertyDef == def)
                {
                    viewModelToSelect = child;
                }
            }

            // Issue property changed event.
            this.OnPropertyChanged("PropertyAndTypeNameValuePairList");
            this.OnPropertyChanged("PropertyDefinitionViewModels");

            // Remember the created property.
            var binder = new EffectMakerSetMemberBinder("LastCreatedProperty", true);
            this.TrySetMember(binder, viewModelToSelect);

            if (DataModelPropertyListChanged != null)
            {
                DataModelPropertyListChanged(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// Handle execution for creating a new include definition.
        /// </summary>
        /// <param name="parameter">The execution parameter.</param>
        private void OnCreateIncludeDefinition(object parameter)
        {
            var myDataModel = this.DataModel as RuntimeDataModelDefinition;
            if (myDataModel == null)
            {
                return;
            }

            // Create the new include definition and add it to the list.
            var newDefinition = new RuntimeDataModelIncludeDefinition()
            {
                File = "MyIncludeFile.h"
            };

            myDataModel.Includes.Add(newDefinition);

            // Create a view model for the new property definition.
            var child = new RuntimeDataModelIncludeDefinitionViewModel(this, newDefinition);
            this.Children.Add(child);
            this.includeDefViewModels.Add(child);

            // Issue property changed event.
            this.OnPropertyChanged("IncludeDefinitionViewModels");
            this.OnPropertyChanged("IncludeDefinitionNameValuePairList");

            // Remember the created property.
            var binder = new EffectMakerSetMemberBinder("LastCreatedIncludeDefinition", true);
            this.TrySetMember(binder, child);
        }

        /// <summary>
        /// Handle execution for deleting a namespace.
        /// </summary>
        /// <param name="parameter">The execution parameter.</param>
        private void OnDeleteIncludeDefinition(object parameter)
        {
            var viewModel = parameter as RuntimeDataModelIncludeDefinitionViewModel;
            if (viewModel == null)
            {
                return;
            }

            this.DeleteIncludeDefinition(viewModel);
        }

        /// <summary>
        /// Find an unused serial number from the given used serial number list.
        /// </summary>
        /// <param name="usedSerialNumList">The serial numbers that has already being used.</param>
        /// <returns>An unused serial number.</returns>
        private int FindUnusedSerialNumber(List<int> usedSerialNumList)
        {
            int serialNumber = 1;
            bool found = true;
            while (found == true)
            {
                found = false;
                foreach (int i in usedSerialNumList)
                {
                    if (i == serialNumber)
                    {
                        ++serialNumber;
                        found = true;
                        break;
                    }
                }
            }

            return serialNumber;
        }
    }
}
