﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;

using EffectMaker.DataModelMaker.Core.CommonData;
using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.DataTypes;
using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelMaker.UILogic.ViewModels
{
    /// <summary>
    /// View model class for the editor data model definition.
    /// </summary>
    public class RootEditorDataModelDefinitionViewModel : ViewModelBase
    {
        /// <summary>The editor data model definition view models.</summary>
        private List<EditorDataModelDefinitionViewModel> dataModelViewModels =
            new List<EditorDataModelDefinitionViewModel>();

        /// <summary>The selected data model definition view model.</summary>
        private EditorDataModelDefinitionViewModel selectedDataModelVM = null;

        /// <summary>The data model definition view model to be displayed.</summary>
        private EditorDataModelDefinitionViewModel displayedDataModelVM = null;

        /// <summary>The selected version set from UI.</summary>
        private Version selectedVersionFromUI = null;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public RootEditorDataModelDefinitionViewModel(
            ViewModelBase parent,
            EditorDataModelRootDefinition dataModel) :
            base(parent, dataModel)
        {
            this.OnCreateDataModelExecutable = new AnonymousExecutable(this.OnCreateDataModel);
            this.OnCreateVersionExecutable = new AnonymousExecutable(this.OnCreateVersion);
            this.OnDeleteVersionExecutable = new AnonymousExecutable(this.OnDeleteVersion);
            this.OnSelectVersionExecutable = new AnonymousExecutable(this.OnSelectVersion);

            EditorDataModelPropertyDefinitionViewModel.DataModelPropertyModified += (s, e) =>
            {
                this.OnPropertyChanged("VersionedItemsAtSelectedVersion");
            };

            foreach (EditorDataModelDefinition def in dataModel.DataModels)
            {
                var child = new EditorDataModelDefinitionViewModel(this, def);
                this.Children.Add(child);
                this.dataModelViewModels.Add(child);
            }
        }

        /// <summary>
        /// Static event triggered when the data model list is changed.
        /// </summary>
        public static event EventHandler DataModelListChanged = null;

        /// <summary>
        /// Enumerate the versions from the version definitions.
        /// </summary>
        public IEnumerable<Version> DefinedVersions
        {
            get
            {
                var dm = this.DataModel as EditorDataModelRootDefinition;
                return from versionDef in dm.Versions
                       select versionDef.Version.Version;
            }
        }

        /// <summary>
        /// Get the versioned items at the selected version.
        /// The selected version is set through OnSelectVersionExecutable.
        /// </summary>
        public IEnumerable<VersionedItemData> VersionedItemsAtSelectedVersion
        {
            get
            {
                var version = this.selectedVersionFromUI;
                if (version == null)
                {
                    return Enumerable.Empty<VersionedItemData>();
                }

                var versionedItems = new List<VersionedItemData>();

                var root = this.DataModel as EditorDataModelRootDefinition;

                int versionIndex = root.FindVersionIndex(version);
                if (versionIndex < 0)
                {
                    return Enumerable.Empty<VersionedItemData>();
                }

                // Sort the data models by name.
                var dataModels = root.DataModels.OrderBy(dm => dm.Name);

                // Loop through data models.
                VersionedItemData dataModelData = null;
                foreach (EditorDataModelDefinition dm in dataModels)
                {
                    dataModelData = null;
                    if (dm.CreatedVersionIndex == versionIndex)
                    {
                        // The data model uses the selected version as the created version.
                        dataModelData = new VersionedItemData()
                        {
                            Name = dm.Name,
                            VersionAction = VersionActions.Create,
                        };
                    }
                    else if (dm.DeletedVersionIndex == versionIndex)
                    {
                        // The data model uses the selected version as the deleted version.
                        dataModelData = new VersionedItemData()
                        {
                            Name = dm.Name,
                            VersionAction = VersionActions.Delete,
                        };
                    }

                    // Check for the properties in the data model.
                    foreach (EditorDataModelPropertyDefinition property in dm.AllProperties)
                    {
                        // Does the property has version history at the selected version?
                        var versionDef =
                            property.Versions.FirstOrDefault(def => def.VersionIndex == versionIndex);

                        if (versionDef != null &&
                            versionDef.Action != VersionActions.None)
                        {
                            if (dataModelData == null)
                            {
                                // The data model has no version history at the version but
                                // we need it to store the property as a child, so we need
                                // to create the data for the data model.
                                dataModelData = new VersionedItemData()
                                {
                                    Name = dm.Name,
                                    VersionAction = VersionActions.None,
                                };
                            }

                            var propertyData = new VersionedItemData()
                            {
                                Name = property.Name,
                                VersionAction = versionDef.Action,
                            };

                            dataModelData.AddChild(propertyData);
                        }
                    }

                    if (dataModelData != null)
                    {
                        // Add the data to our list.
                        versionedItems.Add(dataModelData);
                    }
                }

                return versionedItems;
            }
        }

        /// <summary>
        /// Enumerate editor data model definition view models.
        /// </summary>
        public IEnumerable<EditorDataModelDefinitionViewModel> DataModelDefinitionViewModels
        {
            get { return from vm in this.dataModelViewModels orderby vm.DataModelName select vm; }
        }

        /// <summary>
        /// Get or set the data model definition view model to be displayed.
        /// </summary>
        public object DisplayedDataModel
        {
            get
            {
                if (this.displayedDataModelVM != null)
                {
                    return this.displayedDataModelVM;
                }
                else
                {
                    return this.selectedDataModelVM;
                }
            }

            set
            {
                if (value != null && (value is EditorDataModelDefinitionViewModel) == false)
                {
                    return;
                }

                this.SetValue(
                    ref this.displayedDataModelVM,
                    (EditorDataModelDefinitionViewModel)value);
            }
        }

        /// <summary>
        /// Get or set the selected data model definition view model.
        /// </summary>
        public object SelectedDataModel
        {
            get
            {
                return this.selectedDataModelVM;
            }

            set
            {
                if (value != null && (value is EditorDataModelDefinitionViewModel) == false)
                {
                    return;
                }

                this.SetValue(
                    ref this.selectedDataModelVM,
                    (EditorDataModelDefinitionViewModel)value);
            }
        }

        /// <summary>
        /// Get or set the executable for creating a new data model.
        /// </summary>
        public IExecutable OnCreateDataModelExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for creating a new version definition.
        /// </summary>
        public IExecutable OnCreateVersionExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for deleting the specified version definition.
        /// </summary>
        public IExecutable OnDeleteVersionExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for selecting a specific version.
        /// </summary>
        public IExecutable OnSelectVersionExecutable { get; set; }

        /// <summary>
        /// Delete the specified data model.
        /// </summary>
        /// <param name="viewModel">The view model of the data model to delete.</param>
        public void DeleteDataModel(EditorDataModelDefinitionViewModel viewModel)
        {
            var myDataModel = this.DataModel as EditorDataModelRootDefinition;
            if (myDataModel == null)
            {
                return;
            }

            int index = this.dataModelViewModels.IndexOf(viewModel);
            if (index < 0)
            {
                return;
            }

            if (this.SelectedDataModel == viewModel)
            {
                this.SelectedDataModel = null;
            }

            this.dataModelViewModels.RemoveAt(index);

            var def = viewModel.DataModel as EditorDataModelDefinition;
            myDataModel.DataModels.Remove(def);

            viewModel.Dispose();
            def.Dispose();

            this.OnPropertyChanged("DataModelDefinitionViewModels");

            // Trigger the event.
            if (DataModelListChanged != null)
            {
                DataModelListChanged(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// Handle execution for creating a new data model.
        /// </summary>
        /// <param name="parameter">The execution parameter.</param>
        private void OnCreateDataModel(object parameter)
        {
            var createParams = parameter as object[];
            if (createParams == null || createParams.Length < 2)
            {
                return;
            }

            if (createParams.Length >= 3 && createParams[2] is bool)
            {
                // First set the return value to false. (fail)
                createParams[2] = false;
            }

            var myDataModel = this.DataModel as EditorDataModelRootDefinition;
            if (myDataModel == null)
            {
                return;
            }

            string nameSpace = createParams[0] as string;
            string typeName = createParams[1] as string;
            if (string.IsNullOrEmpty(typeName) == true)
            {
                return;
            }

            // Check if the given namespace and name has already been registered.
            EditorTypeInfo info =
                TypeManager.FindEditorType(nameSpace, typeName);
            if (info != null)
            {
                return;
            }

            // Create the new data model and add it to the list.
            var dataModelDef = new EditorDataModelDefinition()
            {
                CreatedVersionIndex = myDataModel.EditingVersionIndex,
                Namespace = nameSpace,
                Name = typeName
            };

            myDataModel.DataModels.Add(dataModelDef);

            // Create a view model for the new data model definition.
            var child = new EditorDataModelDefinitionViewModel(this, dataModelDef);
            this.Children.Add(child);
            this.dataModelViewModels.Add(child);

            // Issue property changed event.
            this.OnPropertyChanged("DataModelDefinitionViewModels");

            this.SelectedDataModel = child;

            if (createParams.Length >= 3 && createParams[2] is bool)
            {
                // Set the return value to true. (succeed)
                createParams[2] = true;
            }

            // Trigger the event.
            if (DataModelListChanged != null)
            {
                DataModelListChanged(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// Handle execution for deleting a data model.
        /// </summary>
        /// <param name="parameter">The execution parameter.</param>
        private void OnDeleteDataModel(object parameter)
        {
            var viewModel = parameter as EditorDataModelDefinitionViewModel;
            if (viewModel == null)
            {
                return;
            }

            this.DeleteDataModel(viewModel);
        }

        /// <summary>
        /// Handle execution for creating a new version.
        /// </summary>
        /// <param name="parameter">The execution parameter.</param>
        private void OnCreateVersion(object parameter)
        {
            var version = parameter as Version;
            if (version == null)
            {
                return;
            }

            var root = this.DataModel as EditorDataModelRootDefinition;

            // Find the index to add new version definition to.
            int index = 0;
            for (; index < root.Versions.Count; ++index)
            {
                EditorVersionDefinition def = root.Versions[index];
                if (def.Version > version)
                {
                    break;
                }
            }

            // Create the version definition.
            var newDef = new EditorVersionDefinition()
            {
                Version = version,
            };

            if (index >= root.Versions.Count)
            {
                root.Versions.Add(newDef);
                root.EditingVersionIndex = root.LatestVersionIndex;
            }
            else
            {
                root.Versions.Insert(index, newDef);

                // We need to update version indices set in the data models and
                // their properties because the index of the versions later
                // the inserted version has changed.
                this.AdjustVersionIndex(index + 1, 1);
            }

            // Notify the UI to update.
            this.OnPropertyChanged("DefinedVersions");
        }

        /// <summary>
        /// Handle execution for deleting a new version.
        /// </summary>
        /// <param name="parameter">The execution parameter.</param>
        private void OnDeleteVersion(object parameter)
        {
            var version = parameter as Version;
            if (version == null)
            {
                return;
            }

            var root = this.DataModel as EditorDataModelRootDefinition;
            int index = root.FindVersionIndex(version);
            if (index < 0)
            {
                return;
            }

            root.Versions.RemoveAt(index);

            // We need to update version indices set in the data models and
            // their properties because the index of the versions later
            // the deleted version has changed.
            this.AdjustVersionIndex(index + 1, -1);

            // Notify the UI to update.
            this.OnPropertyChanged("DefinedVersions");
        }

        /// <summary>
        /// Handle execution for selecting the specific version.
        /// </summary>
        /// <param name="parameter">The execution parameter.</param>
        private void OnSelectVersion(object parameter)
        {
            var version = parameter as Version;
            if (version == null)
            {
                return;
            }

            this.selectedVersionFromUI = version;

            this.OnPropertyChanged("VersionedItemsAtSelectedVersion");
        }

        /// <summary>
        /// Adjust version index in the data models and their properties.
        /// The indices that should be adjusted are those being equal or greater
        /// than "startIndex", with the specified amount set in "adjustment".
        /// </summary>
        /// <param name="startIndex">The start index.</param>
        /// <param name="adjustment">The adjustment.</param>
        private void AdjustVersionIndex(int startIndex, int adjustment)
        {
            var root = this.DataModel as EditorDataModelRootDefinition;

            foreach (EditorDataModelDefinition dm in root.DataModels)
            {
                if (dm.CreatedVersionIndex >= startIndex)
                {
                    dm.CreatedVersionIndex += adjustment;
                }
                else if (dm.DeletedVersionIndex >= startIndex)
                {
                    dm.DeletedVersionIndex += adjustment;
                }

                foreach (EditorDataModelPropertyDefinition property in dm.AllProperties)
                {
                    for (int i = 0; i < property.Versions.Count; ++i)
                    {
                        EditorPropertyVersionDefinition versionDef = property.Versions[i];
                        if (versionDef.VersionIndex >= startIndex)
                        {
                            versionDef.VersionIndex += adjustment;
                        }
                    }
                }
            }
        }
    }
}
