﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;

using EffectMaker.DataModelMaker.Core.Converters;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.Renderable;
using EffectMaker.Foundation.Utility;

using EffectMaker.UIControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.DataModelMaker.UIControls.SelectConverter
{
    /// <summary>
    /// Class for rendering converter list item.
    /// </summary>
    internal class ConverterListItemRenderable : RectangleShape
    {
        /// <summary>Constant value for the margin at the top and bottom of the item.</summary>
        private const float ItemMargin = 5.0f;

        /// <summary>The large font for rendering.</summary>
        private static Font largeFont = null;

        /// <summary>The small font for rendering.</summary>
        private static Font smallFont = null;

        /// <summary>The brush for rendering the converter information.</summary>
        private Brush converterInfoBrush = Brushes.Orange;

        /// <summary>The brush for rendering the input value information.</summary>
        private Brush inputValueInfoBrush = Brushes.DarkSlateBlue;

        /// <summary>The brush for rendering the output value information.</summary>
        private Brush outputValueInfoBrush = Brushes.DarkOliveGreen;

        /// <summary>The flag indicating if mouse is over this item.</summary>
        private bool isMouseOver = false;

        /// <summary>The flag indicating if this item is selected.</summary>
        private bool isSelected = false;

        /// <summary>
        /// Static constructor.
        /// </summary>
        static ConverterListItemRenderable()
        {
            // Setup the fonts for rendering.
            largeFont = new Font(FontFamily.GenericSansSerif, 12.0f, FontStyle.Bold);
            smallFont = new Font(FontFamily.GenericSansSerif, 7.0f);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="owner">The owner of the renderable.</param>
        public ConverterListItemRenderable(ConverterListViewport owner) :
            base()
        {
            this.Owner = owner;

            this.UpdateAppearance();
        }

        /// <summary>
        /// Get the owner converter list viewport.
        /// </summary>
        public ConverterListViewport Owner { get; private set; }

        /// <summary>
        /// Get or set the converter information.
        /// </summary>
        public ConverterInfo ConverterInfo { get; set; }

        /// <summary>
        /// Get or set the flag indicating if mouse is over this item.
        /// </summary>
        public bool IsMouseOver
        {
            get
            {
                return this.isMouseOver;
            }

            set
            {
                if (this.isMouseOver != value)
                {
                    this.isMouseOver = value;
                    this.UpdateAppearance();
                }
            }
        }

        /// <summary>
        /// Get or set the flag indicating if this item is selected.
        /// </summary>
        public bool IsSelected
        {
            get
            {
                return this.isSelected;
            }

            set
            {
                if (this.isSelected != value)
                {
                    this.isSelected = value;
                    this.UpdateAppearance();
                }
            }
        }

        /// <summary>
        /// Update the item for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected override void UpdateSelf(RenderContext context)
        {
            Graphics g = context.Graphics;

            using (var strFormat = new StringFormat())
            {
                // Top-left alignment.
                strFormat.Alignment = StringAlignment.Near;
                strFormat.LineAlignment = StringAlignment.Near;

                var textRect = this.MeasureText(
                    g,
                    this.ConverterInfo.Name,
                    largeFont,
                    strFormat,
                    new PointF(0.0f, ItemMargin));

                textRect = this.MeasureText(
                    g,
                    this.ConverterInfo.Description,
                    smallFont,
                    strFormat,
                    new PointF(0.0f, textRect.Bottom));

                int index = 1;
                foreach (string desc in this.ConverterInfo.InputValueDescriptions)
                {
                    textRect = this.MeasureText(
                        g,
                        string.Format(Properties.Resources.ConverterInputValueTitle, index),
                        smallFont,
                        strFormat,
                        new PointF(0.0f, textRect.Bottom));

                    textRect = this.MeasureText(
                        g,
                        desc,
                        smallFont,
                        strFormat,
                        new PointF(textRect.Right, textRect.Top));

                    ++index;
                }

                textRect = this.MeasureText(
                    g,
                    Properties.Resources.ConverterOutputValueTitle,
                    smallFont,
                    strFormat,
                    new PointF(0.0f, textRect.Bottom));

                textRect = this.MeasureText(
                    g,
                    this.ConverterInfo.OutputValueDescription,
                    smallFont,
                    strFormat,
                    new PointF(textRect.Right, textRect.Top));

                // Update the height.
                this.Size = new SizeF(this.Width, textRect.Bottom + ItemMargin);
            }

            base.UpdateSelf(context);
        }

        /// <summary>
        /// Render the item's border.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawBorder(Graphics g)
        {
            /*
            using (var pen = new Pen(this.BorderColor, this.BorderThickness))
            {
                g.DrawLine(
                    pen,
                    this.TransformedBounds.Left,
                    this.TransformedBounds.Top,
                    this.TransformedBounds.Right,
                    this.TransformedBounds.Top);

                g.DrawLine(
                    pen,
                    this.TransformedBounds.Left,
                    this.TransformedBounds.Bottom,
                    this.TransformedBounds.Right,
                    this.TransformedBounds.Bottom);
            }
             * */
        }

        /// <summary>
        /// Render the item's foreground.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawForeground(Graphics g)
        {
            var rect = this.TransformedBounds;
            rect.Y += ItemMargin;

            using (var strFormat = new StringFormat())
            {
                // Top-left alignment.
                strFormat.Alignment = StringAlignment.Near;
                strFormat.LineAlignment = StringAlignment.Near;

                var textRect = this.DrawText(
                    g,
                    this.ConverterInfo.Name,
                    largeFont,
                    strFormat,
                    this.converterInfoBrush,
                    new PointF(rect.X, rect.Y));

                textRect = this.DrawText(
                    g,
                    this.ConverterInfo.Description,
                    smallFont,
                    strFormat,
                    this.converterInfoBrush,
                    new PointF(rect.X, textRect.Bottom));

                int index = 1;
                foreach (string desc in this.ConverterInfo.InputValueDescriptions)
                {
                    textRect = this.DrawText(
                        g,
                        string.Format(Properties.Resources.ConverterInputValueTitle, index),
                        smallFont,
                        strFormat,
                        this.inputValueInfoBrush,
                        new PointF(rect.X, textRect.Bottom));

                    textRect = this.DrawText(
                        g,
                        desc,
                        smallFont,
                        strFormat,
                        this.inputValueInfoBrush,
                        new PointF(textRect.Right, textRect.Top));

                    ++index;
                }

                textRect = this.DrawText(
                    g,
                    Properties.Resources.ConverterOutputValueTitle,
                    smallFont,
                    strFormat,
                    this.outputValueInfoBrush,
                    new PointF(rect.X, textRect.Bottom));

                textRect = this.DrawText(
                    g,
                    this.ConverterInfo.OutputValueDescription,
                    smallFont,
                    strFormat,
                    this.outputValueInfoBrush,
                    new PointF(textRect.Right, textRect.Top));
            }
        }

        /// <summary>
        /// Update the appearance of the item.
        /// </summary>
        protected virtual void UpdateAppearance()
        {
            if (this.IsSelected == true)
            {
                this.BorderThickness = 2.0f;
                this.BorderColor = Color.White;
                this.FillColor = Color.DarkOliveGreen;
                this.converterInfoBrush = Brushes.Orange;
                this.inputValueInfoBrush = Brushes.PaleTurquoise;
                this.outputValueInfoBrush = Brushes.PaleGreen;
            }
            else if (this.IsMouseOver == true)
            {
                this.BorderThickness = 2.0f;
                this.BorderColor = Color.White;
                this.FillColor = Color.Khaki;
                this.converterInfoBrush = Brushes.OrangeRed;
                this.inputValueInfoBrush = Brushes.DarkSlateBlue;
                this.outputValueInfoBrush = Brushes.DarkOliveGreen;
            }
            else
            {
                this.BorderThickness = 2.0f;
                this.BorderColor = Color.White;
                this.FillColor = Color.Beige;
                this.converterInfoBrush = Brushes.Orange;
                this.inputValueInfoBrush = Brushes.DarkSlateBlue;
                this.outputValueInfoBrush = Brushes.DarkOliveGreen;
            }
        }

        /// <summary>
        /// Helper method for measuring the size of the text.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        /// <param name="text">The text to rendering.</param>
        /// <param name="font">The font for the text.</param>
        /// <param name="format">The string format.</param>
        /// <param name="offset">The rendering position offset.</param>
        /// <returns>The rendering area of the text.</returns>
        private RectangleF MeasureText(
            Graphics g,
            string text,
            Font font,
            StringFormat format,
            PointF offset)
        {
            // Compute the width for the text.
            int width = (int)Math.Floor(this.Width - offset.X);

            // Measure the exact size of the text.
            SizeF textSize = g.MeasureString(text, font, width, format);

            // The final rendering area of the text.
            return new RectangleF(offset, textSize);
        }

        /// <summary>
        /// Helper method for rendering text.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        /// <param name="text">The text to rendering.</param>
        /// <param name="font">The font for the text.</param>
        /// <param name="format">The string format.</param>
        /// <param name="brush">The brush for rendering.</param>
        /// <param name="offset">The rendering position offset.</param>
        /// <returns>The rendering area of the text.</returns>
        private RectangleF DrawText(
            Graphics g,
            string text,
            Font font,
            StringFormat format,
            Brush brush,
            PointF offset)
        {
            // The final rendering area of the text.
            var textRect = this.MeasureText(g, text, font, format, offset);

            // Draw string.
            g.DrawString(text, font, brush, textRect, format);

            return textRect;
        }
    }
}
