﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using EffectMaker.Foundation.Render.Renderable;

using EffectMaker.UIControls;

namespace EffectMaker.DataModelMaker.UIControls.Extenders
{
    /// <summary>
    /// Extender class for wrapping the child renderable operations.
    /// </summary>
    internal class ChildRenderableOperationExtender
    {
        /// <summary>The logical tree element that is extended.</summary>
        private ILogicalTreeElement extendee;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="extendee">The logical tree element that is extended.</param>
        public ChildRenderableOperationExtender(ILogicalTreeElement extendee)
        {
            this.extendee = extendee;
        }

        /// <summary>
        /// Add a child renderable.
        /// </summary>
        /// <param name="child">The child to add.</param>
        public void AddRenderable(RenderableBase child)
        {
            var childLogicalTreeElement = child as ILogicalTreeElement;
            if (childLogicalTreeElement != null)
            {
                this.extendee.Children.Add(childLogicalTreeElement);
            }
        }

        /// <summary>
        /// Add child renderables.
        /// </summary>
        /// <param name="children">The children to add.</param>
        public void AddRenderableRange(IEnumerable<RenderableBase> children)
        {
            foreach (RenderableBase child in children)
            {
                var childLogicalTreeElement = child as ILogicalTreeElement;
                if (childLogicalTreeElement != null)
                {
                    this.extendee.Children.Add(childLogicalTreeElement);
                }
            }
        }

        /// <summary>
        /// Remove a child renderable.
        /// </summary>
        /// <param name="child">The child to remove.</param>
        public void RemoveRenderable(RenderableBase child)
        {
            var childLogicalTreeElement = child as ILogicalTreeElement;
            if (childLogicalTreeElement != null)
            {
                this.extendee.Children.Remove(childLogicalTreeElement);
                child.Dispose();
            }
        }

        /// <summary>
        /// Remove child renderables.
        /// </summary>
        /// <param name="children">The children to remove.</param>
        public void RemoveRenderableRange(IEnumerable<RenderableBase> children)
        {
            foreach (RenderableBase child in children)
            {
                var childLogicalTreeElement = child as ILogicalTreeElement;
                if (childLogicalTreeElement != null)
                {
                    this.extendee.Children.Remove(childLogicalTreeElement);
                    child.Dispose();
                }
            }
        }

        /// <summary>
        /// Clear child renderables.
        /// </summary>
        public void ClearRenderables()
        {
            var extendedRenderable = this.extendee as RenderableBase;
            if (extendedRenderable == null)
            {
                return;
            }

            foreach (RenderableBase child in extendedRenderable.Renderables)
            {
                var childLogicalTreeElement = child as ILogicalTreeElement;
                if (childLogicalTreeElement != null)
                {
                    this.extendee.Children.Remove(childLogicalTreeElement);
                    child.Dispose();
                }
            }
        }
    }
}
