﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;

using EffectMaker.DataModelMaker.UIControls.Extenders;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.Renderable;
using EffectMaker.Foundation.Serialization;
using EffectMaker.Foundation.Utility;

using EffectMaker.UIControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.DataModelMaker.UIControls.DataModelVersionView
{
    /// <summary>
    /// Class for rendering data model version information.
    /// </summary>
    internal class DataModelVersionRenderable : RoundedRectangleShape, ILogicalTreeElement
    {
        /// <summary>Constant for the margin of the property renderables.</summary>
        private const float ChildMargin = 5.0f;

        /// <summary>The extender for easily implementing ILogicalTreeElement.</summary>
        private LogicalTreeElementExtender logicalTreeElementExtender = null;

        /// <summary>Backing field for Controls property.</summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>The extender for easily implement operations for child renderables.</summary>
        private ChildRenderableOperationExtender childRenderableOperationExtender = null;

        /// <summary>The child renderables for rendering the properties of the data model.</summary>
        private List<DataModelPropertyVersionRenderable> propertyRenderables =
            new List<DataModelPropertyVersionRenderable>();

        /// <summary>
        /// Static constructor.
        /// </summary>
        static DataModelVersionRenderable()
        {
            // Setup the fonts for rendering.
            NamespaceFont = new Font(FontFamily.GenericSansSerif, 8.0f);
            VersionFont = new Font(FontFamily.GenericSansSerif, 7.0f);
            DataModelNameFont = new Font(FontFamily.GenericSansSerif, 14.0f, FontStyle.Bold);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="owner">The owner of the view.</param>
        public DataModelVersionRenderable(DataModelVersionViewport owner) :
            base()
        {
            this.Translation = new PointF(ChildMargin, this.DataModelInfoHeight);
            this.BorderThickness = 2.0f;
            this.FillColor = Color.Beige;
            this.Size = new SizeF(0.0f, this.DataModelInfoHeight);
            this.SetCornerRadius(5.0f);

            this.Bindings = new BindingContainer(this);
            this.logicalTreeElementExtender = new LogicalTreeElementExtender(this);
            this.childRenderableOperationExtender = new ChildRenderableOperationExtender(this);

            this.Owner = owner;

            // Set up binding.
            this.Bindings.Add(new Binder(this, "Namespace", "Namespace"));
            this.Bindings.Add(new Binder(this, "DataModelName", "Name"));
            this.Bindings.Add(new Binder(this, "Description", "Description"));
            this.Bindings.Add(new Binder(this, "PropertySource", "AllPropertyDefinitionViewModels"));
            this.Bindings.Add(new Binder(this, "CreatedVersion", "CreatedVersion"));
            this.Bindings.Add(new Binder(this, "DeletedVersion", "DeletedVersion"));
            this.Bindings.Add(new Binder(this, "IsDeletedInCurrentVersion", "IsDeletedInCurrentVersion"));
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// Get the font for rendering the data model name.
        /// </summary>
        public static Font DataModelNameFont { get; private set; }

        /// <summary>
        /// Get the font for rendering the version string.
        /// </summary>
        public static Font VersionFont { get; private set; }

        /// <summary>
        /// Get the font for rendering the namespace string.
        /// </summary>
        public static Font NamespaceFont { get; private set; }

        /// <summary>
        /// Get the owner data model info viewport.
        /// </summary>
        public DataModelVersionViewport Owner { get; private set; }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return this.Owner; }
            protected set { this.Owner = value as DataModelVersionViewport; }
        }

        /// <summary>
        /// Gets a collection of children controls of the the current control instance.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new IndexableCollection<ILogicalTreeElement>();
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.logicalTreeElementExtender; }
        }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See LogicalTreeElementExtender for more information.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.logicalTreeElementExtender.DataContext; }
            set { this.logicalTreeElementExtender.DataContext = value; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Get or set the namespace of the data model class.
        /// </summary>
        public string Namespace { get; set; }

        /// <summary>
        /// Get or set the class name of the data model.
        /// </summary>
        public string DataModelName { get; set; }

        /// <summary>
        /// Get or set the data model description.
        /// </summary>
        public string Description { get; set; }

        /// <summary>
        /// Get or set the created version of the data model.
        /// </summary>
        public VersionXml CreatedVersion { get; set; }

        /// <summary>
        /// Get or set the deleted version of the data model.
        /// </summary>
        public VersionXml DeletedVersion { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether the data model
        /// has been deleted in the current version.
        /// </summary>
        public bool IsDeletedInCurrentVersion { get; set; }

        /// <summary>
        /// Get or set the data model property source.
        /// </summary>
        public IEnumerable PropertySource
        {
            get { return this.propertyRenderables.Select(pr => pr.DataContext).AsEnumerable(); }
            set { this.UpdateItemsFromDataSource(value); }
        }

        /// <summary>
        /// Enumerates all the child property renderables.
        /// </summary>
        public IEnumerable<DataModelPropertyVersionRenderable> PropertyRenderables
        {
            get { return this.propertyRenderables; }
        }

        /// <summary>
        /// Get the height of the render area that the data model information takes.
        /// </summary>
        protected float DataModelInfoHeight
        {
            get { return 55.0f; }
        }

        /// <summary>
        /// Clears the DataContext.
        /// See LogicalTreeElementExtender for more details.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.logicalTreeElementExtender.ClearDataContext();
        }

        /// <summary>
        /// Add a child renderable.
        /// </summary>
        /// <param name="child">The child to add.</param>
        public override void AddRenderable(RenderableBase child)
        {
            this.childRenderableOperationExtender.AddRenderable(child);
            base.AddRenderable(child);

            var propertyRenderable = child as DataModelPropertyVersionRenderable;
            if (propertyRenderable != null)
            {
                this.propertyRenderables.Add(propertyRenderable);
            }
        }

        /// <summary>
        /// Add child renderables.
        /// </summary>
        /// <param name="children">The children to add.</param>
        public override void AddRenderableRange(IEnumerable<RenderableBase> children)
        {
            this.childRenderableOperationExtender.AddRenderableRange(children);
            base.AddRenderableRange(children);

            foreach (RenderableBase child in children)
            {
                var propertyRenderable = child as DataModelPropertyVersionRenderable;
                if (propertyRenderable != null)
                {
                    this.propertyRenderables.Add(propertyRenderable);
                }
            }
        }

        /// <summary>
        /// Remove a child renderable.
        /// </summary>
        /// <param name="child">The child to remove.</param>
        public override void RemoveRenderable(RenderableBase child)
        {
            this.childRenderableOperationExtender.RemoveRenderable(child);
            base.RemoveRenderable(child);

            var propertyRenderable = child as DataModelPropertyVersionRenderable;
            if (propertyRenderable != null)
            {
                this.propertyRenderables.Remove(propertyRenderable);
            }
        }

        /// <summary>
        /// Remove child renderables.
        /// </summary>
        /// <param name="children">The children to remove.</param>
        public override void RemoveRenderableRange(IEnumerable<RenderableBase> children)
        {
            this.childRenderableOperationExtender.RemoveRenderableRange(children);
            base.RemoveRenderableRange(children);

            // To prevent modifying the property renderable list while looping it,
            // copy the children to another array first.
            var tmpChildren = children;
            if (tmpChildren == this.propertyRenderables)
            {
                tmpChildren = children.ToArray();
            }

            foreach (RenderableBase child in tmpChildren)
            {
                var propertyRenderable = child as DataModelPropertyVersionRenderable;
                if (propertyRenderable != null)
                {
                    this.propertyRenderables.Remove(propertyRenderable);
                }
            }
        }

        /// <summary>
        /// Clear child renderables.
        /// </summary>
        public override void ClearRenderables()
        {
            this.childRenderableOperationExtender.ClearRenderables();
            base.ClearRenderables();

            this.propertyRenderables.Clear();
        }

        /// <summary>
        /// Update the children for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected override void UpdateChildren(RenderContext context)
        {
            float childWidth = this.Width - (ChildMargin * 2.0f);

            // Loop through data model list items to update the content size.
            foreach (RenderableBase child in this.propertyRenderables)
            {
                child.Size = new SizeF(childWidth, child.Height);
            }

            base.UpdateChildren(context);
        }

        /// <summary>
        /// Render the item's background.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawBackground(Graphics g)
        {
            base.DrawBackground(g);

            // Draw the separation line on the top edge of the child renderables.
            using (var pen = new Pen(Color.Silver, 1.0f))
            {
                foreach (var child in this.propertyRenderables)
                {
                    var rect = child.TransformedBounds;
                    g.DrawLine(pen, rect.Location, new PointF(rect.Right, rect.Top));
                }
            }
        }

        /// <summary>
        /// Render the item's foreground.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawForeground(Graphics g)
        {
            var rect = this.TransformedBounds;

            // Compute the text size of the namespace.
            SizeF namespaceTypeTextSize = g.MeasureString(
                this.Namespace,
                DataModelVersionRenderable.NamespaceFont);

            // Compute the text size of the data model type.
            SizeF dataModelTypeTextSize = g.MeasureString(
                this.DataModelName,
                DataModelVersionRenderable.DataModelNameFont);

            // The rectangle for the namespace.
            RectangleF namespaceTextRect = rect;
            namespaceTextRect.X += 3.0f;
            namespaceTextRect.Y += 6.0f;
            namespaceTextRect.Width = rect.Width - 3.0f;
            namespaceTextRect.Height = namespaceTypeTextSize.Height;

            // The rectangle for the versions.
            RectangleF versionTextRect = rect;
            versionTextRect.X += 4.0f;
            versionTextRect.Y = namespaceTextRect.Bottom - 8.0f;
            versionTextRect.Width = rect.Width - 3.0f;
            versionTextRect.Height = 20;

            // The rectangle for the data model type name.
            RectangleF dataModelTypeTextRect = rect;
            dataModelTypeTextRect.X += 2.0f;
            dataModelTypeTextRect.Y = rect.Top + this.DataModelInfoHeight - dataModelTypeTextSize.Height - 3.0f;
            dataModelTypeTextRect.Width = rect.Width - 2.0f;
            dataModelTypeTextRect.Height = dataModelTypeTextSize.Height + 5.0f;

            // Set up the text colors.
            Brush namespaceBrush = Brushes.DarkCyan;
            Brush versionBrush = Brushes.SteelBlue;
            Brush dataModelNameBrush = Brushes.DarkBlue;
            if (this.IsDeletedInCurrentVersion == true)
            {
                namespaceBrush = Brushes.Firebrick;
                versionBrush = Brushes.Crimson;
                dataModelNameBrush = Brushes.Firebrick;
            }

            // Compose the versions. (created and deleted versions)
            string versionText = this.CreatedVersion.Value + " - ";
            if (this.DeletedVersion != null)
            {
                versionText += this.DeletedVersion.Value;
            }
            else
            {
                versionText += "the latest version";
            }

            using (var strFormat = new StringFormat())
            {
                // Top-left alignment.
                strFormat.Alignment = StringAlignment.Near;
                strFormat.LineAlignment = StringAlignment.Far;

                strFormat.Trimming = StringTrimming.EllipsisCharacter;

                g.DrawString(
                    this.Namespace,
                    DataModelVersionRenderable.NamespaceFont,
                    Brushes.DarkCyan,
                    namespaceTextRect,
                    strFormat);

                g.DrawString(
                    versionText,
                    DataModelVersionRenderable.VersionFont,
                    versionBrush,
                    versionTextRect,
                    strFormat);

                strFormat.LineAlignment = StringAlignment.Near;

                g.DrawString(
                    this.DataModelName,
                    DataModelVersionRenderable.DataModelNameFont,
                    Brushes.DarkBlue,
                    dataModelTypeTextRect,
                    strFormat);
            }
        }

        /// <summary>
        /// Update items from data source.
        /// </summary>
        /// <param name="items">The items from data source.</param>
        private void UpdateItemsFromDataSource(IEnumerable items)
        {
            // Clear all the data model renderers first.
            this.RemoveRenderableRange(this.propertyRenderables);

            float height = 0.0f;

            // Loop through data model source items.
            foreach (object item in items)
            {
                // Create a property renderable for the item source.
                var renderable = new DataModelPropertyVersionRenderable(this)
                {
                    DataContext = item,
                };

                // Setup location and size.
                renderable.Location = new PointF(0.0f, height);

                height += renderable.Height;

                // Add the list item to the child renderable list.
                this.AddRenderable(renderable);
            }

            this.Size = new SizeF(this.Width, height + this.DataModelInfoHeight);
        }
    }
}
