﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;

using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.DataTypes;

using EffectMaker.DataModelMaker.UIControls.DataModelPropertyEditor;
using EffectMaker.DataModelMaker.UIControls.Interfaces;

using EffectMaker.Foundation.Interfaces;

using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;

namespace EffectMaker.DataModelMaker.UIControls.DataModelEditor
{
    /// <summary>
    /// User control class for editor data model editing.
    /// </summary>
    public partial class EditorDataModelEditor : UIUserControl, IDataModelEditor
    {
        /// <summary>The suggestion strings for the export folder text box.</summary>
        private AutoCompleteStringCollection exportFolderSuggestions =
            new AutoCompleteStringCollection();

        /// <summary>The suggestion strings for the namespace text box.</summary>
        private AutoCompleteStringCollection namespaceSuggestions =
            new AutoCompleteStringCollection();

        /// <summary>The suggestion strings for the inherit type text box.</summary>
        private AutoCompleteStringCollection inheritTypeSuggestions =
            new AutoCompleteStringCollection();

        /// <summary>The namespaces the data model uses. (received from data source)</summary>
        private List<KeyValuePair<string, object>> usingNamespaceListSource =
            new List<KeyValuePair<string, object>>();

        /// <summary>The properties the data model has. (received from data source)</summary>
        private List<KeyValuePair<string, object>> propertyListSource =
            new List<KeyValuePair<string, object>>();

        /// <summary>
        /// Constructor.
        /// </summary>
        public EditorDataModelEditor()
        {
            this.InitializeComponent();

            this.exportFolderSuggestions.AddRange(PathManager.EditorDataModelExportFolders.ToArray());
            this.txtFolder.AutoCompleteCustomSource = this.exportFolderSuggestions;

            this.namespaceSuggestions.AddRange(TypeManager.EditorNamespaces.ToArray());
            this.txtNamespace.AutoCompleteCustomSource = this.namespaceSuggestions;

            this.inheritTypeSuggestions.AddRange(
                (from tp in TypeManager.EditorTypes
                 where tp.IsDataModel == true
                 select tp.FullName).ToArray());
            this.txtInheritingTypes.AutoCompleteCustomSource = this.inheritTypeSuggestions;

            this.propertyList.SizeChanged += (s, e) =>
            {
                this.propertyList.Columns[0].Width = this.propertyList.ClientSize.Width;
            };

            this.usingNamespaceList.SizeChanged += (s, e) =>
            {
                this.usingNamespaceList.Columns[0].Width = this.usingNamespaceList.ClientSize.Width;
            };

            this.Bindings.Add(new Binder(this, "ExportFolder", "ExportFolder"));
            this.Bindings.Add(new Binder(this, "DataModelNamespace", "Namespace"));
            this.Bindings.Add(new Binder(this, "DataModelName", "Name"));
            this.Bindings.Add(new Binder(this, "InheritingTypes", "InheritingDataModels"));
            this.Bindings.Add(new Binder(this, "DataModelDescription", "Description"));
            this.Bindings.Add(new Binder(this, "ExportSourceFile", "Export"));
            this.Bindings.Add(new Binder(this, "IsVersionEnabled", "IsVersionEnabled"));
            this.Bindings.Add(new Binder(this, "UsingNamespaceList", "UsingNamespaceNameValuePairList"));
            this.Bindings.Add(new Binder(this, "PropertyList", "PropertyAndTypeNameValuePairList"));
            this.Bindings.Add(new Binder(this, "LastCreatedProperty", "LastCreatedProperty"));
            this.Bindings.Add(new Binder(this, "LastCreatedNamespace", "LastCreatedNamespace"));
            this.Bindings.Add(new Binder(this, "OnCommitEditingExecutable", "OnCommitEditingExecutable"));
            this.Bindings.Add(new Binder(this, "OnCancelEditingExecutable", "OnCancelEditingExecutable"));
            this.Bindings.Add(new Binder(this, "OnCreatePropertyExecutable", "OnCreatePropertyExecutable"));
            this.Bindings.Add(new Binder(this, "OnDeletePropertyExecutable", "OnDeletePropertyExecutable"));
            this.Bindings.Add(new Binder(this, "OnMovePropertyExecutable", "OnMovePropertyExecutable"));
            this.Bindings.Add(new Binder(this, "OnCreateUsingNamespaceExecutable", "OnCreateUsingNamespaceExecutable"));
            this.Bindings.Add(new Binder(this, "OnDeleteUsingNamespaceExecutable", "OnDeleteUsingNamespaceExecutable"));
        }

        /// <summary>
        /// Get or set the export folder of the data model source file.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public string ExportFolder
        {
            get { return this.txtFolder.Text; }
            set { this.txtFolder.Text = value; }
        }

        /// <summary>
        /// Get or set the namespace of the data model.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public string DataModelNamespace
        {
            get { return this.txtNamespace.Text; }
            set { this.txtNamespace.Text = value; }
        }

        /// <summary>
        /// Get or set the class name of the data model.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public string DataModelName
        {
            get { return this.txtName.Text; }
            set { this.txtName.Text = value; }
        }

        /// <summary>
        /// Get or set the inheriting types.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public string InheritingTypes
        {
            get { return this.txtInheritingTypes.Text; }
            set { this.txtInheritingTypes.Text = value; }
        }

        /// <summary>
        /// Get or set the description of the data model.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public string DataModelDescription
        {
            get { return this.txtDescription.Text; }
            set { this.txtDescription.Text = value; }
        }

        /// <summary>
        /// Get or set the flag indicating whether to export source file for this data model.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool ExportSourceFile
        {
            get { return this.chkExportSourceFile.Checked; }
            set { this.chkExportSourceFile.Checked = value; }
        }

        /// <summary>
        /// Get or set the flag indicating whether to enable versioning for this data model.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool IsVersionEnabled
        {
            get { return this.chkEnableVersion.Checked; }
            set { this.chkEnableVersion.Checked = value; }
        }

        /// <summary>
        /// Get or set the list of namespaces the data model uses.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IEnumerable<KeyValuePair<string, object>> UsingNamespaceList
        {
            get
            {
                return this.usingNamespaceListSource;
            }

            set
            {
                // Remember the current selection.
                int selectedIndex = 0;
                object selectedDataSource = null;
                if (this.usingNamespaceList.SelectedIndices.Count > 0)
                {
                    selectedIndex = this.usingNamespaceList.SelectedIndices[0];
                    selectedDataSource = this.usingNamespaceList.SelectedItems[0].Tag;
                }

                // First clear all the items.
                this.usingNamespaceList.Items.Clear();

                if (value == null)
                {
                    return;
                }

                // Populate the items.
                bool selectedItemFound = false;
                foreach (KeyValuePair<string, object> srcItem in value)
                {
                    var item = new ListViewItem(srcItem.Key)
                    {
                        Tag = srcItem.Value,
                        Font = new Font("Courier New", 8.0f)
                    };

                    this.usingNamespaceListSource.Add(srcItem);
                    this.usingNamespaceList.Items.Add(item);

                    // Restore the selection.
                    if (srcItem.Value == selectedDataSource)
                    {
                        item.Selected = true;
                        selectedItemFound = true;
                    }
                }

                // The selected item was not found, try to select the item at the same index.
                if (selectedItemFound == false && selectedIndex >= 0)
                {
                    int itemCount = this.usingNamespaceList.Items.Count;
                    if (selectedIndex < itemCount)
                    {
                        this.usingNamespaceList.Items[selectedIndex].Selected = true;
                    }
                    else if (itemCount > 0)
                    {
                        this.usingNamespaceList.Items[itemCount - 1].Selected = true;
                    }
                }
            }
        }

        /// <summary>
        /// Get or set the list of property the data model has.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IEnumerable<KeyValuePair<string, object>> PropertyList
        {
            get
            {
                return this.propertyListSource;
            }

            set
            {
                // Remember the current selection.
                int selectedIndex = 0;
                object selectedDataSource = null;
                if (this.propertyList.SelectedIndices.Count > 0)
                {
                    selectedIndex = this.propertyList.SelectedIndices[0];
                    selectedDataSource = this.propertyList.SelectedItems[0].Tag;
                }

                // Suspend update of the list view.
                this.propertyList.BeginUpdate();

                // First clear all the items.
                this.propertyList.Items.Clear();

                if (value == null)
                {
                    return;
                }

                // Populate the items.
                ListViewItem itemToSelect = null;
                foreach (KeyValuePair<string, object> srcItem in value)
                {
                    var item = new ListViewItem(srcItem.Key)
                    {
                        Tag = srcItem.Value,
                        Font = new Font("Courier New", 8.0f)
                    };

                    this.propertyListSource.Add(srcItem);
                    this.propertyList.Items.Add(item);

                    // Restore the selection.
                    if (srcItem.Value == selectedDataSource)
                    {
                        itemToSelect = item;
                    }
                }

                // The selected item was not found, try to select the item at the same index.
                if (itemToSelect == null && selectedIndex >= 0)
                {
                    int itemCount = this.propertyList.Items.Count;
                    if (selectedIndex < itemCount)
                    {
                        itemToSelect = this.propertyList.Items[selectedIndex];
                    }
                    else if (itemCount > 0)
                    {
                        itemToSelect = this.propertyList.Items[itemCount - 1];
                    }
                }

                // Resume update.
                this.propertyList.EndUpdate();

                if (itemToSelect != null)
                {
                    itemToSelect.Selected = true;
                }
            }
        }

        /// <summary>
        /// Get the last created editor data model property view model.
        /// </summary>
        public object LastCreatedProperty { get; set; }

        /// <summary>
        /// Get the last created editor data model namespace view model.
        /// </summary>
        public object LastCreatedNamespace { get; set; }

        /// <summary>
        /// Get or set the executable for creating a new property.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IExecutable OnCreatePropertyExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for deleting a property.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IExecutable OnDeletePropertyExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for moving a property.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IExecutable OnMovePropertyExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for creating a new namespace.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IExecutable OnCreateUsingNamespaceExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for deleting a namespace.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IExecutable OnDeleteUsingNamespaceExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for committing edited contents.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IExecutable OnCommitEditingExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for canceling edited contents.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IExecutable OnCancelEditingExecutable { get; set; }

        /// <summary>
        /// Confirm the edited contents and commit the modification to the data source.
        /// </summary>
        /// <returns>
        /// False if there are problems in the modifications and should cancel the action.
        /// </returns>
        public bool CommitEditing()
        {
            // Parse the inheriting type list.
            List<Guid> inheritingTypeGuidList = new List<Guid>();
            if (string.IsNullOrEmpty(this.InheritingTypes) == false)
            {
                string[] tokens = this.InheritingTypes.Split(
                    new char[] { ',' },
                    StringSplitOptions.RemoveEmptyEntries);

                foreach (string token in tokens)
                {
                    string inheritingTypeName = token.Trim();

                    // Parse the type name and find the type info.
                    EditorTypeInfo primaryInfo, elementInfo;
                    if (TypeManager.ParseEditorType(
                        inheritingTypeName,
                        out primaryInfo,
                        out elementInfo) == false)
                    {
                        MessageBox.Show(
                            string.Format(Properties.Resources.WarningInheritingTypeUnknown, inheritingTypeName),
                            Properties.Resources.WarningMessageBoxTitle,
                            MessageBoxButtons.OK,
                            MessageBoxIcon.Exclamation);

                        this.txtInheritingTypes.Focus();
                        return false;
                    }

                    // Check if the type is a data model type.
                    if (primaryInfo == null ||
                        primaryInfo.IsDataModel == false ||
                        primaryInfo.DataModelDefinition == null)
                    {
                        MessageBox.Show(
                            string.Format(Properties.Resources.WarningInheritingTypeIsNotDataModel, inheritingTypeName),
                            Properties.Resources.WarningMessageBoxTitle,
                            MessageBoxButtons.OK,
                            MessageBoxIcon.Exclamation);

                        this.txtInheritingTypes.Focus();
                        return false;
                    }

                    // Add the Guid of the data model definition to the list.
                    inheritingTypeGuidList.Add(primaryInfo.DataModelDefinition.Guid);
                }
            }

            if (this.OnCommitEditingExecutable != null)
            {
                this.OnCommitEditingExecutable.Execute(new KeyValuePair<string, object>[]
                {
                    new KeyValuePair<string, object>("ExportFolder", this.ExportFolder),
                    new KeyValuePair<string, object>("Namespace", this.DataModelNamespace),
                    new KeyValuePair<string, object>("Name", this.DataModelName),
                    new KeyValuePair<string, object>("Description", this.DataModelDescription),
                    new KeyValuePair<string, object>("SuperClasses", inheritingTypeGuidList),
                    new KeyValuePair<string, object>("Export", this.ExportSourceFile),
                    new KeyValuePair<string, object>("IsVersionEnabled", this.IsVersionEnabled),
                });
            }

            return true;
        }

        /// <summary>
        /// Cancel the edited contents.
        /// </summary>
        /// <param name="deleteContents">True to delete the editing contents.</param>
        public void CancelEditing(bool deleteContents)
        {
            if (deleteContents == true &&
                this.OnCancelEditingExecutable != null)
            {
                this.OnCancelEditingExecutable.Execute(null);
            }
        }

        /// <summary>
        /// Handle Click event for the "Move Property Up" button.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnClickPropertyUpButton(object sender, EventArgs e)
        {
            if (this.propertyList.SelectedItems == null ||
                this.propertyList.SelectedItems.Count <= 0)
            {
                return;
            }

            ListViewItem selectedItem = this.propertyList.SelectedItems[0];
            if (selectedItem == null)
            {
                return;
            }

            if (this.OnMovePropertyExecutable != null)
            {
                this.OnMovePropertyExecutable.Execute(Tuple.Create(selectedItem.Tag, -1));

                // Select the property.
                if (this.LastCreatedProperty != null)
                {
                    foreach (ListViewItem item in this.propertyList.Items)
                    {
                        if (item.Tag == this.LastCreatedProperty)
                        {
                            item.Selected = true;
                            break;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// Handle Click event for the "Move Property Down" button.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnClickPropertyDownButton(object sender, EventArgs e)
        {
            if (this.propertyList.SelectedItems == null ||
                this.propertyList.SelectedItems.Count <= 0)
            {
                return;
            }

            ListViewItem selectedItem = this.propertyList.SelectedItems[0];
            if (selectedItem == null)
            {
                return;
            }

            if (this.OnMovePropertyExecutable != null)
            {
                this.OnMovePropertyExecutable.Execute(Tuple.Create(selectedItem.Tag, 1));

                // Select the property.
                if (this.LastCreatedProperty != null)
                {
                    foreach (ListViewItem item in this.propertyList.Items)
                    {
                        if (item.Tag == this.LastCreatedProperty)
                        {
                            item.Selected = true;
                            break;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// Handle Click event for the "Add Property" button.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnClickAddPropertyButton(object sender, EventArgs e)
        {
            if (this.OnCreatePropertyExecutable != null)
            {
                this.OnCreatePropertyExecutable.Execute(null);

                if (this.LastCreatedProperty != null)
                {
                    var dialog = new DataModelPropertyEditorDialog()
                    {
                        DataSource = this.LastCreatedProperty,
                        ShouldDeleteDataSourceOnCancel = true,
                        StartPosition = FormStartPosition.CenterParent
                    };

                    dialog.ShowDialog(this);
                }
            }
        }

        /// <summary>
        /// Handle Click event for the "Delete Property" button.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnClickDelPropertyButton(object sender, EventArgs e)
        {
            if (this.propertyList.SelectedItems == null ||
                this.propertyList.SelectedItems.Count <= 0)
            {
                return;
            }

            ListViewItem item = this.propertyList.SelectedItems[0];
            if (item == null)
            {
                return;
            }

            if (this.OnDeletePropertyExecutable != null)
            {
                this.OnDeletePropertyExecutable.Execute(item.Tag);
            }
        }

        /// <summary>
        /// Handle Click event for the "Add UsingNamespace" button.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnClickAddUsingNamespaceButton(object sender, EventArgs e)
        {
            if (this.OnCreateUsingNamespaceExecutable != null)
            {
                this.OnCreateUsingNamespaceExecutable.Execute(null);

                if (this.LastCreatedNamespace != null)
                {
                    var dialog = new EditorDataModelNamespaceEditorDialog()
                    {
                        DataSource = this.LastCreatedNamespace,
                        ShouldDeleteDataSourceOnCancel = true,
                        StartPosition = FormStartPosition.CenterParent
                    };

                    dialog.ShowDialog(this);
                }
            }
        }

        /// <summary>
        /// Handle Click event for the "Delete UsingNamespace" button.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnClickDelUsingNamespaceButton(object sender, EventArgs e)
        {
            if (this.usingNamespaceList.SelectedItems == null ||
                this.usingNamespaceList.SelectedItems.Count <= 0)
            {
                return;
            }

            ListViewItem item = this.usingNamespaceList.SelectedItems[0];
            if (item == null)
            {
                return;
            }

            if (this.OnDeleteUsingNamespaceExecutable != null)
            {
                this.OnDeleteUsingNamespaceExecutable.Execute(item.Tag);
            }
        }

        /// <summary>
        /// Handle DoubleClick event for the property list view.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnDoubleClickPropertyItem(object sender, MouseEventArgs e)
        {
            ListViewItem item = this.propertyList.GetItemAt(e.X, e.Y);
            if (item != null)
            {
                var dialog = new DataModelPropertyEditorDialog()
                {
                    DataSource = item.Tag,
                    ShouldDeleteDataSourceOnCancel = false,
                    StartPosition = FormStartPosition.CenterParent
                };

                dialog.ShowDialog(this);
            }
        }

        /// <summary>
        /// Handle DoubleClick event for the using namespace list view.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnDoubleClickUsingNamespaceItem(object sender, MouseEventArgs e)
        {
            ListViewItem item = this.usingNamespaceList.GetItemAt(e.X, e.Y);
            if (item != null)
            {
                var dialog = new EditorDataModelNamespaceEditorDialog()
                {
                    DataSource = item.Tag,
                    ShouldDeleteDataSourceOnCancel = false,
                    StartPosition = FormStartPosition.CenterParent
                };

                dialog.ShowDialog(this);
            }
        }
    }
}
