﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Drawing;
using System.Linq;

using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.DataTypes;
using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.ObjectPicking;
using EffectMaker.Foundation.Render.Renderable;

using EffectMaker.UIControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.DataModelMaker.UIControls.ConversionView
{
    /// <summary>
    /// Class for rendering source data model property information.
    /// </summary>
    internal class SourceDataModelPropertyRenderable :
        RectangleShape,
        ILogicalTreeElement,
        ISourceConnectible
    {
        /// <summary>Constant for the margin of the child renderables.</summary>
        public const float ChildMargin = 5.0f;

        /// <summary>Constant for the height of the fields when it contains no field group.</summary>
        private const float NonDataModelPropertyHeight = 20.0f;

        /// <summary>The extender for easily implementing ILogicalTreeElement.</summary>
        private LogicalTreeElementExtender logicalTreeElementExtender = null;

        /// <summary>Backing field for Controls property.</summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>The renderable for rendering the binary field group.</summary>
        private SourceDataModelRenderable dataModelRenderable = null;

        /// <summary>The Guid of the source property instance definition.</summary>
        private Guid sourcePropertyInstanceGuid = Guid.Empty;

        /// <summary>The flag indicating if mouse is over this item.</summary>
        private bool isMouseOver = false;

        /// <summary>The flag indicating if the item is selected.</summary>
        private bool isSelected = false;

        /// <summary>
        /// Static constructor.
        /// </summary>
        static SourceDataModelPropertyRenderable()
        {
            // Set up the fonts for rendering.
            PropertyTypeFont = new Font(FontFamily.GenericSansSerif, 8.0f);
            PropertyNameFont = new Font(FontFamily.GenericSansSerif, 8.0f);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="owner">The owner of the view.</param>
        public SourceDataModelPropertyRenderable(SourceDataModelRenderable owner) :
            base()
        {
            this.Bindings = new BindingContainer(this);
            this.logicalTreeElementExtender = new LogicalTreeElementExtender(this);

            this.Size = new SizeF(0.0f, NonDataModelPropertyHeight);

            this.BorderColor = Color.Transparent;
            this.BorderThickness = 0.0f;
            this.FillColor = Color.Transparent;

            this.Owner = owner;

            this.Connector = new ConnectorRenderable(this)
            {
                ConnectorLength = 20.0f,
                ConnectorDirection = ConnectorRenderable.ConnectorDirections.Right,
                Offset = new SizeF(0.0f, NonDataModelPropertyHeight * 0.5f)
            };

            this.AddRenderable(this.Connector);

            this.Bindings.Add(new Binder(this, "PropertyType", "PropertyType"));
            this.Bindings.Add(new Binder(this, "PropertyName", "PropertyName"));
            this.Bindings.Add(new Binder(this, "Description", "PropertyDescription"));
            this.Bindings.Add(new Binder(this, "DataModelInstanceDataContext", "UnderlyingDataModelInstance"));
            this.Bindings.Add(new Binder(this, "SourceGuid", "Guid"));
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// Get the font for rendering the property type.
        /// </summary>
        public static Font PropertyTypeFont { get; private set; }

        /// <summary>
        /// Get the font for rendering the property name.
        /// </summary>
        public static Font PropertyNameFont { get; private set; }

        /// <summary>
        /// Get the owner field group.
        /// </summary>
        public SourceDataModelRenderable Owner { get; private set; }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return this.Owner; }
            protected set { this.Owner = value as SourceDataModelRenderable; }
        }

        /// <summary>
        /// Gets a collection of children controls of the the current control instance.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new IndexableCollection<ILogicalTreeElement>();
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.logicalTreeElementExtender; }
        }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See LogicalTreeElementExtender for more information.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.logicalTreeElementExtender.DataContext; }
            set { this.logicalTreeElementExtender.DataContext = value; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Get the connector.
        /// </summary>
        public ConnectorRenderable Connector { get; private set; }

        /// <summary>
        /// Get the location of the connector.
        /// </summary>
        public PointF ConnectorLocation
        {
            get
            {
                if (this.Connector.Visible == false &&
                    this.dataModelRenderable != null)
                {
                    return this.dataModelRenderable.ConnectorLocation;
                }
                else
                {
                    return this.Connector.ConnectorLocation;
                }
            }
        }

        /// <summary>
        /// Get the transformed location of the connector.
        /// </summary>
        public PointF TransformedConnectorLocation
        {
            get
            {
                if (this.Connector.Visible == false &&
                    this.dataModelRenderable != null)
                {
                    return this.dataModelRenderable.TransformedConnectorLocation;
                }
                else
                {
                    return this.Connector.TransformedConnectorLocation;
                }
            }
        }

        /// <summary>
        /// Get or set the connection source Guid.
        /// </summary>
        public Guid SourceGuid
        {
            get
            {
                return this.sourcePropertyInstanceGuid;
            }

            set
            {
                // First unregister the original source property instance Guid.
                if (this.sourcePropertyInstanceGuid.Equals(Guid.Empty) == false)
                {
                    ConnectionSourceManager.UnregisterSourceConnectible(
                        this.sourcePropertyInstanceGuid);
                }

                // Set the new source property instance Guid.
                if (value == null)
                {
                    this.sourcePropertyInstanceGuid = Guid.Empty;
                }
                else
                {
                    this.sourcePropertyInstanceGuid = value;
                }

                // Register the new source property instance Guid.
                if (this.sourcePropertyInstanceGuid.Equals(Guid.Empty) == false)
                {
                    if (this.dataModelRenderable.DataContext == null)
                    {
                        ConnectionSourceManager.RegisterSourceConnectible(
                            this.sourcePropertyInstanceGuid, this);
                    }
                    else
                    {
                        ConnectionSourceManager.RegisterSourceConnectible(
                            this.sourcePropertyInstanceGuid, this.dataModelRenderable);
                    }
                }
            }
        }

        /// <summary>
        /// Get or set the description of the connectible.
        /// </summary>
        public string Description { get; set; }

        /// <summary>
        /// Get the string as the type of the connectible value.
        /// </summary>
        public string ValueType
        {
            get { return this.PropertyType; }
        }

        /// <summary>
        /// Get the name of the connectible.
        /// </summary>
        public string DisplayName
        {
            get { return this.PropertyName; }
        }

        /// <summary>
        /// Get or set name of the property name.
        /// </summary>
        public string PropertyName { get; set; }

        /// <summary>
        /// Get or set type of the property.
        /// </summary>
        public string PropertyType { get; set; }

        /// <summary>
        /// Get the flag indicating whether the connection of the connectible is set.
        /// </summary>
        public bool IsConnectionSet
        {
            get
            {
                if (ConnectionRenderable.FindConnections(this.Connector).Count() > 0)
                {
                    return true;
                }

                return false;
            }
        }

        /// <summary>
        /// Get or set the data context of the underlying data model instance renderable.
        /// </summary>
        public object DataModelInstanceDataContext
        {
            get
            {
                if (this.dataModelRenderable == null)
                {
                    return null;
                }
                else
                {
                    return this.dataModelRenderable.DataContext;
                }
            }

            set
            {
                if (this.dataModelRenderable == null)
                {
                    this.dataModelRenderable = new SourceDataModelRenderable(this)
                    {
                        DataContext = value,
                        Bounds = new RectangleF(
                            0.0f, ChildMargin, this.Width, 0.0f)
                    };

                    this.AddRenderable(this.dataModelRenderable);
                }
                else
                {
                    this.dataModelRenderable.DataContext = value;
                }
            }
        }

        /// <summary>
        /// Get or set the flag indicating if mouse is over this item.
        /// </summary>
        public bool IsMouseOver
        {
            get
            {
                return this.isMouseOver;
            }

            set
            {
                if (this.isMouseOver != value)
                {
                    this.isMouseOver = value;
                    this.UpdateAppearance();
                }
            }
        }

        /// <summary>
        /// Get or set the flag indicating if this item is selected.
        /// </summary>
        public bool IsSelected
        {
            get
            {
                return this.isSelected;
            }

            set
            {
                if (this.isSelected != value)
                {
                    this.isSelected = value;
                    this.UpdateAppearance();
                }
            }
        }

        /// <summary>
        /// Clears the DataContext.
        /// See LogicalTreeElementExtender for more details.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.logicalTreeElementExtender.ClearDataContext();
        }

        /// <summary>
        /// Determine if the specified target can be connected to this connectible item.
        /// </summary>
        /// <param name="target">The connection target.</param>
        /// <returns>True if can connect.</returns>
        public bool CanConnect(IConnectible target)
        {
            if (target == null ||
                target is ISourceConnectible)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// Dispose this renderable.
        /// </summary>
        /// <param name="isDisposing">Flag indicating whether this renderable is already disposing.</param>
        protected override void Dispose(bool isDisposing)
        {
            // Unregister the original source property instance Guid.
            if (this.sourcePropertyInstanceGuid.Equals(Guid.Empty) == false)
            {
                ConnectionSourceManager.UnregisterSourceConnectible(
                    this.sourcePropertyInstanceGuid);
            }

            base.Dispose(isDisposing);
        }

        /// <summary>
        /// Update the children for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected override void UpdateChildren(RenderContext context)
        {
            // Adjust the field group's width according to mine.
            if (this.dataModelRenderable != null && this.dataModelRenderable.Visible == true)
            {
                this.dataModelRenderable.Size =
                    new SizeF(this.Width, this.dataModelRenderable.Height);
            }

            var myRC = context as ConnectorRenderContext;
            if (myRC != null &&
                myRC.ShowConnector == true &&
                this.dataModelRenderable.Visible == false)
            {
                // Update connector length
                float length = 15.0f;
                ILogicalTreeElement element = this.Parent;
                while (element != null)
                {
                    if (element is SourceDataModelPropertyRenderable)
                    {
                        length += SourceDataModelPropertyRenderable.ChildMargin;
                    }

                    element = element.Parent;
                }

                this.Connector.Visible = true;
                this.Connector.ConnectorLength = length;
            }
            else
            {
                this.Connector.Visible = false;
            }

            base.UpdateChildren(context);

            // Adjust my own height according to the field group's height.
            if (this.dataModelRenderable != null && this.dataModelRenderable.Visible == true)
            {
                this.Size =
                    new SizeF(this.Width, this.dataModelRenderable.Height + (ChildMargin * 2.0f));
            }
            else
            {
                this.Size = new SizeF(this.Width, NonDataModelPropertyHeight);
            }
        }

        /// <summary>
        /// Render the item's foreground.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawForeground(Graphics g)
        {
            // Do not render if contains a field group.
            if (this.dataModelRenderable != null &&
                this.dataModelRenderable.Visible == true)
            {
                return;
            }

            const float SpaceWidth = 5.0f;

            SizeF typeTextSize = g.MeasureString(
                this.PropertyType,
                SourceDataModelPropertyRenderable.PropertyTypeFont);

            SizeF nameTextSize = g.MeasureString(
                this.PropertyName,
                SourceDataModelPropertyRenderable.PropertyNameFont);

            // The value type text rectangle.
            RectangleF valueTypeTextRect = this.TransformedBounds;
            valueTypeTextRect.Width = typeTextSize.Width;

            // The property name occupies the right side of the item.
            RectangleF propertyNameRect = this.TransformedBounds;
            propertyNameRect.X += typeTextSize.Width + SpaceWidth;
            propertyNameRect.Width = nameTextSize.Width;

            using (var strFormat = new StringFormat())
            {
                // Bottom-left alignment.
                strFormat.Alignment = StringAlignment.Near;
                strFormat.LineAlignment = StringAlignment.Center;

                strFormat.Trimming = StringTrimming.EllipsisPath;

                g.DrawString(
                    this.PropertyType,
                    SourceDataModelPropertyRenderable.PropertyTypeFont,
                    Brushes.DimGray,
                    valueTypeTextRect,
                    strFormat);

                strFormat.Trimming = StringTrimming.EllipsisCharacter;

                g.DrawString(
                    this.PropertyName,
                    SourceDataModelPropertyRenderable.PropertyNameFont,
                    Brushes.Blue,
                    propertyNameRect,
                    strFormat);
            }
        }

        /// <summary>
        /// Update the appearance of the item.
        /// </summary>
        private void UpdateAppearance()
        {
            if (this.dataModelRenderable != null)
            {
                if (this.IsSelected == true)
                {
                    this.dataModelRenderable.SeparatorLineColor = Color.White;
                    this.dataModelRenderable.BorderColor = Color.Black;
                    this.dataModelRenderable.FillColor = Color.DarkSeaGreen;
                }
                else if (this.IsMouseOver == true)
                {
                    this.dataModelRenderable.SeparatorLineColor = Color.Silver;
                    this.dataModelRenderable.BorderColor = Color.Black;
                    this.dataModelRenderable.FillColor = Color.Beige;
                }
                else
                {
                    this.dataModelRenderable.SeparatorLineColor = Color.Silver;
                    this.dataModelRenderable.BorderColor = Color.CadetBlue;
                    this.dataModelRenderable.FillColor = Color.Beige;
                }
            }
        }
    }
}
