﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Xml;
using System.Xml.Serialization;

using EffectMaker.DataModelMaker.Core;
using EffectMaker.DataModelMaker.Core.Definitions;
using EffectMaker.DataModelMaker.Core.Template;

namespace EffectMaker.DataModelMaker.Core.Writers
{
    /// <summary>
    /// コンバート処理コードを書き込むライタークラスです.
    /// </summary>
    public class ConvertCodeWriter : SourceCodeWriterBase
    {
        /// <summary>
        /// 定義データです.
        /// </summary>
        private ConvertDataRootDefinition root;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public ConvertCodeWriter()
        {
            this.root = new ConvertDataRootDefinition();
        }

        /// <summary>
        /// 定義からソースコードの文字列を生成します.
        /// </summary>
        /// <param name="def">定義データ.</param>
        /// <returns>生成されたソースコードを返却します.</returns>
        public string Write(ConvertDataRootDefinition def)
        {
            // 各ライターのリスト.
            var namespaceWriters = new List<SourceCodeWriterBase>();
            var converterWriters = new List<SourceCodeWriterBase>();
            var binaryWriters    = new List<SourceCodeWriterBase>();

            // 連想マップ生成.
            var entryListMap = new Dictionary<string, List<SourceCodeWriterBase>>();

            // 連想マップに追加.
            entryListMap.Add("UsingNamespace", namespaceWriters);
            entryListMap.Add("Converter", converterWriters);
            entryListMap.Add("Binarize", binaryWriters);

            // クラスライターを生成.
            var classWriter = new SourceCodeWriter<ConvertDataRootDefinition>(Properties.Resources.CD_Main);

            // テンプレートトークンを評価.
            int length = classWriter.EvaluateTemplateTokens(def);

            // 名前空間を評価.
            length += this.EvaluateUsingNamespace(def, ref namespaceWriters);

            // コンバート方法を評価.
            length += this.EvaluateConverter(def, ref converterWriters);

            // 送信バイナリを評価.
            //length += this.EvaluateSendBinary(binarizeDef, ref binaryWriters);

            // ストリングビルダーを生成.
            StringBuilder builder = new StringBuilder(length);

            // コードファイルとなる文字列を生成.
            for (int i = 0; i < classWriter.EvaluatedTokens.Length; ++i)
            {
                string token = classWriter.EvaluatedTokens[i];
                if (token != null)
                {
                    builder.Append(token);
                }
                else
                {
                    TemplateToken templateToken = classWriter.GetToken(i);
                    if ((templateToken != null) &&
                        (string.IsNullOrEmpty(templateToken.EntryListName) == false) &&
                        (entryListMap.ContainsKey(templateToken.EntryListName) == true))
                    {
                        this.WriteEntryList(builder, entryListMap[templateToken.EntryListName]);
                    }
                }
            }

            // 生成された文字列を返却.
            return builder.ToString();
        }

        #region Private Methods

        /// <summary>
        /// 名前空間定義を評価します.
        /// </summary>
        /// <param name="def">定義データ.</param>
        /// <param name="writers">ライターのリスト.</param>
        /// <returns>評価された文字数を返却します.</returns>
        private int EvaluateUsingNamespace(ConvertDataRootDefinition def, ref List<SourceCodeWriterBase> writers)
        {
            int length = 0;

            // 各名前空間定義分だけループ.
            foreach (ConvertDataUsingNamespaceDefinition nameSpace in def.UsingNamespaces)
            {
                // ライターを生成.
                var writer = new SourceCodeWriter<ConvertDataUsingNamespaceDefinition>(Properties.Resources.CD_UsingNamespace);

                // 出力文字列を生成し，文字数を加算.
                length += writer.EvaluateTemplateTokens(nameSpace);

                // ライターを追加.
                writers.Add(writer);
            }

            return length;
        }

        /// <summary>
        /// データモデル定義を評価します.
        /// </summary>
        /// <param name="def">定義データです.</param>
        /// <param name="writers">ライターです.</param>
        /// <returns>評価された文字数を返却します.</returns>
        private int EvaluateConverter(ConvertDataRootDefinition def, ref List<SourceCodeWriterBase> writers)
        {
            int length = 0;

                // 各コンバート方法ごとにループ.
            foreach (ConverterDefinition convertWay in def.Converters)
            {
                // コンバート方法のライターを生成.
                var wayWriter = new SourceCodeWriter<ConverterDefinition>(Properties.Resources.CD_DataModel);

                // トークンを評価.
                length += wayWriter.EvaluateTemplateTokens(convertWay);

                // トークンの長さを取得.
                int count = wayWriter.EvaluatedTokens.Length;

                {
                    // 入力ライターを生成.
                    var inputWriters = new List<SourceCodeWriterBase>();

                    // 連想マップを生成.
                    var entryListMap = new Dictionary<string, List<SourceCodeWriterBase>>();

                    // 連想マップに追加.
                    entryListMap.Add("Input", inputWriters);

                    int resultCount = 0;

                    // コンバート入力を評価.
                    length += this.EvaluateConvertInput(convertWay, ref inputWriters, out resultCount);

                    // トークン数を加算.
                    count += resultCount;

                    // 出力トークンを生成.
                    string[] tokens = new string[count];

                    int idx = 0;
                    for (int i = 0; i < wayWriter.EvaluatedTokens.Length; ++i)
                    {
                        string token = wayWriter.EvaluatedTokens[i];
                        if (token != null)
                        {
                            tokens[idx] = token;
                            idx++;
                        }
                        else
                        {
                            TemplateToken templateToken = wayWriter.GetToken(i);
                            if ((templateToken != null) &&
                                (string.IsNullOrEmpty(templateToken.EntryListName) == false) &&
                                (entryListMap.ContainsKey(templateToken.EntryListName) == true))
                            {
                                foreach (var writer in entryListMap[templateToken.EntryListName])
                                {
                                    foreach (string val in writer.EvaluatedTokens)
                                    {
                                        tokens[idx] = val;
                                        idx++;
                                    }
                                }
                            }
                        }
                    }

                    // 評価されたトークンを設定.
                    wayWriter.EvaluatedTokens = tokens;
                }

                // ライターを追加.
                writers.Add(wayWriter);
            }


            // 評価された文字数を返却.
            return length;
        }

        /// <summary>
        /// コンバート入力定義を評価します.
        /// </summary>
        /// <param name="def">定義データ.</param>
        /// <param name="writers">ライターリスト.</param>
        /// <param name="count">トークン数.</param>
        /// <returns>評価した文字数を返却します.</returns>
        private int EvaluateConvertInput(ConverterDefinition def, ref List<SourceCodeWriterBase> writers, out int count)
        {
            int length = 0;
            int tokenCount = 0;

            // 各コンバート入力ごとにループ.
            foreach (ConverterInputDefinition input in def.Inputs)
            {
                // ライターを生成.
                var writer = new SourceCodeWriter<ConverterInputDefinition>(Properties.Resources.CD_DataModel_ConverterInput);

                // テンプレートトークンを評価.
                length += writer.EvaluateTemplateTokens(input);

                // トークン数を加算.
                tokenCount += writer.EvaluatedTokens.Length;

                // ライターを追加.
                writers.Add(writer);
            }

            // 出力トークン数を設定.
            count = tokenCount;

            // 評価された文字数を返却.
            return length;
        }

        ///// <summary>
        ///// 送信バイナリ定義を評価します.
        ///// </summary>
        ///// <param name="def">定義データ.</param>
        ///// <param name="writers">ライターリスト.</param>
        ///// <returns>評価した文字数を返却します.</returns>
        //private int EvaluateSendBinary(SendBinaryDefinition def, ref List<SourceCodeWriterBase> writers)
        //{
        //    int length = 0;

        //    foreach (ConvertDataDataModelDefinition dataModel in def.DataModel)
        //    {
        //        // バイナリ出力が定義されていないものはスキップ.
        //        if (dataModel.IsSendBinary == false)
        //        {
        //            continue;
        //        }

        //        // バイナライズのライターを生成.
        //        var binarizeWriter = new SourceCodeWriter<ConvertDataDataModelDefinition>(Properties.Resources.CD_Binarize);

        //        // テンプレートトークンを評価.
        //        length += binarizeWriter.EvaluateTemplateTokens(dataModel);

        //        // トークンの数を設定.
        //        int count = binarizeWriter.EvaluatedTokens.Length;

        //        {
        //            // 各ライターのリストを生成.
        //            var convertWayWriter = new List<SourceCodeWriterBase>();
        //            var bufferWriteWayWriter = new List<SourceCodeWriterBase>();

        //            // 連想マップを生成.
        //            var entryListMap = new Dictionary<string, List<SourceCodeWriterBase>>();

        //            // 連想マップに追加.
        //            entryListMap.Add("ConvertWay", convertWayWriter);
        //            entryListMap.Add("BufferWriteWay", bufferWriteWayWriter);

        //            // 各コンバート方法の定義ごとにループ.
        //            foreach (ConvertDataConvertWayDefinition convertWay in dataModel.ConvertWays)
        //            {
        //                // 入力方法
        //                {
        //                    // ライターを生成.
        //                    var writer = new SourceCodeWriter<ConvertDataConvertWayDefinition>(Properties.Resources.CD_Binarize_ConvertWay);

        //                    // テンプレートを評価.
        //                    length += writer.EvaluateTemplateTokens(convertWay);

        //                    // トークン数を加算.
        //                    count += writer.EvaluatedTokens.Length;

        //                    // ライターを追加.
        //                    convertWayWriter.Add(writer);
        //                }

        //                // 書き込み方法.
        //                {
        //                    // ライターを生成.
        //                    var writer = new SourceCodeWriter<ConvertDataConvertWayDefinition>(Properties.Resources.CD_Binarize_BufferWriteWay);

        //                    // テンプレートを評価.
        //                    length += writer.EvaluateTemplateTokens(convertWay);

        //                    // トークン数を加算.
        //                    count += writer.EvaluatedTokens.Length;

        //                    // ライターを追加.
        //                    bufferWriteWayWriter.Add(writer);
        //                }
        //            }

        //            // トークンの生成.
        //            string[] tokens = new string[count];
        //            int idx = 0;

        //            for (int i = 0; i < binarizeWriter.EvaluatedTokens.Length; ++i)
        //            {
        //                string token = binarizeWriter.EvaluatedTokens[i];
        //                if (token != null)
        //                {
        //                    tokens[idx] = token;
        //                    idx++;
        //                }
        //                else
        //                {
        //                    TemplateToken templateToken = binarizeWriter.GetToken(i);
        //                    if ((templateToken != null)
        //                        && (string.IsNullOrEmpty(templateToken.EntryListName) == false)
        //                        && (entryListMap.ContainsKey(templateToken.EntryListName) == true))
        //                    {
        //                        foreach (var writer in entryListMap[templateToken.EntryListName])
        //                        {
        //                            foreach (string val in writer.EvaluatedTokens)
        //                            {
        //                                tokens[idx] = val;
        //                                idx++;
        //                            }
        //                        }
        //                    }
        //                }
        //            }

        //            // 評価されたトークンを上書き.
        //            binarizeWriter.EvaluatedTokens = tokens;

        //            // ライターリストに追加.
        //            writers.Add(binarizeWriter);
        //        }
        //    }

        //    // 評価された文字数を返却.
        //    return length;
        //}

        #endregion
    }
}
