﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

using EffectMaker.Foundation.Core;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelMaker.Core.Template
{
    /// <summary>
    /// Helper class for encapsulating template related tasks.
    /// </summary>
    public class TemplateHelper
    {
        /// <summary>
        /// Static regular expression for parsing template tags.
        /// </summary>
        private static readonly Regex RegexTemplateTag = new Regex("{TemplateTag:([^.]+).([^}]+)}");

        /// <summary>
        /// Static regular expression for parsing template tags.
        /// </summary>
        private static readonly Regex RegexTemplateTagSimple = new Regex("{TemplateTag:([^}]+)}");

        /// <summary>
        /// Static list of parsed template tokens.
        /// </summary>
        private List<TemplateToken> templateTokens = new List<TemplateToken>();

        /// <summary>
        /// Get or set the handler for each token being replaced with the actual value.
        /// The delegation is as below:
        /// <code>
        /// string TemplateReplaceHandler(TemplateToken token, string original)
        /// </code>
        /// Where token is the token being replacing,
        /// original is the original value that is replacing the token,
        /// and the return value is the actual value to replace the token.
        /// </summary>
        public Func<TemplateToken, string, string> TemplateReplaceHandler { get; set; }

        /// <summary>
        /// Get the template token at the specified index.
        /// </summary>
        /// <param name="index">The index to the template token.</param>
        /// <returns>The template token.</returns>
        public TemplateToken GetToken(int index)
        {
            if ((index < 0) || (index >= this.templateTokens.Count))
            {
                return null;
            }

            return this.templateTokens[index];
        }

        /// <summary>
        /// Parse template string into template tokens.
        /// </summary>
        /// <param name="template">The template.</param>
        /// <returns>True on success.</returns>
        public bool ParseSimpleTemplate(string template)
        {
            // Start index of the text before the current match.
            int startIndex = 0;

            // Match template tags.
            foreach (Match m in RegexTemplateTagSimple.Matches(template))
            {
                if (m.Groups.Count != 2)
                {
                    Logger.Log(LogLevels.Warning, "An invalid template tag is found : {0}", m.Value);
                    return false;
                }

                // Add the text before this template tag.
                if (m.Index - startIndex > 0)
                {
                    string text = template.Substring(startIndex, m.Index - startIndex);
                    this.templateTokens.Add(new TemplateToken()
                    {
                        Type = TemplateTokenTypes.PlainText,
                        PlainText = text,
                    });
                }

                // Add the match as an entry list name.
                this.templateTokens.Add(new TemplateToken()
                {
                    Type = TemplateTokenTypes.TemplateTag,
                    Name = m.Groups[1].Value,
                });

                // Advance the index.
                startIndex = m.Index + m.Length;
            }

            // Add the last part of the plain text in the template.
            if (startIndex < template.Length)
            {
                string text = template.Substring(startIndex, template.Length - startIndex);
                this.templateTokens.Add(new TemplateToken()
                {
                    Type = TemplateTokenTypes.PlainText,
                    PlainText = text,
                });
            }

            return true;
        }

        /// <summary>
        /// Set the value to the specified template tag.
        /// (replace the template tag with the value)
        /// </summary>
        /// <param name="tagName">The name of the template tag.</param>
        /// <param name="value">The value to replace with.</param>
        /// <returns>The number of template tags that are replaced with the value.</returns>
        public int SetTemplateTagValue(string tagName, string value)
        {
            int count = 0;
            foreach (TemplateToken token in this.templateTokens)
            {
                if (string.Equals(token.Name, tagName) == true)
                {
                    token.PlainText = value;
                    ++count;
                }
            }

            return count;
        }

        /// <summary>
        /// Compose the template into a complete text.
        /// </summary>
        /// <param name="indent">Extra indent to add to the text.</param>
        /// <returns>The composed text.</returns>
        public string Compose(int indent = 0)
        {
            var builder = new StringBuilder();
            foreach (TemplateToken token in this.templateTokens)
            {
                builder.Append(token.PlainText);
            }

            string text = builder.ToString();
            if (indent > 0)
            {
                text = RegexUtility.IndentString(text, "\r\n", indent);
            }

            return text;
        }

        /// <summary>
        /// Reset the template tags.
        /// </summary>
        public void ResetTemplateTags()
        {
            foreach (TemplateToken token in this.templateTokens)
            {
                if (token.Type == TemplateTokenTypes.EntryList ||
                    token.Type == TemplateTokenTypes.TemplateTag)
                {
                    token.PlainText = null;
                }
            }
        }

        /// <summary>
        /// Parse template string into template tokens.
        /// </summary>
        /// <param name="template">The template.</param>
        /// <param name="nameSpace">The namespace.</param>
        /// <returns>True on success.</returns>
        public bool ParseTemplate(string template, string nameSpace)
        {
            string definitionNamespace = string.Copy(nameSpace);

            // Start index of the text before the current match.
            int startIndex = 0;

            // Match template tags.
            foreach (Match m in RegexTemplateTag.Matches(template))
            {
                if (m.Groups.Count != 3)
                {
                    Logger.Log(LogLevels.Warning, "An invalid template tag is found : {0}", m.Value);
                    return false;
                }

                // Add the text before this template tag.
                if (m.Index - startIndex > 0)
                {
                    string text = template.Substring(startIndex, m.Index - startIndex);
                    this.templateTokens.Add(new TemplateToken()
                    {
                        Type = TemplateTokenTypes.PlainText,
                        PlainText = text,
                    });
                }

                if (m.Groups[1].Value == "EntryList")
                {
                    // Add the match as an entry list name.
                    this.templateTokens.Add(new TemplateToken()
                    {
                        Type = TemplateTokenTypes.EntryList,
                        Name = m.Groups[2].Value,
                    });
                }
                else
                {
                    Type classType = Type.GetType(definitionNamespace + m.Groups[1].Value + "Definition");
                    if (classType == null)
                    {
                        return false;
                    }

                    PropertyInfo propertyInfo = classType.GetProperty(m.Groups[2].Value);

                    // Add the property info for the match.
                    this.templateTokens.Add(new TemplateToken()
                    {
                        Type = TemplateTokenTypes.TemplateTag,
                        Name = m.Groups[2].Value,
                        PropertyInfo = propertyInfo
                    });
                }

                // Advance the index.
                startIndex = m.Index + m.Length;
            }

            // Add the last part of the plain text in the template.
            if (startIndex < template.Length)
            {
                string text = template.Substring(startIndex, template.Length - startIndex);
                this.templateTokens.Add(new TemplateToken()
                {
                    Type = TemplateTokenTypes.PlainText,
                    PlainText = text,
                });
            }

            return true;
        }

        /// <summary>
        /// Evaluate the actual text represented by the template tokens.
        /// </summary>
        /// <param name="def">The definition.</param>
        /// <param name="evaluatedTokens">The strings evaluated from template tokens.</param>
        /// <returns>The total length of the evaluated text.</returns>
        public int EvaluateTokens(object def, out string[] evaluatedTokens)
        {
            evaluatedTokens = new string[this.templateTokens.Count];

            int length = 0;
            for (int i = 0; i < this.templateTokens.Count; ++i)
            {
                TemplateToken token = this.templateTokens[i];

                if (token.Type == TemplateTokenTypes.EntryList)
                {
                    // The entry list should be evaluated by the owner writer.
                    evaluatedTokens[i] = null;
                    continue;
                }
                else if (token.PropertyInfo != null)
                {
                    object value = token.PropertyInfo.GetValue(def);
                    string replacement = value.ToString();
                    if (this.TemplateReplaceHandler != null)
                    {
                        replacement = this.TemplateReplaceHandler(token, replacement);
                    }

                    evaluatedTokens[i] = replacement;

                    length += evaluatedTokens[i].Length;
                }
                else if (token.PlainText != null)
                {
                    evaluatedTokens[i] = token.PlainText;
                    length += token.PlainText.Length;
                }
            }

            return length;
        }
    }
}
