﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace EffectMaker.DataModelMaker.Core.Definitions
{
    /// <summary>
    /// Interface for objects that the editing version can be locked through EditingVersionLock.
    /// </summary>
    public interface IEditingVersionLockable
    {
        /// <summary>
        /// Get or set the EditingVersionLock instance for locking the editing version.
        /// </summary>
        EditingVersionLock EditingVersionLock { get; set; }
    }

    /// <summary>
    /// Lock the editing version, instead of synchronizing to the global editing version.
    /// </summary>
    public class EditingVersionLock : IDisposable
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="objectToLock">The object to lock.</param>
        /// <param name="editVersion">The version to edit.</param>
        public EditingVersionLock(
            IEditingVersionLockable objectToLock,
            Version editVersion)
        {
            if (objectToLock == null)
            {
                throw new ArgumentNullException("objectToLock");
            }

            this.LockedObject = objectToLock;
            this.EditingVersion = editVersion;

            this.LockedObject.EditingVersionLock = this;
        }

        /// <summary>
        /// Get the locked object.
        /// </summary>
        public IEditingVersionLockable LockedObject { get; private set; }

        /// <summary>
        /// Get the editing version the object is locked at.
        /// </summary>
        public Version EditingVersion { get; private set; }

        /// <summary>
        /// Dispose the object.
        /// </summary>
        public void Dispose()
        {
            this.LockedObject.EditingVersionLock = null;
        }
    }
}
