﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Xml.Serialization;

using EffectMaker.DataModelMaker.Core.Collections;
using EffectMaker.DataModelMaker.Core.Core;

namespace EffectMaker.DataModelMaker.Core.Definitions
{
    /// <summary>
    /// The base class of the definitions.
    /// </summary>
    [Serializable]
    public abstract class DefinitionBase : IGuidObject, IDisposable
    {
        /// <summary>The Guid of the definition.</summary>
        private Guid guid = Guid.NewGuid();

        /// <summary>
        /// Constructor.
        /// </summary>
        public DefinitionBase()
        {
            this.Parent = null;
            WorkspaceManager.RegisterDefinition(this);
        }

        /// <summary>
        /// Get or set the Guid of the definition.
        /// </summary>
        [XmlElement("Guid")]
        public Guid Guid
        {
            get
            {
                return this.guid;
            }

            set
            {
                if (this.guid.Equals(value) == false)
                {
                    WorkspaceManager.UnregisterDefinition(this);
                    this.guid = value;
                    WorkspaceManager.RegisterDefinition(this);
                }
            }
        }

        /// <summary>
        /// Get or set the parent definition.
        /// </summary>
        [XmlIgnore]
        public virtual DefinitionBase Parent { get; set; }

        /// <summary>
        /// Dispose the definition.
        /// </summary>
        public virtual void Dispose()
        {
            WorkspaceManager.UnregisterDefinition(this);
        }

        /// <summary>
        /// Find the first parent definition with the given type.
        /// </summary>
        /// <typeparam name="T">The definition type to find.</typeparam>
        /// <returns>The parent with the type or null if not found.</returns>
        public T FindParent<T>() where T : DefinitionBase
        {
            DefinitionBase parent = this.Parent;
            while (parent != null)
            {
                if (parent is T)
                {
                    return (T)parent;
                }

                parent = parent.Parent;
            }

            return null;
        }

        /// <summary>
        /// Dispose a child.
        /// </summary>
        /// <param name="child">The child to be disposed.</param>
        protected void DisposeChild(IDisposable child)
        {
            if (child != null)
            {
                child.Dispose();
            }
        }

        /// <summary>
        /// Dispose children.
        /// </summary>
        /// <typeparam name="TElement">The type of the elements.</typeparam>
        /// <param name="children">The children to be disposed.</param>
        protected void DisposeChildren<TElement>(IEnumerable<TElement> children)
            where TElement : IDisposable
        {
            foreach (TElement child in children)
            {
                child.Dispose();
            }
        }

        /// <summary>
        /// Set the child definition list.
        /// </summary>
        /// <typeparam name="TElement">The type of the elements.</typeparam>
        /// <param name="child">The child to be set.</param>
        /// <param name="value">The value to set.</param>
        protected void SetChildDefinition<TElement>(
            ref TElement child,
            TElement value) where TElement : DefinitionBase
        {
            if (child != null)
            {
                child.Parent = null;
            }

            child = value;

            if (child != null)
            {
                child.Parent = this;
            }
        }

        /// <summary>
        /// Set the child definition list.
        /// </summary>
        /// <typeparam name="TElement">The type of the elements.</typeparam>
        /// <param name="children">The children to be set.</param>
        /// <param name="value">The value to set to the list.</param>
        protected void SetChildDefinitionList<TElement>(
            ref DefinitionList<TElement> children,
            DefinitionList<TElement> value) where TElement : DefinitionBase
        {
            if (children != null)
            {
                children.Clear();
            }

            children.AddRange(value);
        }
    }
}
