﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;

using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelMaker.Core.Core
{
    /// <summary>
    /// The static class that manages the workspace and all the definitions in the workspace.
    /// </summary>
    public static class WorkspaceManager
    {
        /// <summary>The dictionary maps the definitions and their Guid in the workspace.</summary>
        private static Dictionary<Guid, DefinitionBase> definitions =
            new Dictionary<Guid, DefinitionBase>();

        /// <summary>
        /// Register a new definition.
        /// </summary>
        /// <param name="def">The definition.</param>
        public static void RegisterDefinition(DefinitionBase def)
        {
            if (definitions.ContainsKey(def.Guid) == true)
            {
                Logger.Log(LogLevels.Warning, "WorkspaceManager.RegisterDefinition : The definition Guid {0} has already been registered.", def.Guid.ToString("B"));
                return;
            }

            definitions.Add(def.Guid, def);
        }

        /// <summary>
        /// Unregister a definition.
        /// </summary>
        /// <param name="def">The definition.</param>
        public static void UnregisterDefinition(DefinitionBase def)
        {
            if (definitions.Remove(def.Guid) == false)
            {
                Logger.Log(LogLevels.Warning, "WorkspaceManager.UnregisterDefinition : The definition Guid {0} hasn't been registered.", def.Guid.ToString("B"));
            }
        }

        /// <summary>
        /// Debug method for outputting registered definitions.
        /// </summary>
        public static void OutputRegisteredDefinitions()
        {
            if (definitions.Count <= 0)
            {
                return;
            }

            Logger.Log(LogLevels.Warning, "================================================================================");
            Logger.Log(LogLevels.Warning, "{0} definitions are still registered.", definitions.Count);
            Logger.Log(LogLevels.Warning, "================================================================================");

            foreach (DefinitionBase def in definitions.Select(pair => pair.Value))
            {
                Logger.Log(LogLevels.Warning, "{0} {1}", def.GetType().Name, def.Guid.ToString("D"));
            }

            Logger.Log(LogLevels.Warning, "================================================================================");
        }

        /// <summary>
        /// Find the definition with the specified Guid.
        /// </summary>
        /// <param name="guid">The Guid of the definition to find.</param>
        /// <param name="mute">False to show warning if the reguested Guid is not found.</param>
        /// <returns>The definition with the Guid or null if the definition is not found.</returns>
        public static DefinitionBase FindDefinition(Guid guid, bool mute = false)
        {
            DefinitionBase def;
            if (definitions.TryGetValue(guid, out def) == false)
            {
                if (mute == false)
                {
                    Logger.Log(LogLevels.Warning, "WorkspaceManager.FindDefinition : The definition Guid {0} is not found.", guid.ToString("B"));
                }

                return null;
            }

            return def;
        }
    }
}
