﻿// ========================================================================
// <copyright file="TypeUtility.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Reflection;

    /// <summary>
    /// 型関連のユーティリティです。
    /// </summary>
    public static class TypeUtility
    {
        private static List<Type> nonGlobalAssembyCacheTypes = null;

        //-----------------------------------------------------------------
        // 型情報の取得
        //-----------------------------------------------------------------

        /// <summary>
        /// インターフェースをサポートしているかどうかを調査します。
        /// </summary>
        /// <param name="type">調査する型です。</param>
        /// <param name="interfaceType">
        /// サポートしているかどうかを調べるインターフェースの型です。
        /// </param>
        /// <returns>対象の型がインターフェースをサポートしていれば true を返します。</returns>
        public static bool HasInterface(this Type type, Type interfaceType)
        {
            Ensure.Argument.NotNull(interfaceType);
            Ensure.Argument.True(interfaceType.IsInterface);

            if (type == interfaceType)
            {
                return true;
            }

            Type[] types = type.GetInterfaces();

            foreach (Type t in types)
            {
                if (t == interfaceType)
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// 型をサポートしているかどうかを調査します。
        /// </summary>
        /// <param name="type">調査する型です。</param>
        /// <param name="target">サポートしているかどうかを調べる型です。</param>
        /// <returns>対象の型がサポートしていれば true を返します。</returns>
        public static bool IsSupported(this Type type, Type target)
        {
            Ensure.Argument.NotNull(target);

            if (type == target)
            {
                return true;
            }

            if (target.IsInterface)
            {
                return HasInterface(type, target);
            }
            else
            {
                return type.IsSubclassOf(target);
            }
        }

        /// <summary>
        /// アプリケーションドメインに存在するすべての型から検索対象の型を取得します。
        /// <remarks>
        /// TODO:各所で利用しているが、本来型の列挙はシステムの一箇所で集約して
        /// おこなうべきです。設計を見直して将来的に、本メソッドを廃止するほうが好ましいです。
        /// </remarks>
        /// </summary>
        /// <param name="isSearchTargetType">検索対象の型であるかを決めるデリゲートです。</param>
        /// <returns>検索対象の型のリストです。</returns>
        public static IEnumerable<Type> EnumrateTypes(Predicate<Type> isSearchTargetType)
        {
            if (nonGlobalAssembyCacheTypes == null)
            {
                nonGlobalAssembyCacheTypes = new List<Type>();

                foreach (Assembly domainAssembly in AppDomain.CurrentDomain.GetAssemblies())
                {
                    if (domainAssembly.GlobalAssemblyCache == true)
                    {
                        continue;
                    }

                    nonGlobalAssembyCacheTypes.AddRange(domainAssembly.GetTypes());
                }
            }

            foreach (Type type in nonGlobalAssembyCacheTypes)
            {
                if (isSearchTargetType(type))
                {
                    yield return type;
                }
            }
        }

        //-----------------------------------------------------------------
        // インスタンス生成
        //-----------------------------------------------------------------

        /// <summary>
        /// デフォルトコンストラクタによりインスタンスを生成します。
        /// </summary>
        /// <param name="type">インスタンスを生成する型です。</param>
        /// <returns>生成したインスタンスです。</returns>
        public static object CreateInstance(this Type type)
        {
            return type.Assembly.CreateInstance(type.FullName);
        }

        /// <summary>
        /// 任意のジェネリック型のインスタンスを生成します。
        /// </summary>
        /// <param name="genericType">ジェネリック型です。</param>
        /// <param name="genericTypeArguments">ジェネリック型引数型セットです。</param>
        /// <returns>任意のジェネリック型のインスタンスです。</returns>
        public static object DynamicCreateGenericType(
            Type genericType,
            Type[] genericTypeArguments)
        {
            Ensure.Argument.True(
                genericType.IsGenericType &&
                genericType == genericType.GetGenericTypeDefinition());

            int genericTypeArgumentCount = genericType.GetGenericArguments().Length;
            Ensure.Argument.True(genericTypeArgumentCount == genericTypeArguments.Length);

            Type genericTypeDefinition =
                genericType.GetGenericTypeDefinition();
            Type resultGenericType =
                genericTypeDefinition.MakeGenericType(genericTypeArguments);

            return Activator.CreateInstance(resultGenericType);
        }
    }
}
