﻿// ========================================================================
// <copyright file="LinearRandom.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit
{
    using System;

    /// <summary>
    /// 線形合同法で乱数を生成するクラスです。
    /// <remarks>乱数種はコストラクタで自動的には生成されません。</remarks>
    /// </summary>
    public class LinearRandom : IRandom
    {
        private uint seed = 0;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public LinearRandom()
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定する LinearRandom です。</param>
        public LinearRandom(LinearRandom source)
        {
            this.Set(source);
        }

        /// <summary>
        /// 乱数種を設定します。
        /// </summary>
        /// <param name="seed">設定する乱数種です。</param>
        public void Srand(uint seed)
        {
            this.seed = seed;
        }

        /// <summary>
        /// 0からmaxValueまでの乱数をintで得ます。
        /// <remarks>
        /// 0は含みますが、maxValueを含みません。
        /// 最大65535です。範囲外の場合は、ArgumentOutOfRangeExceptionを送出します。
        /// </remarks>
        /// </summary>
        /// <param name="maxValue">乱数の最大値です。</param>
        /// <returns>0からmaxValueまでの乱数を返します。</returns>
        public int Next(int maxValue)
        {
            if (maxValue < 0 || maxValue > 65535)
            {
                throw new ArgumentOutOfRangeException();
            }

            this.MixRandomSeed();
            return (int)(this.seed >> 8) % maxValue;
        }

        /// <summary>
        /// 0から1の乱数をfloatで得ます。
        /// <remarks>0は含みますが、1を含みません。</remarks>
        /// </summary>
        /// <returns>0から1の乱数を返します。</returns>
        public float NextFloat()
        {
            this.MixRandomSeed();
            return (float)((this.seed >> 16) & 0xffff) / 65536.0f;
        }

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        /// <param name="source">設定するオブジェクトです。</param>
        public void Set(object source)
        {
            this.Set(source as LinearRandom);
        }

        /// <summary>
        /// LinerRandomを設定します。
        /// </summary>
        /// <param name="source">設定するLinearRandomです。</param>
        public void Set(LinearRandom source)
        {
            Ensure.Argument.NotNull(source);

            this.seed = source.seed;
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public object Clone()
        {
            return new LinearRandom(this);
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public bool Equals(LinearRandom other)
        {
            if (other == null)
            {
                return false;
            }

            return this.seed == other.seed;
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public override bool Equals(object other)
        {
            return this.Equals(other as LinearRandom);
        }

        /// <summary>
        /// ハッシュ値を取得します。
        /// </summary>
        /// <returns>ハッシュ値です。</returns>
        public override int GetHashCode()
        {
            return (int)this.seed;
        }

        /// <summary>
        /// 乱数種を更新します。
        /// </summary>
        private void MixRandomSeed()
        {
            this.seed = (this.seed * 214013u) + 2531011u;
        }
    }
}
