﻿// ========================================================================
// <copyright file="BoundingSphere.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;

    /// <summary>
    /// 球のボリューム情報クラスです。
    /// </summary>
    public class BoundingSphere : BoundingVolume
    {
        /// <summary>
        /// 球のデフォルトサイズです。
        /// </summary>
        public const float DefaultSize = 1.0f;

        private Vector3 centerPosition = new Vector3();
        private float radius = DefaultSize;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public BoundingSphere()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">コピー元のデータです。</param>
        public BoundingSphere(BoundingSphere source)
            : this()
        {
            this.Set(source);
        }

        /// <summary>
        /// 中心位置を取得します。
        /// </summary>
        public Vector3 CenterPosition
        {
            get
            {
                return this.centerPosition;
            }

            set
            {
                this.centerPosition = value;
            }
        }

        /// <summary>
        /// 半径を取得します。
        /// </summary>
        public float Radius
        {
            get { return this.radius; }
            set { this.radius = value; }
        }

        /// <summary>
        /// 現在のインスタンスのコピーである新しいオブジェクトを作成します。
        /// </summary>
        /// <returns>このインスタンスのコピーである新しいオブジェクトです。</returns>
        public override object Clone()
        {
            return new BoundingSphere(this);
        }

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        /// <param name="source">設定するオブジェクトです。</param>
        public override void Set(object source)
        {
            this.Set(source as BoundingSphere);
        }

        /// <summary>
        /// 点の集合から AABB を設定します。
        /// </summary>
        /// <param name="positions">内包させたい点座標です。</param>
        public void Set(IList<Vector3> positions)
        {
            //// とりあえず実装。AABB の中心を中心点として、半径は中心から最も離れている点を検索している。
            //// 正規分布計算した主軸から中心点と半径を求めて、外れる点が見つかったら調整する方法がベターそう。

            if (positions.Count > 0)
            {
                Vector3 min_pos = new Vector3(float.MaxValue, float.MaxValue, float.MaxValue);
                Vector3 max_pos = new Vector3(float.MinValue, float.MinValue, float.MinValue);

                foreach (Vector3 pos in positions)
                {
                    min_pos.X = Math.Min(min_pos.X, pos.X);
                    min_pos.Y = Math.Min(min_pos.Y, pos.Y);
                    min_pos.Z = Math.Min(min_pos.Z, pos.Z);

                    max_pos.X = Math.Max(max_pos.X, pos.X);
                    max_pos.Y = Math.Max(max_pos.Y, pos.Y);
                    max_pos.Z = Math.Max(max_pos.Z, pos.Z);
                }

                Assertion.Operation.True(min_pos.X <= max_pos.X);
                Assertion.Operation.True(min_pos.Y <= max_pos.Y);
                Assertion.Operation.True(min_pos.Z <= max_pos.Z);

                this.CenterPosition.Set((max_pos + min_pos) * 0.5f);
                float maxDistance = 0.0f;

                foreach (Vector3 pos in positions)
                {
                    float distance = (pos - this.CenterPosition).Length;

                    if (maxDistance < distance)
                    {
                        maxDistance = distance;
                    }
                }

                this.Radius = maxDistance;
            }
        }

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        /// <param name="source">設定するオブジェクトです。</param>
        protected void Set(BoundingSphere source)
        {
            Ensure.Argument.NotNull(source);

            this.CenterPosition.Set(source.CenterPosition);
            this.Radius = source.Radius;
        }
    }
}
