﻿// ========================================================================
// <copyright file="XmlComplexWriter.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================
namespace NintendoWare.ToolDevelopmentKit.Xml.Complex
{
    using System.IO;
    using System.Text;
    using System.Xml;

    /// <summary>
    /// 複数種類のリソースが格納されたデータを、XML文章に書き出すクラスです。
    /// </summary>
    public class XmlComplexWriter : IXmlComplexWriter
    {
        private XmlWriterSettings xmlWriterSettings;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public XmlComplexWriter()
        {
            this.xmlWriterSettings = new XmlWriterSettings();
            this.xmlWriterSettings.Indent = true;
            this.xmlWriterSettings.IndentChars = "\t";
            this.xmlWriterSettings.OmitXmlDeclaration = false;
            this.xmlWriterSettings.CloseOutput = false;
        }

        /// <summary>
        /// ファイルにデータ構造を、XML文章として書き出します。
        /// </summary>
        /// <param name="document">書き出し対象となるデータ構造です。</param>
        /// <param name="path">ファイルパスです。</param>
        /// <param name="mode">ファイルモードです。</param>
        public void Write(IXmlComplexDocument document, string path, FileMode mode)
        {
            Ensure.Argument.NotNull(document);
            Ensure.Argument.StringNotEmpty(path);

            this.Write(document, new FileStream(path, mode));
        }

        /// <summary>
        /// ストリームにデータ構造を、XML文章として書き出します。
        /// </summary>
        /// <param name="document">書き出し対象となるデータ構造です。</param>
        /// <param name="stream">書き出し先ストリームです。</param>
        public void Write(IXmlComplexDocument document, Stream stream)
        {
            Ensure.Argument.NotNull(document);
            Ensure.Argument.NotNull(stream);
            var xmlTextWriter = XmlTextWriter.Create(stream, this.xmlWriterSettings);
            WriteXml(document, xmlTextWriter);
            xmlTextWriter.Flush();
        }

        /// <summary>
        /// ストリームにデータ構造を、XML文章として書き出します。
        /// </summary>
        /// <param name="document">書き出し対象となるデータ構造です。</param>
        /// <param name="textWriter">書き出し先ストリームです。</param>
        public void Write(IXmlComplexDocument document, TextWriter textWriter)
        {
            Ensure.Argument.NotNull(document);
            Ensure.Argument.NotNull(textWriter);
            var xmlTextWriter = XmlTextWriter.Create(textWriter, this.xmlWriterSettings);
            WriteXml(document, xmlTextWriter);
            xmlTextWriter.Flush();
        }

        /// <summary>
        /// データ構造を、XML文章として書き出します。
        /// </summary>
        /// <param name="document">書き出し対象となるデータ構造です。</param>
        /// <returns>XML文章です。</returns>
        public string CreateString(IXmlComplexDocument document)
        {
            Ensure.Argument.NotNull(document);

            // XmlWriterSettings.Encodingを機能させるため、
            // StringBuilderを使ったインスタンス生成はできない
            // (エンコーディングがutf-16になってしまう)。
            // ただし、ここでは指定しないため問題になっていない。
            MemoryStream resultStream = new MemoryStream();
            XmlWriter writer = XmlTextWriter.Create(resultStream, this.xmlWriterSettings);

            WriteXml(document, writer);

            writer.Close();

            resultStream.Position = 0;
            StreamReader reader = new StreamReader(resultStream, Encoding.Unicode);
            return reader.ReadToEnd();
        }

        /// <summary>
        /// データ構造を、XML文章としてXmlWriterに書き出します。
        /// </summary>
        /// <param name="document">書き出し対象となるデータ構造です。</param>
        /// <param name="writer">XML書き出しを担当するXmlWriterクラスです。</param>
        private static void WriteXml(IXmlComplexDocument document, XmlWriter writer)
        {
            Ensure.Argument.NotNull(document);
            Ensure.Argument.NotNull(writer);

            // RootNode開始タグを書きます。
            writer.WriteStartDocument();
            writer.WriteStartElement(document.RootElementName);
            foreach (IXmlContent content in document.Contents)
            {
                content.Export(writer);
            }

            // RootNode終了タグを書きます。
            writer.WriteEndElement();
        }
    }
}
