﻿// ========================================================================
// <copyright file="LogInput.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit.Logs
{
    using System;

    /// <summary>
    /// ログを入力するログ入力です。
    /// </summary>
    public class LogInput : ILogInput
    {
        //-----------------------------------------------------------------
        private LogMessageTranslator messageTranslater =
            delegate(string message) { return message; };

        private object lockObject = new object();

        //-----------------------------------------------------------------
        // ログのリスナ
        //-----------------------------------------------------------------

        /// <summary>
        /// ログのリスナイベントです。
        /// </summary>
        public event LogListener LogListener;

        //-----------------------------------------------------------------
        // ログの翻訳
        //-----------------------------------------------------------------

        /// <summary>
        /// ログメッセージの翻訳デリゲートを取得または設定します。
        /// </summary>
        public LogMessageTranslator MessageTranslater
        {
            get
            {
                return this.messageTranslater;
            }

            set
            {
                Ensure.Argument.NotNull(value);

                this.messageTranslater = value;
            }
        }

        //-----------------------------------------------------------------
        // 出力位置を指定せずに、メッセージのみのログを追加
        //-----------------------------------------------------------------

        /// <summary>
        /// ログメッセージを追加します。
        /// </summary>
        /// <param name="message">ログのメッセージです。</param>
        public virtual void AddMessage(string message)
        {
            this.Add(new NullLogLocation(), message);
        }

        /// <summary>
        /// メッセージをフォーマットしてログメッセージを追加します。
        /// </summary>
        /// <param name="message">フォーマットするログのメッセージです。</param>
        /// <param name="args">フォーマット引数です。</param>
        public virtual void FormatMessage(string message, params object[] args)
        {
            this.Format(new NullLogLocation(), message, args);
        }

        //-----------------------------------------------------------------
        // 文字列により出力位置を指定したログを追加
        //-----------------------------------------------------------------

        /// <summary>
        /// ログを追加します。
        /// </summary>
        /// <param name="location">ログの出力位置です。</param>
        /// <param name="message">ログのメッセージです。</param>
        public virtual void Add(string location, string message)
        {
            this.Add(new StringLogLocation(location), message);
        }

        /// <summary>
        /// メッセージをフォーマットしてログを追加します。
        /// </summary>
        /// <param name="location">ログの出力位置です。</param>
        /// <param name="message">フォーマットするログのメッセージです。</param>
        /// <param name="args">フォーマット引数です。</param>
        public virtual void Format(string location, string message, params object[] args)
        {
            this.Format(new StringLogLocation(location), message, args);
        }

        //-----------------------------------------------------------------
        // 例外により出力位置を指定したログを追加
        //-----------------------------------------------------------------

        /// <summary>
        /// ログを追加します。
        /// </summary>
        /// <param name="exception">追加する例外です。</param>
        public virtual void Add(Exception exception)
        {
            this.Add(new ExceptionLogLocation(exception), exception.Message);
        }

        /// <summary>
        /// ログを追加します。
        /// </summary>
        /// <param name="location">ログの出力位置です。</param>
        /// <param name="message">ログのメッセージです。</param>
        public virtual void Add(Exception location, string message)
        {
            this.Add(new ExceptionLogLocation(location), message);
        }

        /// <summary>
        /// メッセージをフォーマットしてログを追加します。
        /// </summary>
        /// <param name="location">ログの出力位置です。</param>
        /// <param name="message">フォーマットするログのメッセージです。</param>
        /// <param name="args">フォーマット引数です。</param>
        public virtual void Format(Exception location, string message, params object[] args)
        {
            // TODO: Add, Format, AddMessage 系関数のまぎらわしさに関する対応検討
            this.Format(new ExceptionLogLocation(location), message, args);
        }

        //-----------------------------------------------------------------
        // ログの追加
        //-----------------------------------------------------------------

        /// <summary>
        /// ログを追加します。
        /// </summary>
        /// <param name="location">ログの出力位置です。</param>
        /// <param name="message">ログのメッセージです。</param>
        protected virtual void Add(ILogLocation location, string message)
        {
            if (this.LogListener == null)
            {
                return;
            }

            string translatedMessage = this.MessageTranslater(message);
            ILog log = this.CreateLog(location, translatedMessage);
            this.AddLog(log);
        }

        /// <summary>
        /// メッセージをフォーマットしてログを追加します。
        /// </summary>
        /// <param name="location">ログの出力位置です。</param>
        /// <param name="message">フォーマットするログのメッセージです。</param>
        /// <param name="args">フォーマット引数です。</param>
        protected virtual void Format(ILogLocation location, string message, params object[] args)
        {
            if (this.LogListener == null)
            {
                return;
            }

            string translatedMessage = string.Format(this.MessageTranslater(message), args);
            ILog log = this.CreateLog(location, translatedMessage);
            this.AddLog(log);
        }

        /// <summary>
        /// ログを作成します。
        /// </summary>
        /// <param name="location">ログの出力位置です。</param>
        /// <param name="translatedMessage">翻訳済みのログのメッセージです。</param>
        /// <returns>作成したログです。</returns>
        protected virtual ILog CreateLog(ILogLocation location, string translatedMessage)
        {
            ILog log = new Log(location);
            log.Message = translatedMessage;
            return log;
        }

        /// <summary>
        /// ログを追加します。
        /// </summary>
        /// <param name="log">追加するログです。</param>
        protected void AddLog(ILog log)
        {
            lock (this.lockObject)
            {
                this.LogListener(log);
            }
        }
    }
}
