﻿// ========================================================================
// <copyright file="GarbageCacheCollector.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit.ComponentModel
{
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// 無効なキャッシュを収集するためのガベージコレクターです。
    /// </summary>
    public sealed class GarbageCacheCollector
    {
#if DEBUG
        private const int DefaultInterval = 10 * 1000;          // 10秒間隔（デバッグ時）
#else
        private const int DefaultInterval = 10 * 60 * 1000;     // 10分間隔
#endif

        private static readonly GarbageCacheCollector instance;

        private readonly HashSet<CacheManager> cacheManagers = new HashSet<CacheManager>();

        private readonly object syncRoot = new object();
        private readonly AutoResetEvent loopEvent = new AutoResetEvent(false);

        private Thread thread;
        private bool isThreadExiting = true;
        private int interval = DefaultInterval;

        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        static GarbageCacheCollector()
        {
            instance = new GarbageCacheCollector();
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        private GarbageCacheCollector()
        {
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// ガベージコレクションの自動実行の有無を取得または設定します。
        /// </summary>
        public static bool IsAutoExecution
        {
            get
            {
                return instance.thread != null;
            }

            set
            {
                if (IsAutoExecution == value)
                {
                    return;
                }

                if (value)
                {
                    instance.StartThread();
                }
                else
                {
                    instance.StopThread();
                }
            }
        }

        /// <summary>
        /// ガベージコレクションの自動実行間隔（ミリ秒）を取得または設定します。
        /// </summary>
        public static int AutoExecutionInterval
        {
            get
            {
                return instance.interval;
            }

            set
            {
                if (instance.interval == value)
                {
                    return;
                }

                IsAutoExecution = value > 0;
                instance.interval = value;
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// キャッシュマネージャを登録します。
        /// </summary>
        /// <param name="cacheManager">キャッシュマネージャを指定します。</param>
        internal static void RegisterCacheManager(CacheManager cacheManager)
        {
            Ensure.Argument.NotNull(cacheManager);

            lock (instance.syncRoot)
            {
                instance.cacheManagers.Add(cacheManager);
            }

            instance.StartThread();
        }

        /// <summary>
        /// キャッシュマネージャの登録を解除します。
        /// </summary>
        /// <param name="cacheManager">キャッシュマネージャを指定します。</param>
        internal static void UnregisterCacheManager(CacheManager cacheManager)
        {
            Ensure.Argument.NotNull(cacheManager);

            lock (instance.syncRoot)
            {
                instance.cacheManagers.Remove(cacheManager);
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// ガベージコレクションスレッドを開始します。
        /// </summary>
        private void StartThread()
        {
            if (this.thread != null)
            {
                return;
            }

            if (AutoExecutionInterval <= 0)
            {
                return;
            }

            this.isThreadExiting = false;

            this.thread = new Thread(parameter => this.GarbageCollectionMain())
            {
                IsBackground = true,
                Priority = ThreadPriority.Lowest,
            };

            this.thread.Start();
        }

        /// <summary>
        /// ガベージコレクションスレッドを停止します。
        /// </summary>
        private void StopThread()
        {
            if (this.thread == null)
            {
                return;
            }

            this.isThreadExiting = true;

            this.loopEvent.Set();
            this.thread.Join();

            this.thread = null;
        }

        /// <summary>
        /// ガベージコレクションスレッドのメインループです。
        /// </summary>
        private void GarbageCollectionMain()
        {
            while (true)
            {
                this.loopEvent.WaitOne(AutoExecutionInterval);

                if (this.isThreadExiting)
                {
                    break;
                }

                lock (this.syncRoot)
                {
                    this.CollectGarbageCaches();
                }
            }
        }

        /// <summary>
        /// 無効なキャッシュを収集します。
        /// </summary>
        private void CollectGarbageCaches()
        {
            foreach (CacheManager cacheManager in this.cacheManagers)
            {
                lock (cacheManager.LockObject)
                {
                    foreach (CacheItem item in cacheManager.CacheItems.ToArray())
                    {
                        if (item.IsValid)
                        {
                            continue;
                        }

                        cacheManager.RemoveCache(item);
                    }
                }
            }
        }
    }
}
