﻿// ========================================================================
// <copyright file="ObservableDictionaryBase.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================
namespace NintendoWare.ToolDevelopmentKit.Collections
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using System.Collections.Specialized;
    using System.ComponentModel;

    /// <summary>
    /// 通知機能付き辞書コレクション基本クラスです。
    /// </summary>
    /// <typeparam name="TKey">ディクショナリー内のキーのテンプレート型です。</typeparam>
    /// <typeparam name="TValue">ディクショナリー内の値のテンプレート型です。</typeparam>
    public class ObservableDictionaryBase<TKey, TValue> :
        IDictionary<TKey, TValue>, INotifyCollectionChanged, INotifyPropertyChanged
    {
        //-----------------------------------------------------------------
        // フィールド
        //-----------------------------------------------------------------
        private object objectLock = new object();
        private Dictionary<TKey, TValue> dictionary;

        //-----------------------------------------------------------------
        // コンストラクタ
        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ObservableDictionaryBase()
        {
            this.dictionary = new Dictionary<TKey, TValue>();
        }

        //-----------------------------------------------------------------
        // イベント
        //-----------------------------------------------------------------

        /// <summary>
        /// コレクション変更イベントです。
        /// </summary>
        public virtual event NotifyCollectionChangedEventHandler CollectionChanged;

        //-----------------------------------------------------------------

        /// <summary>
        /// プロパティ変更イベントです。
        /// </summary>
        protected virtual event PropertyChangedEventHandler PropertyChanged;

        event PropertyChangedEventHandler INotifyPropertyChanged.PropertyChanged
        {
            add
            {
                lock (this.objectLock)
                {
                    this.PropertyChanged += value;
                }
            }

            remove
            {
                lock (this.objectLock)
                {
                    this.PropertyChanged -= value;
                }
            }
        }

        //-----------------------------------------------------------------
        // public プロパティ
        //-----------------------------------------------------------------

        /// <summary>
        /// 格納されている要素の数を取得します。
        /// </summary>
        /// <returns>格納されている要素の数です。</returns>
        public int Count
        {
            get { return this.dictionary.Count; }
        }

        /// <summary>
        /// 読み取り専用かどうかを示す値を取得します。
        /// </summary>
        /// <returns>読み取り専用の場合は trueです。それ以外の場合は false です。</returns>
        bool ICollection<KeyValuePair<TKey, TValue>>.IsReadOnly
        {
            get { return false; }
        }

        /// <summary>
        /// 設定されているすべてのキーを取得します。
        /// </summary>
        /// <returns>設定されているすべてのキーのコレクションです。</returns>
        public ICollection<TKey> Keys
        {
            get { return this.dictionary.Keys; }
        }

        /// <summary>
        /// 設定されているすべての値を取得します。
        /// </summary>
        /// <returns>設定されているすべての値のコレクションです。</returns>
        public ICollection<TValue> Values
        {
            get { return this.dictionary.Values; }
        }

        //-----------------------------------------------------------------
        // protected プロパティ
        //-----------------------------------------------------------------

        /// <summary>
        /// 内部辞書です。
        /// </summary>
        protected IDictionary<TKey, TValue> Dictionary
        {
            get
            {
                return this.dictionary;
            }
        }

        //-----------------------------------------------------------------
        // public インデクサ
        //-----------------------------------------------------------------

        /// <summary>
        /// 指定したキーを持つ要素を取得または設定します。
        /// </summary>
        /// <param name="key">取得または設定する要素のキーです。</param>
        /// <returns>指定したキーを持つ要素です。</returns>
        public virtual TValue this[TKey key]
        {
            get
            {
                return this.dictionary[key];
            }

            set
            {
                TValue oldValue = this.dictionary[key];
                this.dictionary[key] = value;
                PropertyChangedEventArgs ei = new PropertyChangedEventArgs("Item");
                PropertyChangedEventArgs ev = new PropertyChangedEventArgs("Values");
                this.OnPropertyChanged(ei);
                this.OnPropertyChanged(ev);
                NotifyCollectionChangedEventArgs e = new NotifyCollectionChangedEventArgs(
                    NotifyCollectionChangedAction.Replace, value, oldValue);
                this.OnCollectionChanged(e);
            }
        }

        //-----------------------------------------------------------------
        // public メソッド
        //-----------------------------------------------------------------

        /// <summary>
        /// 指定したキーおよび値を持つ要素を追加します。
        /// </summary>
        /// <param name="key">追加する要素のキーです。</param>
        /// <param name="value">追加する要素の値です。</param>
        public void Add(TKey key, TValue value)
        {
            this.dictionary.Add(key, value);

            KeyValuePair<TKey, TValue> pair = new KeyValuePair<TKey, TValue>(key, value);
            NotifyCollectionChangedEventArgs e =
                new NotifyCollectionChangedEventArgs(NotifyCollectionChangedAction.Add, pair);
            this.OnCollectionChanged(e);

            PropertyChangedEventArgs ep = new PropertyChangedEventArgs(null);
            this.OnPropertyChanged(ep);
        }

        /// <summary>
        /// すべての要素を削除します。
        /// </summary>
        public void Clear()
        {
            this.dictionary.Clear();

            NotifyCollectionChangedEventArgs e =
                new NotifyCollectionChangedEventArgs(NotifyCollectionChangedAction.Reset);
            this.OnCollectionChanged(e);

            PropertyChangedEventArgs ep = new PropertyChangedEventArgs(null);
            this.OnPropertyChanged(ep);
        }

        /// <summary>
        /// 指定したキーの要素が格納されているかどうかを確認します。
        /// </summary>
        /// <param name="key">検索されるキーです。</param>
        /// <returns>存在する場合は true です。それ以外の場合は false です。</returns>
        public bool ContainsKey(TKey key)
        {
            return this.dictionary.ContainsKey(key);
        }

        /// <summary>
        /// 指定したキーの要素が格納されているかどうかを確認します。
        /// </summary>
        /// <param name="value">検索されるキーです。</param>
        /// <returns>存在する場合は true です。それ以外の場合は false です。</returns>
        public bool ContainsValue(TValue value)
        {
            return this.dictionary.ContainsValue(value);
        }

        /// <summary>
        /// コレクションを反復処理する列挙子を返します。
        /// </summary>
        /// <returns>コレクションを反復処理するために使用できる列挙子です。</returns>
        public IEnumerator GetEnumerator()
        {
            return this.dictionary.GetEnumerator();
        }

        /// <summary>
        /// 指定したキーを持つ値を削除します。
        /// </summary>
        /// <param name="key">削除する要素のキーです。</param>
        /// <returns>
        /// 要素が見つかり正常に削除された場合は true です。それ以外の場合は false です。
        /// </returns>
        public virtual bool Remove(TKey key)
        {
            TValue value = default(TValue);

            if (this.ContainsKey(key) == true)
            {
                value = this[key];
            }

            bool result = this.dictionary.Remove(key);

            if (result == true)
            {
                KeyValuePair<TKey, TValue> pair = new KeyValuePair<TKey, TValue>(key, value);
                NotifyCollectionChangedEventArgs e = new NotifyCollectionChangedEventArgs(
                    NotifyCollectionChangedAction.Remove, pair);

                this.OnCollectionChanged(e);

                PropertyChangedEventArgs ep = new PropertyChangedEventArgs(null);
                this.OnPropertyChanged(ep);
            }

            return result;
        }

        /// <summary>
        /// 指定したキーに関連付けられている値を取得します。
        /// </summary>
        /// <param name="key">取得する値のキーです。</param>
        /// <param name="value">指定したキーに関連付けられている値が格納されます。</param>
        /// <returns>存在する場合は true です。それ以外の場合は false です。</returns>
        public bool TryGetValue(TKey key, out TValue value)
        {
            return this.dictionary.TryGetValue(key, out value);
        }

        //-----------------------------------------------------------------
        // その他 インターフェイス実装
        //-----------------------------------------------------------------

        /// <summary>
        /// 指定したキーおよび値を持つ要素を追加します。
        /// </summary>
        /// <param name="keyValuePair">追加する要素のキーと値のペアです。</param>
        void ICollection<KeyValuePair<TKey, TValue>>.Add(KeyValuePair<TKey, TValue> keyValuePair)
        {
            this.Add(keyValuePair.Key, keyValuePair.Value);
        }

        /// <summary>
        /// 特定のキーと値のペアが格納されているかどうかを判断します。
        /// </summary>
        /// <param name="keyValuePair">検索されるキーと値のペアです。</param>
        /// <returns>存在する場合は true です。それ以外の場合は false です。</returns>
        bool ICollection<KeyValuePair<TKey, TValue>>.Contains(
            KeyValuePair<TKey, TValue> keyValuePair)
        {
            return
                this.dictionary.ContainsKey(keyValuePair.Key) &&
                this.dictionary.ContainsValue(keyValuePair.Value);
        }

        /// <summary>
        /// 要素を配列にコピーします。配列の特定のインデックスからコピーが開始されます。
        /// </summary>
        /// <param name="array">要素がコピーされる 1 次元のキーと値のペアの配列です。</param>
        /// <param name="arrayIndex">
        /// コピーの開始位置となる array の 0 から始まるインデックス番号です。
        /// </param>
        void ICollection<KeyValuePair<TKey, TValue>>.CopyTo(
            KeyValuePair<TKey, TValue>[] array, int arrayIndex)
        {
            Ensure.Argument.NotNull(array);

            if (arrayIndex < 0)
            {
                throw new ArgumentOutOfRangeException();
            }

            Ensure.Argument.True(
                array.Length > arrayIndex && array.Length - arrayIndex >= this.dictionary.Count);

            foreach (TKey key in this.dictionary.Keys)
            {
                array[arrayIndex] = new KeyValuePair<TKey, TValue>(key, this.dictionary[key]);
                ++arrayIndex;
            }
        }

        /// <summary>
        /// コレクションを反復処理する列挙子を返します。
        /// </summary>
        /// <returns>コレクションを反復処理するために使用できる列挙子です。</returns>
        IEnumerator<KeyValuePair<TKey, TValue>>
            IEnumerable<KeyValuePair<TKey, TValue>>.GetEnumerator()
        {
            return (IEnumerator<KeyValuePair<TKey, TValue>>)this.dictionary.GetEnumerator();
        }

        /// <summary>
        /// 指定したキーを持つ値を削除します。
        /// </summary>
        /// <param name="keyValuePair">削除する要素のキーと値のペアです。</param>
        /// <returns>
        /// 要素が見つかり正常に削除された場合は true です。それ以外の場合は false です。
        /// </returns>
        bool ICollection<KeyValuePair<TKey, TValue>>.Remove(
            KeyValuePair<TKey, TValue> keyValuePair)
        {
            if (((IDictionary)this).Contains(keyValuePair) == true)
            {
                return this.Remove(keyValuePair.Key);
            }
            else
            {
                return false;
            }
        }

        //-----------------------------------------------------------------
        // protected メソッド
        //-----------------------------------------------------------------

        /// <summary>
        /// 指定された引数を使用して CollectionChanged イベントを発生します。
        /// </summary>
        /// <param name="e">発生させるイベントの引数です。</param>
        protected virtual void OnCollectionChanged(NotifyCollectionChangedEventArgs e)
        {
            if (this.CollectionChanged != null)
            {
                this.CollectionChanged(this, e);
            }
        }

        /// <summary>
        /// 指定された引数を使用して PropertyChanged イベントを発生します。
        /// </summary>
        /// <param name="e">発生させるイベントの引数です。</param>
        protected virtual void OnPropertyChanged(PropertyChangedEventArgs e)
        {
            if (this.PropertyChanged != null)
            {
                this.PropertyChanged(this, e);
            }
        }
    }
}
