﻿// ========================================================================
// <copyright file="ListUtility.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================
namespace NintendoWare.ToolDevelopmentKit.Collections
{
    using System;
    using System.Collections;
    using System.Collections.Generic;

    /// <summary>
    /// リスト用ユーティリティ
    /// </summary>
    public static class ListUtility
    {
        //-----------------------------------------------------------------
        // 移動、コピー
        //-----------------------------------------------------------------

        /// <summary>
        /// リストの内容を追加します。
        /// </summary>
        /// <typeparam name="TItem">リストのテンプレート型です。</typeparam>
        /// <param name="source">コピー元の列挙子です。</param>
        /// <param name="destination">コピー先のリストです。</param>
        public static void Add<TItem>(IEnumerable<TItem> source, IList<TItem> destination)
        {
            using (destination.SuspendNotifyCollectionChanged())
            {
                foreach (TItem item in source)
                {
                    destination.Add(item);
                }
            }
        }

        /// <summary>
        /// リストの内容を追加します。
        /// </summary>
        /// <typeparam name="TItem">リストのテンプレート型です。</typeparam>
        /// <param name="source">コピー元の列挙子です。</param>
        /// <param name="destination">コピー先のリストです。</param>
        /// <param name="predicate">追加する条件です。</param>
        public static void AddIf<TItem>(
            IEnumerable<TItem> source,
            IList<TItem> destination,
            Predicate<TItem> predicate)
        {
            using (destination.SuspendNotifyCollectionChanged())
            {
                foreach (TItem item in source)
                {
                    if (predicate(item))
                    {
                        destination.Add(item);
                    }
                }
            }
        }

        /// <summary>
        /// リストの内容をコピーします。
        /// </summary>
        /// <typeparam name="TItem">リストのテンプレート型です。</typeparam>
        /// <param name="source">コピー元の列挙子です。</param>
        /// <param name="destination">コピー先のリストです。</param>
        public static void Copy<TItem>(IEnumerable<TItem> source, IList<TItem> destination)
        {
            using (destination.SuspendNotifyCollectionChanged())
            {
                destination.Clear();
                Add(source, destination);
            }
        }

        /// <summary>
        /// リストの内容を複製して追加します。
        /// </summary>
        /// <typeparam name="TItem">リストのテンプレート型です。</typeparam>
        /// <param name="source">コピー元の列挙子です。</param>
        /// <param name="destination">コピー先のリストです。</param>
        public static void CloneAdd<TItem>(IEnumerable<TItem> source, IList<TItem> destination)
            where TItem : class, ICloneable
        {
            foreach (TItem item in source)
            {
                if (item != null)
                {
                    destination.Add((TItem)item.Clone());
                }
                else
                {
                    destination.Add(null);
                }
            }
        }

        /// <summary>
        /// リストの内容を複製してコピーします。
        /// </summary>
        /// <typeparam name="TItem">リストのテンプレート型です。</typeparam>
        /// <param name="source">コピー元の列挙子です。</param>
        /// <param name="destination">コピー先のリストです。</param>
        public static void CloneCopy<TItem>(IEnumerable<TItem> source, IList<TItem> destination)
            where TItem : class, ICloneable
        {
            destination.Clear();
            CloneAdd(source, destination);
        }

        /// <summary>
        /// リストの内容を全てセットします。
        /// </summary>
        /// <remarks>
        /// sourceとdestinationの要素は必ず同数、NotNullでなくてはなりません。
        /// </remarks>
        /// <typeparam name="TItem">リストのテンプレート型です。</typeparam>
        /// <param name="source">セット元のリストです。</param>
        /// <param name="destination">セット先のリストです。</param>
        public static void SetAll<TItem>(IList<TItem> source, IList<TItem> destination)
            where TItem : class, ISettable
        {
            Ensure.Argument.True(source.Count == destination.Count);

            for (int i = 0; i < destination.Count; i++)
            {
                destination[i].Set(source[i]);
            }
        }

        //-----------------------------------------------------------------
        // 削除
        //-----------------------------------------------------------------

        /// <summary>
        /// 指定された条件にあてはまる要素を削除します。
        /// </summary>
        /// <typeparam name="T">リストアイテムの型です。</typeparam>
        /// <param name="list">処理対象のリストです。</param>
        /// <param name="condition">削除する要素の条件です。</param>
        /// <returns>削除したアイテム数を返します。</returns>
        public static int RemoveAll<T>(
            this IList<T> list, Predicate<T> condition)
        {
            Ensure.Argument.NotNull(list);

            List<T> toRemove = new List<T>();

            int cnt = 0;

            foreach (T value in list)
            {
                if (condition(value))
                {
                    toRemove.Add(value);
                    ++cnt;
                }
            }

            using (list.SuspendNotifyCollectionChanged())
            {
                //// TODO:バイディングしているINotifyCollectionChangedのインスタンスを,
                //// Clear() すると例外が発生します。バグなのか、あとで要調査。
                //// 現状、この処理で回避しています。
                //// RemoveAt の方が、いいか微妙です。

                foreach (T removeItem in toRemove)
                {
                    list.Remove(removeItem);
                }
            }

            return cnt;
        }

        //-----------------------------------------------------------------
        // 検索
        //-----------------------------------------------------------------

        /// <summary>
        /// リストの先頭から条件に一致する要素を検索します。
        /// </summary>
        /// <typeparam name="TItem">リストのテンプレート型です。</typeparam>
        /// <param name="list">対象となるリストです。</param>
        /// <param name="match">
        /// 一致する要素を調べる<see cref="T:System.Predicate`1"/>です。
        /// </param>
        /// <returns>検索対象が存在した場合は、指定した型のインスタンスを返します。存在しない場合は、null を返します。</returns>
        public static TItem Find<TItem>(IList<TItem> list, Predicate<TItem> match)
            where TItem : class
        {
            foreach (TItem item in list)
            {
                if (match(item))
                {
                    return item;
                }
            }

            return null;
        }

        /// <summary>
        /// リストの先頭から条件に一致する要素を検索し番号を返します。
        /// </summary>
        /// <typeparam name="TItem">リストのテンプレート型です。</typeparam>
        /// <param name="list">対象となるリストです。</param>
        /// <param name="match">
        /// 一致する要素を調べる<see cref="T:System.Predicate`1"/>です。
        /// </param>
        /// <returns>検索対象が存在した場合は、指定した型の番号を返します。存在しない場合は、-1 を返します。</returns>
        public static int FindIndex<TItem>(IList<TItem> list, Predicate<TItem> match)
            where TItem : class
        {
            int index = 0;
            foreach (TItem item in list)
            {
                if (match(item))
                {
                    return index;
                }

                index++;
            }

            return -1;
        }

        /// <summary>
        /// リストの先頭から条件に一致する要素を検索します。
        /// </summary>
        /// <typeparam name="TItem">リストのテンプレート型です。</typeparam>
        /// <param name="list">対象となるリストです。</param>
        /// <param name="match">
        /// 一致する要素を調べる<see cref="T:System.Predicate`1"/>です。
        /// </param>
        /// <returns>検索対象が存在した場合は、指定した型のリストを返します。存在しない場合は、空のリスト を返します。</returns>
        public static IList<TItem> FindAll<TItem>(IList<TItem> list, Predicate<TItem> match)
            where TItem : class
        {
            IList<TItem> result = new List<TItem>();
            foreach (TItem item in list)
            {
                if (match(item))
                {
                    result.Add(item);
                }
            }

            return result;
        }

        /// <summary>
        /// リストの先頭から条件に一致する要素を検索します。
        /// </summary>
        /// <typeparam name="TItem">リストのテンプレート型です。</typeparam>
        /// <param name="list">対象となるリストです。</param>
        /// <param name="match">
        /// 一致する要素を調べる<see cref="T:System.Predicate`1"/>です。
        /// </param>
        /// <returns>検索対象が存在した場合は、指定した型のインスタンスを返します。存在しない場合は、null を返します。</returns>
        public static bool Exist<TItem>(IList<TItem> list, Predicate<TItem> match)
            where TItem : class
        {
            return Find(list, match) != null;
        }

        /// <summary>
        /// リストの先頭から指定オブジェクト参照を検索し番号を返します。
        /// </summary>
        /// <typeparam name="TItem">アイテムの型を指定します。</typeparam>
        /// <param name="list">対象となるリストです。</param>
        /// <param name="item">検索対象のオブジェクトです。</param>
        /// <returns>検索対象が存在した場合は、指定した型の番号を返します。存在しない場合は、-1 を返します。</returns>
        public static int FindObjectReferenceIndex<TItem>(this IList<TItem> list, TItem item)
        {
            int listItemIndex = 0;

            foreach (var listItem in list)
            {
                if (object.ReferenceEquals(listItem, item))
                {
                    return listItemIndex;
                }

                listItemIndex++;
            }

            return -1;
        }

        //-----------------------------------------------------------------
        // ソート
        //-----------------------------------------------------------------

        /// <summary>
        /// リストの安定なソートです。
        /// </summary>
        /// <typeparam name="TItem">リストのテンプレート型です。</typeparam>
        /// <param name="list">対象となるリストです。</param>
        /// <param name="comparer">
        /// 要素の比較に使用する<see cref="T:System.Comparison`1"/>です。
        /// </param>
        public static void StableSort<TItem>(
            IList<TItem> list, Comparison<TItem> comparer)
        {
            Ensure.Argument.NotNull(list);
            Ensure.Argument.NotNull(comparer);

            // インデックス配列を作成し、これをソートする
            int[] index = new int[list.Count];
            for (int i = 0; i < index.Length; ++i)
            {
                index[i] = i;
            }

            Array.Sort(
                index,
                delegate(int x, int y)
                {
                    int result = comparer(list[x], list[y]);

                    // 結果が同値だった場合はインデックスでソートする
                    return result != 0 ? result : (x - y);
                });

            // ソート結果の書き戻し
            TItem[] array = new TItem[list.Count];
            for (int i = 0; i < index.Length; ++i)
            {
                array[i] = list[i];
            }

            for (int i = 0; i < index.Length; ++i)
            {
                list[i] = array[index[i]];
            }
        }

        //-----------------------------------------------------------------
        // ハッシュ計算、等値比較
        //-----------------------------------------------------------------

        /// <summary>
        /// リストを比較します。
        /// </summary>
        /// <typeparam name="TItem">リスト要素の型パラメータです。</typeparam>
        /// <param name="lhs">左辺リストです。</param>
        /// <param name="rhs">右辺リストです。</param>
        /// <returns>左辺リストと右辺リストが等値であれば true を返します。</returns>
        public static bool Equals<TItem>(IList<TItem> lhs, IList<TItem> rhs)
        {
            if (lhs == null)
            {
                return rhs == null;
            }

            if (rhs == null)
            {
                return false;
            }

            int count = lhs.Count;
            if (rhs.Count != count)
            {
                return false;
            }

            for (int i = 0; i < count; i++)
            {
                TItem lhsItem = lhs[i];
                TItem rhsItem = rhs[i];
                if (lhsItem == null)
                {
                    if (rhsItem != null)
                    {
                        return false;
                    }
                }
                else if (!lhsItem.Equals(rhsItem))
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// ハッシュコードを取得します。
        /// </summary>
        /// <typeparam name="TItem">リスト要素の型パラメータです。</typeparam>
        /// <param name="array">ハッシュコードを計算するリストです。</param>
        /// <returns>リストのハッシュコードです。</returns>
        public static int GetHashCode<TItem>(IList<TItem> array)
        {
            if (array == null)
            {
                return 0;
            }

            int hashCode = typeof(TItem).GetHashCode();
            foreach (TItem item in array)
            {
                hashCode ^= item.GetHashCode();
            }

            return hashCode;
        }

        //-----------------------------------------------------------------
        // スタック
        //-----------------------------------------------------------------

        /// <summary>
        /// リストの先頭に要素を挿入します。
        /// </summary>
        /// <typeparam name="TItem">リストのテンプレート型です。</typeparam>
        /// <param name="list">対象となるリストです。</param>
        /// <param name="item">挿入される要素です。</param>
        public static void PushFront<TItem>(IList<TItem> list, TItem item)
        {
            Ensure.Argument.NotNull(list);

            list.Insert(0, item);
        }

        /// <summary>
        /// リストの末尾に要素を挿入します。
        /// </summary>
        /// <typeparam name="TItem">リストのテンプレート型です。</typeparam>
        /// <param name="list">対象となるリストです。</param>
        /// <param name="item">挿入される要素です。</param>
        public static void PushBack<TItem>(IList<TItem> list, TItem item)
        {
            Ensure.Argument.NotNull(list);

            list.Insert(list.Count, item);
        }

        /// <summary>
        /// リストの先頭の要素をリストから取り除き、返します。
        /// </summary>
        /// <typeparam name="TItem">リストのテンプレート型です。</typeparam>
        /// <param name="list">対象となるリストです。</param>
        /// <returns>リストの先頭から取り除かれた要素です。</returns>
        public static TItem PopFront<TItem>(IList<TItem> list)
        {
            Ensure.Argument.NotNull(list);

            TItem item = list[0];
            list.RemoveAt(0);
            return item;
        }

        /// <summary>
        /// リストの末尾の要素をリストから取り除き、返します。
        /// </summary>
        /// <typeparam name="TItem">リストのテンプレート型です。</typeparam>
        /// <param name="list">対象となるリストです。</param>
        /// <returns>リストの末尾から取り除かれた要素です。</returns>
        public static TItem PopBack<TItem>(IList<TItem> list)
        {
            Ensure.Argument.NotNull(list);

            TItem item = list[list.Count - 1];
            list.RemoveAt(list.Count - 1);
            return item;
        }

        /// <summary>
        /// リストの先頭にある要素を削除せずに返します。
        /// </summary>
        /// <typeparam name="TItem">リストのテンプレート型です。</typeparam>
        /// <param name="list">対象となるリストです。</param>
        /// <returns>リストの先頭にある要素です。</returns>
        public static TItem PeekFront<TItem>(IList<TItem> list)
        {
            Ensure.Argument.NotNull(list);

            return list[0];
        }

        /// <summary>
        /// リストの末尾にある要素を削除せずに返します。
        /// </summary>
        /// <typeparam name="TItem">リストのテンプレート型です。</typeparam>
        /// <param name="list">対象となるリストです。</param>
        /// <returns>リストの末尾にある要素です。</returns>
        public static TItem PeekBack<TItem>(IList<TItem> list)
        {
            Ensure.Argument.NotNull(list);

            return list[list.Count - 1];
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// コレクション変更通知を一時停止します。
        /// </summary>
        /// <typeparam name="TItem">リストアイテムの型を指定します。</typeparam>
        /// <param name="list">対象のリストを指定します。</param>
        /// <returns>コレクション変更通知のロックオブジェクトを返します。</returns>
        public static IDisposable SuspendNotifyCollectionChanged<TItem>(this IList list)
        {
            var observableList = list as ObservableList<TItem>;

            return observableList == null ?
                null : observableList.SuspendNotifyCollectionChanged();
        }

        /// <summary>
        /// コレクション変更通知を一時停止します。
        /// </summary>
        /// <typeparam name="TItem">リストアイテムの型を指定します。</typeparam>
        /// <param name="list">対象のリストを指定します。</param>
        /// <returns>コレクション変更通知のロックオブジェクトを返します。</returns>
        public static IDisposable SuspendNotifyCollectionChanged<TItem>(this IList<TItem> list)
        {
            var observableList = list as ObservableList<TItem>;

            return observableList == null ?
                null : observableList.SuspendNotifyCollectionChanged();
        }
    }
}
