﻿// ========================================================================
// <copyright file="Path.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using System.Diagnostics;

namespace NWCore.Utility
{
    #region Class for path node data

    /// <summary>
    /// Class for path node data.
    /// </summary>
    public class PathNode
    {
        #region Constructors

        /// <summary>
        /// Constructor for the given name/ID pair string.
        /// </summary>
        /// <param name="nameIDPair">Name/ID pair string to parse from.</param>
        /// <param name="delimiter">The delimiter character separates name and ID.</param>
        /// <param name="bIgnoreID">True to ignore the ID.</param>
        public PathNode( string nameIDPair,
                         char delimiter,
                         bool bIgnoreID )
        {
            this.ID = 0;

            int iIndex = nameIDPair.IndexOf(delimiter);
            if ( iIndex<=0 ||
                 iIndex>=nameIDPair.Length-1 )
            {
                this.Name = string.Copy(nameIDPair);
            }
            else
            {
                Name = nameIDPair.Substring( 0, iIndex );

                // Do we want the ID?
                if ( bIgnoreID==false )
                {
                    // Parse ID
                    uint iID;
                    bool bResult =
                        UInt32.TryParse( nameIDPair.Substring(iIndex+1),
                                         System.Globalization.NumberStyles.HexNumber,
                                         System.Globalization.CultureInfo.CurrentCulture,
                                         out iID );
                    if ( bResult==false )
                        this.ID = 0;
                    else
                        this.ID = iID;
                }
            }
        }


        /// <summary>
        /// Constructor for the given name and ID.
        /// </summary>
        /// <param name="strName">Name of the path node.</param>
        /// <param name="iID">ID of the pair node.</param>
        public PathNode( string strName,
                         uint iID )
        {
            this.Name = string.Copy(strName);
            this.ID   = iID;
        }


        /// <summary>
        /// Constructor for the given name. This node WILL NOT have an ID.
        /// </summary>
        /// <param name="nodeName">Name of the path node.</param>
        public PathNode( string nodeName )
        {
            this.Name = string.Copy(nodeName);
            this.ID   = 0;
        }


        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="src">Source path node to copy from.</param>
        public PathNode( PathNode src )
        {
            Copy(src);
        }

        #endregion

        #region Properties

        /// <summary>
        /// Get or set the name of the path node.
        /// </summary>
        public string Name
        {
            get { return m_name; }
            set
            {
                if ( value==null ||
                     value.Length<=0 )
                {
                    m_name     = string.Empty;
                    m_iNameCRC = 0;
                }
                else
                {
                    m_name     = value;
                    m_iNameCRC = NWKernel.CRC32Helper.ComputeCRC32Str(value);
                }
            }
        }

        /// <summary>
        /// Get the CRC32 hash value of the name.
        /// </summary>
        public uint NameCRC
        {
            get{ return m_iNameCRC; }
        }


        /// <summary>
        /// Get or set the ID of the path node.
        /// </summary>
        public uint ID { get; set; }

        #endregion

        #region Comparison method

        /// <summary>
        /// Compare this node with another.
        ///  1 is returned when this is greater than node
        ///  0 is returned when this equals to node
        /// -1 is returned when this is smaller than node
        /// </summary>
        /// <param name="node">The node to compare to.</param>
        /// <param name="bIgnoreID">True to ignore the ID.</param>
        /// <returns>
        ///  1 is returned when this is greater than node
        ///  0 is returned when this equals to node
        /// -1 is returned when this is smaller than node
        /// </returns>
        public int Compare( PathNode node,
                            bool bIgnoreID = false )
        {
            if ( this.NameCRC>node.NameCRC )
            {
                return 1;
            }
            else if ( this.NameCRC<node.NameCRC )
            {
                return -1;
            }
            else if ( bIgnoreID==false )
            {
                if ( this.ID>0 &&
                     node.ID>0 )
                {
                    if ( this.ID>node.ID )
                        return 1;
                    else if ( this.ID<node.ID )
                        return -1;
                    else
                        return 0;
                }
                else
                {
                    return 0;
                }
            }

            return 0;
        }


        /// <summary>
        /// Test if this path node contains the given node.
        /// If this node has ID but the given node does not, false is returned.
        /// If this node does not have an ID while the given one does, and the
        /// name is the same, true is returned.
        /// </summary>
        /// <param name="node"></param>
        /// <returns></returns>
        public bool Contains( PathNode node )
        {
            if ( this.NameCRC==node.NameCRC )
            {
                if ( this.ID==0 ||
                     this.ID==node.ID )
                {
                    return true;
                }
            }

            return false;
        }


        /// <summary>
        /// Match the given regular expression and ID with the path node.
        /// </summary>
        /// <param name="expression">The regular expression.</param>
        /// <param name="iID">The ID to match.</param>
        /// <returns>True if a match is found.</returns>
        public bool MatchRegex( Regex expression,
                                uint iID )
        {
            if ( expression.IsMatch(this.Name)==true )
            {
                if ( (this.ID==0) ||
                     (iID==0) ||
                     (this.ID==iID) )
                {
                    return true;
                }
            }

            return false;
        }


        /// <summary>
        /// Match the given name and ID with this path node.
        /// </summary>
        /// <param name="name">The string to match.</param>
        /// <param name="iID">The ID to match.</param>
        /// <returns>True if the given name and ID matches this path node.</returns>
        public bool Match( string name,
                           uint iID )
        {
            if ( (ID>0) &&
                 (iID>0) &&
                 (ID!=iID) )
            {
                return false;
            }

            return (this.Name==name);
        }


        /// <summary>
        /// Match the given CRC value of the name and ID with this path node.
        /// </summary>
        /// <param name="nameCRC">The CRC value of the name to match.</param>
        /// <param name="iID">The ID to match.</param>
        /// <returns>True if the given name and ID matches this path node.</returns>
        public bool Match( uint nameCRC,
                           uint iID )
        {
            if ( (ID>0) &&
                 (iID>0) &&
                 (ID!=iID) )
            {
                return false;
            }

            return (nameCRC==this.NameCRC);
        }

        #endregion

        #region Utilities

        /// <summary>
        /// Helper function for making name/ID pair string.
        /// </summary>
        /// <param name="delimiter">The delimiter between name and ID.</param>
        /// <returns>The name/ID pair string.</returns>
        public string ToString( char delimiter )
        {
            if ( ID==0 )
                return Name;

            return Name + delimiter + ID.ToString("X");
        }


        /// <summary>
        /// Helper function for making name/ID pair string with the default
        /// delimiter.
        /// </summary>
        /// <returns>The name/ID pair string.</returns>
        public override string ToString()
        {
            if ( ID==0 )
                return Name;

            return Name + '@' + ID.ToString("X");
        }


        /// <summary>
        /// Clone the path node object.
        /// </summary>
        /// <returns>The clone of the original path node.</returns>
        public PathNode Clone()
        {
            return new PathNode( this );
        }


        /// <summary>
        /// Copy from the given source path node.
        /// </summary>
        /// <param name="src">The path node to copy from.</param>
        public void Copy( PathNode src )
        {
            this.m_name     = src.m_name;
            this.m_iNameCRC = src.m_iNameCRC;
            this.ID         = src.ID;
        }

        #endregion

        #region Member variables

        private string m_name;
        private uint   m_iNameCRC;

        #endregion
    }

    #endregion

    #region Class for a path data

    /// <summary>
    /// Class to represent a path.
    /// All the nodes in the path can have an ID, but not required.
    /// The nodes are hashed separately to maintain the flexibility,
    /// while speed up the operations with CRC32 hash codes.
    /// </summary>
    [DebuggerDisplay("{PathName}")]
    public class Path
    {
        #region Constructors

        /// <summary>
        /// Default constructor.
        /// </summary>
        public Path()
        {
        }


        /// <summary>
        /// Constructor for the path string with IDs.
        /// The nodes in the path will be separated with the default
        /// delimiter characters '/', and '\'. And the default
        /// delimiter for path node name and the Id is '@'. Although
        /// a node in the path does not necessarily have an ID, if it has,
        /// first the name, the delimiter for IDs, then
        /// the ID in HEX format.
        /// Eq. PropertyWindow@A8/EmitterProperty@FF/Life@13
        /// </summary>
        /// <param name="strPath">The path string.</param>
        public Path( string strPath )
        {
            if ( strPath==null ||
                 strPath.Length<=0 )
            {
                m_pathStr = string.Empty;
                m_nodes.Clear();

                return;
            }

            bool bResult = ParsePathStrWithID( strPath,
                                               m_dlmtPath,
                                               m_dlmtID,
                                               false,
                                               out m_nodes );
            if ( bResult==true )
                m_pathStr = strPath;
            else
                m_pathStr = string.Empty;
        }


        /// <summary>
        /// Constructor for the path string with IDs.
        /// The nodes in the path will be separated with the default
        /// delimiter characters '/', and '\'. And the default
        /// delimiter for path node name and the Id is '@'. Although
        /// a node in the path does not necessarily have an ID, if it has,
        /// first the name, the delimiter for IDs, then
        /// the ID in HEX format.
        /// Eq. PropertyWindow@A8/EmitterProperty@FF/Life@13
        /// </summary>
        /// <param name="strPath">The path string.</param>
        /// <param name="bIgnoreID">True to ignore the ID.</param>
        public Path( string strPath,
                     bool bIgnoreID )
        {
            if ( strPath==null ||
                 strPath.Length<=0 )
            {
                m_pathStr = string.Empty;
                m_nodes.Clear();

                return;
            }

            bool bResult = ParsePathStrWithID( strPath,
                                               m_dlmtPath,
                                               m_dlmtID,
                                               bIgnoreID,
                                               out m_nodes );
            if ( bResult==true )
                m_pathStr = strPath;
            else
                m_pathStr = string.Empty;
        }


        /// <summary>
        /// Constructor for the path string.
        /// The nodes in the path will be separated with the default
        /// delimiter characters '/', and '\'. And the default
        /// delimiter for path node name and the Id is '@'. Although
        /// a node in the path does not necessarily have an ID, if it has,
        /// first the name, the delimiter for IDs, then
        /// the ID in HEX format.
        /// Eq. PropertyWindow@A8/EmitterProperty@FF/Life@13
        /// </summary>
        /// <param name="strPath">The path string.</param>
        /// <param name="bUseID">False to specify this path contains no IDs.</param>
        /// <param name="bIgnoreID">True to ignore the ID even when bUseID is true.</param>
        public Path( string strPath,
                     bool bUseID,
                     bool bIgnoreID )
        {
            if ( strPath==null ||
                 strPath.Length<=0 )
            {
                m_pathStr = string.Empty;
                m_nodes.Clear();

                return;
            }

            bool bResult = false;
            if ( bUseID==true )
            {
                bResult = ParsePathStrWithID( strPath,
                                              m_dlmtPath,
                                              m_dlmtID,
                                              bIgnoreID,
                                              out m_nodes );
            }
            else
            {
                bResult = ParsePathStrWithoutID( strPath,
                                                 m_dlmtPath,
                                                 out m_nodes );
            }

            if ( bResult==true )
                m_pathStr = strPath;
            else
                m_pathStr = string.Empty;
        }


        /// <summary>
        /// Constructor for the path string without IDs.
        /// The nodes in the path will be separated with the given
        /// delimiter characters.
        /// If this constructor is called, the default delimiters
        /// for this path instance will be replaced by the given ones.
        /// </summary>
        /// <param name="strPathWithoutID">The path string.</param>
        /// <param name="delimiters">The delimiters for path nodes.</param>
        public Path( string strPathWithoutID,
                     char[] delimiters )
        {
            m_dlmtPath = delimiters;

            if ( strPathWithoutID==null ||
                 strPathWithoutID.Length<=0 )
            {
                m_pathStr = string.Empty;
                m_nodes.Clear();

                return;
            }

            bool bResult = ParsePathStrWithoutID( strPathWithoutID,
                                                  delimiters,
                                                  out m_nodes );
            if ( bResult==true )
                m_pathStr = strPathWithoutID;
            else
                m_pathStr = string.Empty;
        }


        /// <summary>
        /// Constructor for the path string with IDs.
        /// The nodes in the path will be separated with the given
        /// path node and ID delimiter characters. Although a node
        /// in the path does not necessarily have an ID, if it has,
        /// first the name, the delimiter for IDs, then
        /// the ID in HEX format.
        /// Eq. PropertyWindow@A8/EmitterProperty@FF/Life@13
        /// </summary>
        /// <param name="strPathWithID">The path string.</param>
        /// <param name="demilitersForPath">The delimiters for path nodes.</param>
        /// <param name="delimiterForID">The delimiter for path node and ID.</param>
        /// <param name="bIgnoreID">True to ignore the ID.</param>
        public Path( string strPathWithID,
                     char[] demilitersForPath,
                     char delimiterForID,
                     bool bIgnoreID )
        {
            m_dlmtPath = demilitersForPath;
            m_dlmtID   = delimiterForID;

            if ( strPathWithID==null ||
                 strPathWithID.Length<=0 )
            {
                m_pathStr = string.Empty;
                m_nodes.Clear();

                return;
            }

            bool bResult = ParsePathStrWithID( strPathWithID,
                                               m_dlmtPath,
                                               m_dlmtID,
                                               bIgnoreID,
                                               out m_nodes );
            if ( bResult==true )
                m_pathStr = strPathWithID;
            else
                m_pathStr = string.Empty;
        }


        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="src">The source path object to copy from.</param>
        public Path( Path src )
        {
            Copy( src );
        }


        /// <summary>
        /// Copy constructor with maximum number of path nodes to copy.
        /// </summary>
        /// <param name="src">The source path object to copy from.</param>
        /// <param name="iNumMaxPathNodes">The maximum number of path nodes to copy.</param>
        public Path( Path src,
                     int iNumMaxPathNodes )
        {
            Copy( src, iNumMaxPathNodes );
        }

        #endregion

        #region Properties

        /// <summary>
        /// Get or set the full path name of the path object.
        /// </summary>
        public string PathName
        {
            get { return m_pathStr; }
            set
            {
                if ( value==null ||
                     value.Length<=0 )
                {
                    m_pathStr = string.Empty;
                    m_nodes.Clear();
                }

                // Parse the path
                bool bResult = ParsePathStrWithID( value,
                                                   m_dlmtPath,
                                                   m_dlmtID,
                                                   false,
                                                   out m_nodes );
                if ( bResult==true )
                    m_pathStr = value;
                else
                    m_pathStr = string.Empty;
            }
        }


        /// <summary>
        /// Get the number of the path nodes.
        /// </summary>
        public int Length
        {
            get { return m_nodes.Count; }
        }


        /// <summary>
        /// Get or set the delimiter for path nodes.
        /// </summary>
        public char PathDelimiter
        {
            get { return m_dlmtPath[0]; }
            set
            {
                m_dlmtPath[0] = value;
                UpdatePathString();
            }
        }


        /// <summary>
        /// Get or set the delimiter for name and ID
        /// </summary>
        public char IDDelimiter
        {
            get { return m_dlmtID; }
            set
            {
                m_dlmtID = value;
                UpdatePathString();
            }
        }

        #endregion

        #region Comparison

        /// <summary>
        /// Compare another path object with this.
        /// </summary>
        /// <param name="pathObj">The path to compare with.</param>
        /// <param name="bIgnoreID">True to ignore the IDs in the paths.</param>
        /// <returns>
        ///  1 is returned when this is greater than the given path
        ///  0 is returned when this equals to the given path
        /// -1 is returned when this is smaller than the given path
        /// </returns>
        public int Compare( Path pathObj,
                            bool bIgnoreID = false )
        {
            if ( pathObj==null )
                return 1;

            List<PathNode> nodes1 = this.m_nodes;
            List<PathNode> nodes2 = pathObj.m_nodes;

            int i;
            for ( i=0;(i<nodes1.Count) && (i<nodes2.Count);++i )
            {
                int iResult = nodes1[i].Compare(nodes2[i], bIgnoreID);
                if ( iResult!=0 )
                    return iResult;
            }

            if ( i>=nodes1.Count &&
                 i<nodes2.Count )
            {
                return -1;
            }
            else if ( i<nodes1.Count &&
                      i>=nodes2.Count )
            {
                return 1;
            }

            return 0;
        }


        /// <summary>
        /// Compare another path object with this from the end of both paths.
        /// </summary>
        /// <param name="path">The other path object.</param>
        /// <returns>
        ///  1 is returned when this is greater than the given path
        ///  0 is returned when this equals to the given path
        /// -1 is returned when this is smaller than the given path
        /// </returns>
        public int CompareFromEnd( Path path )
        {
            if ( path==null )
                return 1;

            List<PathNode> nodes1 = this.m_nodes;
            List<PathNode> nodes2 = path.m_nodes;

            int i, j;
            for ( i=(this.Length-1), j=(path.Length-1);i>=0 && j>=0;--i, --j )
            {
                int iResult = nodes1[i].Compare(nodes2[j]);
                if ( iResult!=0 )
                    return iResult;
            }

            if ( i<0 && j>=0 )
            {
                return 1;
            }
            else if ( j<0 && i>=0 )
            {
                return -1;
            }

            return 0;
        }


        /// <summary>
        /// Match the given path from the beginning of this path.
        /// </summary>
        /// <param name="pathStr">The path to match with.</param>
        /// <returns>True on match.</returns>
        public bool Match( string pathStr )
        {
            Path path = new Path( pathStr );
            return Match( path );
        }


        /// <summary>
        /// Match the given path from the beginning of this path.
        /// </summary>
        /// <param name="path">The path to match with.</param>
        /// <returns>True on match.</returns>
        public bool Match( Path path )
        {
            // It won't match if the given path is longer than this.
            if ( path.Length>this.Length )
                return false;

            // Match the paths
            for ( int i=0;i<this.Length;++i )
            {
                if ( i>=path.Length )
                    return true;

                if ( this[i].Compare( path[i] )!=0 )
                    return false;
            }

            return true;
        }


        /// <summary>
        /// Match the given path from the end of this path.
        /// </summary>
        /// <param name="pathStr">The path to match with.</param>
        /// <param name="bIgnoreID">True to ignore ID.</param>
        /// <returns>True on match.</returns>
        public bool MatchFromEnd( string pathStr,
                                  bool bIgnoreID = false )
        {
            Path path = new Path( pathStr );
            return MatchFromEnd( path, bIgnoreID );
        }


        /// <summary>
        /// Match the given path from the end of this path.
        /// </summary>
        /// <param name="path">The path to match with.</param>
        /// <param name="bIgnoreID">True to ignore ID.</param>
        /// <returns>True on match.</returns>
        public bool MatchFromEnd( Path path,
                                  bool bIgnoreID = false )
        {
            // It won't match if the given path is longer than this.
            if ( path.Length>this.Length )
                return false;

            // Match the paths
            int i, j;
            for ( i=(this.Length-1), j=(path.Length-1);i>=0 && j>=0;--i, --j )
            {
                if ( this[i].Compare(path[j], bIgnoreID)!=0 )
                    return false;
            }

            return true;
        }


        /// <summary>
        /// Find if there is any match of the given path occurs in this path.
        /// </summary>
        /// <param name="pathStr">The path to match with.</param>
        /// <returns>True on match.</returns>
        public bool MatchAny( string pathStr )
        {
            Path path = new Path( pathStr );
            return MatchAny( path );
        }


        /// <summary>
        /// Find if there is any match of the given path occurs in this path.
        /// </summary>
        /// <param name="path">The path to match with.</param>
        /// <returns>True on match.</returns>
        public bool MatchAny( Path path )
        {
            // It won't match if the given path is longer than this.
            if ( path.Length>this.Length )
                return false;

            // Match the paths
            for ( int i=0;i<this.Length;++i )
            {
                if ( i>=path.Length )
                    return false;

                if ( this[i].Compare( path[0] )==0 )
                {
                    for ( int j=1;j<path.Length;++j )
                    {
                        if ( (i + j)>=this.Length )
                            return false;

                        if ( this[i+j].Compare( path[j] )!=0 )
                            return false;
                    }

                    return true;
                }
            }

            return false;
        }


        /// <summary>
        /// Test if this path contains the given path.
        /// </summary>
        /// <param name="path">The path to test.</param>
        /// <returns>True if this path contains the given path.</returns>
        public bool Contains( string pathStr )
        {
            Path path = new Path( pathStr );
            return this.Contains( path );
        }


        /// <summary>
        /// Test if this path contains the given path.
        /// </summary>
        /// <param name="path">The path to test.</param>
        /// <returns>True if this path contains the given path.</returns>
        public bool Contains( Path path )
        {
            // The length of this path must be shorter than the given path
            // to contain it.
            // Eg. "C:\\SomeDir" contains "C:\\SomeDir\SomeFile.ext"
            if ( this.Length>path.Length )
                return false;

            int i;
            for ( i=0;i<this.Length;++i )
            {
                if ( this[i].Contains(path[i])==false )
                    return false;
            }

            return true;
        }

        #endregion

        #region Overloaded operators

        /// <summary>
        /// Overloaded operator +.
        /// </summary>
        /// <param name="objL">Left path object operand.</param>
        /// <param name="strR">Right path string operand.</param>
        /// <returns>The combined path.</returns>
        public static Path operator + ( Path objL,
                                        string strR )
        {
            List<PathNode> nodes;
            bool bResult = ParsePathStrWithID( strR,
                                               objL.m_dlmtPath,
                                               objL.m_dlmtID,
                                               false,
                                               out nodes );

            // Clone the left side operand
            Path result = objL.Clone();

            int iNumNodesL = objL.m_nodes.Count;
            int iNumNodesR = nodes.Count;

            // Reserve memory for path nodes first
            result.m_nodes.Capacity = iNumNodesL + iNumNodesR;

            // Clone the path nodes from the right side operand
            foreach ( PathNode node in nodes )
                result.m_nodes.Add( node.Clone() );

            // Produce the path string
            result.UpdatePathString();

            // For the delimiters,
            // we want to use the ones from the left side operand
            // which we did clone earlier, hence they are already
            // taken care of.

            return result;
        }


        /// <summary>
        /// Overloaded operator +.
        /// </summary>
        /// <param name="objL">Left path object operand.</param>
        /// <param name="objR">Right path object operand.</param>
        /// <returns>The combined path.</returns>
        public static Path operator + ( Path objL,
                                        Path objR )
        {
            // Clone the left side operand
            Path result = objL.Clone();

            int iNumNodesL = objL.m_nodes.Count;
            int iNumNodesR = objR.m_nodes.Count;

            // Reserve memory for path nodes first
            result.m_nodes.Capacity = iNumNodesL + iNumNodesR;

            // Clone the path nodes from the right side operand
            foreach ( PathNode node in objR.m_nodes )
                result.m_nodes.Add( node.Clone() );

            // Produce the path string
            result.UpdatePathString();

            // For the delimiters,
            // we want to use the ones from the left side operand
            // which we did clone earlier, hence they are already
            // taken care of.

            return result;
        }


        /// <summary>
        /// Overloaded indexer to provide an array-like
        /// access to the path nodes.
        /// </summary>
        /// <param name="iIndex">Index of the path node to get.</param>
        /// <returns>The path node at the index.</returns>
        public PathNode this[ int iIndex ]
        {
            get
            {
                if ( iIndex<0 || iIndex>=m_nodes.Count )
                    return null;

                return m_nodes[iIndex];
            }
        }

        #endregion

        #region Find path

        /// <summary>
        /// Find a matching sub path with the given path string.
        /// If a match is found, a new path object of the matching path
        /// is returned, otherwise null is returned.
        /// </summary>
        /// <param name="pathStr">The path string to find.</param>
        /// <returns>The matching path object, null if no match is found.</returns>
        public Path FindSubPath( string pathStr )
        {
            // Parse the path nodes from the given path string.
            List<PathNode> nodes2;
            bool bResult = ParsePathStrWithID( pathStr,
                                               m_dlmtPath,
                                               m_dlmtID,
                                               false,
                                               out nodes2 );
            if ( bResult==false )
                return null;

            // Is the path longer than this path object?
            if ( this.m_nodes.Count<nodes2.Count )
                return null;

            List<PathNode> nodes1 = this.m_nodes;

            int i;
            for ( i=0;i<nodes2.Count;++i )
            {
                // Test if the path nodes match each other
                bResult = nodes1[i].Match( nodes2[i].NameCRC,
                                           nodes2[i].ID );
                // Hash value of the names do not match
                if ( bResult==false )
                    return null;
            }

            return new Path( this, i );
        }


        /// <summary>
        /// Find a matching sub path with the given regular expression path string.
        /// The regular expression only applies to each of the path node, and
        /// each path nodes is still separated by the delimiters. So be careful
        /// if there is any of these delimiters in your regular expressions.
        /// If a match is found, a new path object of the matching path
        /// is returned, otherwise null is returned.
        /// </summary>
        /// <param name="pathStr">The path string to find.</param>
        /// <param name="options">The regular expression options for matching.</param>
        /// <returns>The matching path object, null if no match is found.</returns>
        public Path FindSubPathWithRegex( string pathStr,
                                          RegexOptions options )
        {
            // Parse the path nodes from the given path string.
            List<PathNode> nodes2;
            bool bResult = ParsePathStrWithID( pathStr,
                                               m_dlmtPath,
                                               m_dlmtID,
                                               false,
                                               out nodes2 );
            if ( bResult==false )
                return null;

            // Is the path longer than this path object?
            if ( this.m_nodes.Count<nodes2.Count )
                return null;

            List<PathNode> nodes1 = this.m_nodes;

            int i;
            for ( i=0;i<nodes2.Count;++i )
            {
                // Test if the path nodes match each other
                bResult = nodes1[i].MatchRegex( new Regex(nodes2[i].Name, options),
                                                nodes2[i].ID );
                // Hash value of the names do not match
                if ( bResult==false )
                    return null;
            }

            return new Path( this, i );
        }

        #endregion

        #region Utilities

        /// <summary>
        /// Set the path string.
        /// </summary>
        /// <param name="strPath">The path to set.</param>
        /// <param name="bIgnoreID">True to ignore ID in the string.</param>
        public void SetPath( string strPath,
                             bool bIgnoreID = false )
        {
            m_nodes.Clear();

            if ( strPath==null ||
                 strPath.Length<=0 )
            {
                m_pathStr = string.Empty;
                return;
            }

            bool bResult = ParsePathStrWithID( strPath,
                                               m_dlmtPath,
                                               m_dlmtID,
                                               bIgnoreID,
                                               out m_nodes );
            UpdatePathString();
        }


        /// <summary>
        /// Append the given path to this.
        /// </summary>
        /// <param name="pathStr">The path string to append.</param>
        public void Append( string pathStr )
        {
            Path path = new Path( pathStr );
            this.Append( path );
        }


        /// <summary>
        /// Append the given path to this.
        /// </summary>
        /// <param name="path">The path to append.</param>
        public void Append( Path path )
        {
            int iNumNodes1 = this.m_nodes.Count;
            int iNumNodes2 = path.m_nodes.Count;

            if ( iNumNodes2<=0 )
                return;

            // Reserve memory for path nodes first
            this.m_nodes.Capacity = iNumNodes1 + iNumNodes2;

            // Clone the path nodes from the right side operand
            foreach ( PathNode node in path.m_nodes )
                this.m_nodes.Add( node.Clone() );

            // Produce the path string
            this.UpdatePathString();
        }


        /// <summary>
        /// Append the given path node to this.
        /// </summary>
        /// <param name="pathNode">The path node to append.</param>
        public void Append( PathNode pathNode )
        {
            if ( pathNode.Name.Length<=0 )
                return;

            // Clone and append the path node
            this.m_nodes.Add( pathNode.Clone() );

            // Produce the path string
            this.UpdatePathString();
        }


        /// <summary>
        /// Move the path up by one level.
        /// </summary>
        public void ToUpperLevel()
        {
            if ( m_nodes.Count<=0 )
                return;

            m_nodes.RemoveAt( m_nodes.Count-1 );

            UpdatePathString();
        }


        /// <summary>
        /// Update path string from the nodes.
        /// </summary>
        public void UpdatePathString()
        {
            if ( m_nodes==null ||
                 m_nodes.Count<=0 )
            {
                m_pathStr = string.Empty;
                return;
            }

            // Make the path string from the path nodes.
            int i;
            for ( i=0;i<m_nodes.Count;++i )
            {
                if ( i>0 )
                {
                    m_pathStr = m_pathStr +
                                m_dlmtPath[0] +
                                m_nodes[i].ToString( m_dlmtID );
                }
                else
                {
                    m_pathStr = m_nodes[0].ToString( m_dlmtID );
                }
            }
        }


        /// <summary>
        /// Utility method for parsing the path string into path nodes.
        /// The given path string is treated as a pure path which does
        /// not contain any ID.
        /// </summary>
        /// <param name="path">The path string.</param>
        /// <param name="delimitersForPathNodes">The delimiters to split path nodes.</param>
        /// <param name="outputPathNodes">The path nodes parsed from the path.</param>
        /// <returns>True on success.</returns>
        protected static bool ParsePathStrWithoutID( string path,
                                                     char[] delimitersForPathNodes,
                                                     out List<PathNode> outputPathNodes )
        {
            // Trivial check
            if ( path.Length<=0 )
            {
                outputPathNodes = new List<PathNode>();
                return false;
            }

            // Split the path string
            string[] tokens =
                path.Split( delimitersForPathNodes,
                            StringSplitOptions.RemoveEmptyEntries );
            if ( tokens.Length<=0 )
            {
                outputPathNodes = new List<PathNode>();
                return false;
            }

            // Create the list for the path nodes
            outputPathNodes = new List<PathNode>(tokens.Length);

            // Parse the path nodes
            foreach ( string nodeName in tokens )
            {
                outputPathNodes.Add( new PathNode(nodeName) );
            }

            return true;
        }


        /// <summary>
        /// Utility method for parsing the path string into path nodes.
        /// The given path string is treated as a path contains node names
        /// and IDs. Although a node in the path does not necessarily have
        /// an ID, if it has, first the name, the delimiter for IDs, then
        /// the ID in HEX format.
        /// Eq. PropertyWindow@A8/EmitterProperty@FF/Life@13
        /// </summary>
        /// <param name="path"></param>
        /// <param name="delimitersForPathNodes">Delimiter to split the path nodes.</param>
        /// <param name="delimiterForID">Delimiter to split name and ID.</param>
        /// <param name="bIgnoreID">True to ignore the ID.</param>
        /// <param name="outputPathNodes">The path nodes parsed from the path.</param>
        /// <returns>True on success.</returns>
        protected static bool ParsePathStrWithID( string path,
                                                  char[] delimitersForPathNodes,
                                                  char delimiterForID,
                                                  bool bIgnoreID,
                                                  out List<PathNode> outputPathNodes )
        {
            // Trivial check
            if ( path.Length<=0 )
            {
                outputPathNodes = new List<PathNode>();
                return false;
            }

            // Split the path string
            string[] tokens =
                path.Split( delimitersForPathNodes,
                            StringSplitOptions.RemoveEmptyEntries );
            if ( tokens.Length<=0 )
            {
                outputPathNodes = new List<PathNode>();
                return false;
            }

            // Create the list for the path nodes
            outputPathNodes = new List<PathNode>(tokens.Length);

            // Parse the path nodes
            foreach ( string nodeName in tokens )
            {
                outputPathNodes.Add( new PathNode(nodeName, delimiterForID, bIgnoreID) );
            }

            return true;
        }


        /// <summary>
        /// Clone the path object.
        /// </summary>
        /// <returns>The clone of the original path object.</returns>
        public Path Clone()
        {
            return new Path( this );
        }


        /// <summary>
        /// Copy data from the given source path object.
        /// </summary>
        /// <param name="src">The source path object.</param>
        public void Copy( Path src )
        {
            int i;

            // Copy the delimiter for name/ID pairs
            this.m_dlmtID = src.m_dlmtID;

            // Copy the delimiters for the path node
            this.m_dlmtPath = new char[src.m_dlmtPath.Length];
            for ( i=0;i<src.m_dlmtPath.Length;++i )
                this.m_dlmtPath[i] = src.m_dlmtPath[i];

            // Copy the path string
            this.m_pathStr = string.Copy( src.m_pathStr );

            // Reserve memory for the path nodes first
            this.m_nodes.Clear();
            this.m_nodes.Capacity = src.m_nodes.Count;

            // Clone the path nodes
            for ( i=0;i<src.m_nodes.Count;++i )
            {
                this.m_nodes.Add( src.m_nodes[i].Clone() );
            }
        }


        /// <summary>
        /// Copy data from the given source path object.
        /// </summary>
        /// <param name="src">The source path object.</param>
        /// <param name="iNumMaxPathNodes">
        /// Specify the maximum number of path nodes to copy from the source.
        /// </param>
        public void Copy( Path src,
                          int iNumMaxPathNodes )
        {
            int i;

            // Copy the delimiter for name/ID pairs
            this.m_dlmtID = src.m_dlmtID;

            // Copy the delimiters for the path node
            this.m_dlmtPath = new char[src.m_dlmtPath.Length];
            for ( i=0;i<src.m_dlmtPath.Length;++i )
                this.m_dlmtPath[i] = src.m_dlmtPath[i];

            // How many path nodes to copy?
            int iNumPathNodesToCopy = src.m_nodes.Count;
            if ( iNumPathNodesToCopy>iNumMaxPathNodes )
                iNumPathNodesToCopy = iNumMaxPathNodes;

            // Reserve memory for the path nodes first
            this.m_nodes.Clear();
            this.m_nodes.Capacity = iNumPathNodesToCopy;

            // Clone the path nodes
            for ( i=0;i<iNumPathNodesToCopy;++i )
            {
                this.m_nodes.Add( src.m_nodes[i].Clone() );
            }

            // Update the path string
            this.UpdatePathString();
        }


        /// <summary>
        /// Extract a sub path from the given start index and node count.
        /// </summary>
        /// <param name="iStartIndex">The start index.</param>
        /// <param name="iNodeCount">The node count.</param>
        /// <returns>The extracted sub path.</returns>
        public Path SubPath( int iStartIndex,
                             int iNodeCount = -1 )
        {
            int i;

            Path dst = new Path();

            // Copy the delimiter for name/ID pairs
            dst.m_dlmtID = this.m_dlmtID;

            // Copy the delimiters for the path node
            dst.m_dlmtPath = new char[this.m_dlmtPath.Length];
            for ( i=0;i<this.m_dlmtPath.Length;++i )
                dst.m_dlmtPath[i] = this.m_dlmtPath[i];

            iStartIndex = Math.Min( this.m_nodes.Count-1, Math.Max( iStartIndex, 0 ) );

            // How many path nodes to copy?
            int iNumPathNodesToCopy = iNodeCount;
            if ( iNumPathNodesToCopy==0 )
            {
                return dst;
            }
            else if ( iNumPathNodesToCopy<0 ||
                      iStartIndex+iNumPathNodesToCopy>this.m_nodes.Count )
            {
                iNumPathNodesToCopy = this.m_nodes.Count - iStartIndex;
            }

            // Reserve memory for the path nodes first
            dst.m_nodes.Clear();
            dst.m_nodes.Capacity = iNumPathNodesToCopy;

            // Clone the path nodes
            for ( i=0;i<iNumPathNodesToCopy;++i )
            {
                dst.m_nodes.Add( this.m_nodes[iStartIndex+i].Clone() );
            }

            // Update the path string
            dst.UpdatePathString();

            return dst;
        }


        /// <summary>
        /// Extract a sub path from the given start node and node count.
        /// </summary>
        /// <param name="startNode">The node to start with the sub path.</param>
        /// <param name="iNodeCount">The node count.</param>
        /// <returns>The extracted sub path.</returns>
        public Path SubPath( PathNode startNode,
                             int iNodeCount = -1 )
        {
            int i;

            // Find the start node.
            int  iStartIndex = -1;
            for ( i=0;i<this.m_nodes.Count;++i )
            {
                if ( this.m_nodes[i]==startNode )
                {
                    iStartIndex = i;
                    break;
                }
            }

            if ( iStartIndex<0 || iStartIndex>=this.m_nodes.Count )
            {
                System.Diagnostics.Debug.Assert( false, "The specified start node does not exist in this path." );
                return null;
            }

            Path dst = new Path();

            // Copy the delimiter for name/ID pairs
            dst.m_dlmtID = this.m_dlmtID;

            // Copy the delimiters for the path node
            dst.m_dlmtPath = new char[this.m_dlmtPath.Length];
            for ( i=0;i<this.m_dlmtPath.Length;++i )
                dst.m_dlmtPath[i] = this.m_dlmtPath[i];

            // How many path nodes to copy?
            int iNumPathNodesToCopy = iNodeCount;
            if ( iNumPathNodesToCopy==0 )
            {
                return dst;
            }
            else if ( iNumPathNodesToCopy<0 ||
                      iStartIndex+iNumPathNodesToCopy>this.m_nodes.Count )
            {
                iNumPathNodesToCopy = this.m_nodes.Count - iStartIndex;
            }

            // Reserve memory for the path nodes first
            dst.m_nodes.Clear();
            dst.m_nodes.Capacity = iNumPathNodesToCopy;

            // Clone the path nodes
            for ( i=0;i<iNumPathNodesToCopy;++i )
            {
                dst.m_nodes.Add( this.m_nodes[iStartIndex+i].Clone() );
            }

            // Update the path string
            dst.UpdatePathString();

            return dst;
        }


        /// <summary>
        /// Clear the path.
        /// </summary>
        public void Clear()
        {
            this.m_nodes.Clear();
            this.UpdatePathString();
        }

        #endregion

        #region Member variables

        private char[] m_dlmtPath = new char[] { '.' };
        private char   m_dlmtID   = '@';

        private string         m_pathStr = string.Empty;
        private List<PathNode> m_nodes   = new List<PathNode>();

        #endregion
    }

    #endregion

    #region Class for tree nodes

    /// <summary>
    /// Class for path tree nodes.
    /// </summary>
    public class PathTreeNode
    {
        #region Constructor

        private static int s_iIDCounter = 0;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="srcPathNode">Source path node for this tree node.</param>
        /// <param name="parentPath">The full path of the parent.</param>
        /// <param name="data">Data for this node.</param>
        public PathTreeNode( PathNode srcPathNode,
                             Path parentPath,
                             object data )
        {
            this.ID         = PathTreeNode.s_iIDCounter++;

            this.m_fullPath = parentPath.Clone();
            this.m_pathNode = srcPathNode.Clone();
            this.m_data     = data;

            this.m_fullPath.Append( srcPathNode );
        }


        /// <summary>
        /// Destructor.
        /// </summary>
        ~PathTreeNode()
        {
            //DebugConsole.WriteLine( "Destroying path tree node : {0}, {1}",
            //                        this.ID,
            //                        this.m_fullPath.PathName );
        }

        #endregion

        #region Properties

        /// <summary>
        /// Get the ID of the node.
        /// </summary>
        public int ID { get; private set; }


        /// <summary>
        /// Get the parent of the node.
        /// </summary>
        public PathTreeNode Parent
        {
            get { return m_parent; }
        }


        /// <summary>
        /// Get the next sibling.
        /// </summary>
        public PathTreeNode NextSibling
        {
            get { return m_nextSibling; }
        }


        /// <summary>
        /// Get the previous sibling.
        /// </summary>
        public PathTreeNode PrevSibling
        {
            get { return m_prevSibling; }
        }


        /// <summary>
        /// Get the first child.
        /// </summary>
        public PathTreeNode FirstChild
        {
            get { return m_firstChild; }
        }


        /// <summary>
        /// Get the last child.
        /// </summary>
        public PathTreeNode LastChild
        {
            get
            {
                // Get the first child
                PathTreeNode child = FirstChild;
                if ( child==null )
                    return null;

                // Loop through the children to get their last sibling.
                while ( child.NextSibling!=null )
                {
                    child = child.NextSibling;
                }

                return child;
            }
        }


        /// <summary>
        /// Get the path node of this tree node
        /// </summary>
        public PathNode PathNode
        {
            get { return m_pathNode; }
        }


        /// <summary>
        /// Get the full path of this tree node
        /// </summary>
        public Path FullPath
        {
            get { return m_fullPath; }
        }


        /// <summary>
        /// Get or set the data of the tree node.
        /// </summary>
        public object Data
        {
            get { return m_data; }
            set { m_data = value; }
        }

        #endregion

        #region Add/remove tree node

        /// <summary>
        /// Add a child to the back of the last child.
        /// </summary>
        /// <param name="srcPathNode">Source path node for this tree node.</param>
        /// <param name="data">The data for the added child.</param>
        /// <param name="bIgnoreID">True to ignore ID.</param>
        /// <returns>The added child node, null is returned on fail.</returns>
        public PathTreeNode AddChild( PathNode srcPathNode,
                                      object data,
                                      bool bIgnoreID )
        {
            PathTreeNode child = null;

            // Has the node has any child yet?
            if ( this.FirstChild==null )
            {
                // Create the child
                child = new PathTreeNode( srcPathNode,
                                          this.m_fullPath,
                                          data );
                if ( child==null )
                    return null;

                if ( bIgnoreID==true )
                    child.PathNode.ID = 0;

                // Setup the connections
                this.m_firstChild = child;
                child.m_parent    = this;
            }
            else
            {
                // Create the child
                child = new PathTreeNode( srcPathNode,
                                          this.m_fullPath,
                                          data );
                if ( child==null )
                    return null;

                if ( bIgnoreID==true )
                    child.PathNode.ID = 0;

                // Find the last child
                PathTreeNode lastChild = this.LastChild;

                // Setup the connections
                lastChild.m_nextSibling = child;
                child.m_prevSibling     = lastChild;
                child.m_parent          = this;
            }

            return child;
        }

        /// <summary>
        /// Add a child to the back of the last child.
        /// </summary>
        /// <param name="childNode">The child tree node to add.</param>
        /// <returns>The added child node, null is returned on fail.</returns>
        public PathTreeNode AddChild( PathTreeNode childNode )
        {
            if ( childNode==null )
                return null;

            // Has the node has any child yet?
            if ( this.FirstChild==null )
            {
                // Setup the connections
                this.m_firstChild  = childNode;
                childNode.m_parent = this;
            }
            else
            {
                // Find the last child
                PathTreeNode lastChild = this.LastChild;

                // Setup the connections
                lastChild.m_nextSibling = childNode;
                childNode.m_prevSibling = lastChild;
                childNode.m_parent      = this;
            }

            return childNode;
        }


        /// <summary>
        /// Remove this tree node from the tree.
        /// </summary>
        /// <returns>This removed tree node.</returns>
        public PathTreeNode Remove()
        {
            // Is this the first child of the parent?
            if ( this.m_prevSibling==null &&
                 this.m_parent!=null )
            {
                this.m_parent.m_firstChild = this.m_nextSibling;
            }

            // Connect the next sibling to the previous.
            if ( this.m_prevSibling!=null )
            {
                this.m_prevSibling.m_nextSibling = this.m_nextSibling;
            }

            // Connect the previous sibling to the next.
            if ( this.m_nextSibling!=null )
            {
                this.m_nextSibling.m_prevSibling = this.m_prevSibling;
            }

            // Clean up the connections
            this.m_parent      = null;
            this.m_prevSibling = null;
            this.m_nextSibling = null;

            return this;
        }

        #endregion

        #region Find methods

        /// <summary>
        /// Find the first child matching the given condition.
        /// Pass in 0 as the ID to ignore it.
        /// </summary>
        /// <param name="iNameCRC">The name of the path node to find.</param>
        /// <param name="iID">The ID of the path node to find.</param>
        /// <returns>
        /// The first matching child, null is returned if no match
        /// has been found.
        /// </returns>
        public PathTreeNode FindFirstChild( uint iNameCRC,
                                            uint iID )
        {
            // Start from the first child
            PathTreeNode child = this.FirstChild;
            if ( child==null )
                return null;

            // Loop through all the children to find the match
            while ( child!=null )
            {
                if ( child.PathNode.Match( iNameCRC, iID )==true )
                    return child;

                child = child.NextSibling;
            }

            return null;
        }

        /// <summary>
        /// Find the first child matching the given condition.
        /// Pass in 0 as the ID to ignore it.
        /// </summary>
        /// <param name="name">The name of the path node to find.</param>
        /// <param name="iID">The ID of the path node to find.</param>
        /// <returns>
        /// The first matching child, null is returned if no match
        /// has been found.
        /// </returns>
        public PathTreeNode FindFirstChild( string name,
                                            uint iID )
        {
            // Make CRC hash value of the string
            uint nameCRC = NWKernel.CRC32Helper.ComputeCRC32Str(name);

            // Start from the first child
            PathTreeNode child = this.FirstChild;
            if ( child==null )
                return null;

            // Loop through all the children to find the match
            while ( child!=null )
            {
                if ( child.PathNode.Match( nameCRC, iID )==true )
                    return child;

                child = child.NextSibling;
            }

            return null;
        }


        /// <summary>
        /// Find the first sibling matching the given condition.
        /// Pass in 0 as the ID to ignore it.
        /// </summary>
        /// <param name="name">The name of the path node to find.</param>
        /// <param name="iID">The ID of the path node to find.</param>
        /// <returns>
        /// The first matching sibling, null is returned if no match
        /// has been found.
        /// </returns>
        public PathTreeNode FindNextSibling( string name,
                                             uint iID )
        {
            // Make CRC hash value of the string
            uint nameCRC = NWKernel.CRC32Helper.ComputeCRC32Str(name);

            // Start from the first child
            PathTreeNode sibling = this.NextSibling;
            if ( sibling==null )
                return null;

            // Loop through all the children to find the match
            while ( sibling!=null )
            {
                if ( sibling.PathNode.Match( nameCRC, iID )==true )
                    return sibling;

                sibling = sibling.NextSibling;
            }

            return null;
        }

        /// <summary>
        /// Find the first child matching the given condition.
        /// Pass in 0 as the ID to ignore it.
        /// </summary>
        /// <param name="expression">
        /// The regular expression for matching the path node name.
        /// </param>
        /// <param name="iID">The ID of the path node to find.</param>
        /// <returns>
        /// The first matching child, null is returned if no match
        /// has been found.
        /// </returns>
        public PathTreeNode FindFirstChild( Regex expression,
                                            uint iID )
        {
            // Start from the first child
            PathTreeNode child = this.FirstChild;
            if ( child==null )
                return null;

            // Loop through all the children to find the match
            while ( child!=null )
            {
                if ( child.PathNode.MatchRegex( expression, iID )==true )
                    return child;

                child = child.NextSibling;
            }

            return null;
        }


        /// <summary>
        /// Find the first sibling matching the given condition.
        /// Pass in 0 as the ID to ignore it.
        /// </summary>
        /// <param name="expression">
        /// The regular expression for matching the path node name.
        /// </param>
        /// <param name="iID">The ID of the path node to find.</param>
        /// <returns>
        /// The first matching sibling, null is returned if no match
        /// has been found.
        /// </returns>
        public PathTreeNode FindNextSibling( Regex expression,
                                             uint iID )
        {
            // Start from the first child
            PathTreeNode sibling = this.NextSibling;
            if ( sibling==null )
                return null;

            // Loop through all the children to find the match
            while ( sibling!=null )
            {
                if ( sibling.PathNode.MatchRegex( expression, iID )==true )
                    return sibling;

                sibling = sibling.NextSibling;
            }

            return null;
        }

        #endregion

        #region Member variables

        private PathTreeNode m_parent      = null;
        private PathTreeNode m_nextSibling = null;
        private PathTreeNode m_prevSibling = null;
        private PathTreeNode m_firstChild  = null;

        private PathNode m_pathNode;
        private Path     m_fullPath = null;
        private object   m_data;

        #endregion
    }

    #endregion

    #region Class for representing a directory structure

    /// <summary>
    /// Tree class for representing the directory structure.
    /// </summary>
    public class PathTree
    {
        #region Constructor

        /// <summary>
        /// Constructor.
        /// </summary>
        public PathTree( object dataForRoot )
        {
            PathNode rootPath = new PathNode( string.Empty, 0 );
            m_root = new PathTreeNode( rootPath,
                                       new Path(),
                                       dataForRoot );
        }

        #endregion

        #region Properties

        /// <summary>
        /// Get the root tree node.
        /// </summary>
        public PathTreeNode Root
        {
            get { return m_root; }
        }

        #endregion

        #region Find path data

        /// <summary>
        /// Find the first found path data with the given path string.
        /// </summary>
        /// <param name="pathStr">The path string.</param>
        /// <param name="bIgnoreID">True to ignore ID.</param>
        /// <returns>The path data, null is returned if the path is not found.</returns>
        public PathTreeNode FindPathData( string pathStr,
                                          bool bIgnoreID )
        {
            Path path = new Path( pathStr );

            return FindPathData( path, bIgnoreID );
        }


        /// <summary>
        /// Find the first found path data with the given path object.
        /// </summary>
        /// <param name="path">The path object.</param>
        /// <param name="bIgnoreID">True to ignore ID.</param>
        /// <param name="iMaxDepth">The maximum depth to search on the tree.</param>
        /// <returns>The path data, null is returned if the path is not found.</returns>
        public PathTreeNode FindPathData( Path path,
                                          bool bIgnoreID,
                                          int iMaxDepth = -1 )
        {
            PathTreeNode currNode = m_root;

            int  i;
            uint iID = 0;
            for ( i=0;i<path.Length;++i )
            {
                if ( iMaxDepth>=0 &&
                     i>iMaxDepth )
                {
                    break;
                }

                PathNode pathNode = path[i];
                if ( pathNode==null )
                    continue;

                if ( bIgnoreID==false )
                    iID = pathNode.ID;

                currNode = currNode.FindFirstChild(pathNode.Name, iID);
                if ( currNode==null )
                    return null;
            }

            return currNode;
        }

        #endregion

        #region Add/remove paths

        /// <summary>
        /// Add a path to the tree.
        /// The paths are identical, so if the path is already in the tree,
        /// the call will fail and null is returned.
        /// </summary>
        /// <param name="pathStr">The path to add.</param>
        /// <param name="data">The data to add.</param>
        /// <param name="bIgnoreID">True to ignore ID.</param>
        /// <returns>The added tree node on success.</returns>
        public PathTreeNode AddPath( string pathStr,
                                     object data,
                                     bool bIgnoreID )
        {
            Path path = new Path( pathStr );

            return AddPath( path, data, bIgnoreID );
        }


        /// <summary>
        /// Add a path to the tree.
        /// The paths are identical, so if the path is already in the tree,
        /// the call will fail and null is returned.
        /// </summary>
        /// <param name="path">The path to add.</param>
        /// <param name="data">The data to add.</param>
        /// <param name="bIgnoreID">True to ignore ID.</param>
        /// <returns>The added tree node on success.</returns>
        public PathTreeNode AddPath( Path path,
                                     object data,
                                     bool bIgnoreID )
        {
            PathTreeNode currNode = m_root;

            int  i;
            uint iID = 0;
            // Find the first non-matching child to start adding nodes
            for ( i=0;i<path.Length;++i )
            {
                PathNode pathNode = path[i];
                if ( pathNode==null )
                    continue;

                if ( bIgnoreID==false )
                    iID = pathNode.ID;

                // Find the first matching child
                PathTreeNode child =
                    currNode.FindFirstChild(pathNode.NameCRC, iID);

                // No match is found, that means we should start adding it!
                if ( child==null )
                {
                    // Start adding the path nodes
                    int j;
                    for ( j=i;j<path.Length;++j )
                    {
                        pathNode = path[j];
                        if ( pathNode==null )
                            continue;

                        // Add the path nodes
                        currNode = currNode.AddChild( pathNode,
                                                      null,
                                                      bIgnoreID );
                    }

                    // Set the data to the leaf path node
                    currNode.Data = data;

                    return currNode;
                }

                // Advance to the child
                currNode = child;
            }

            return null;
        }


        /// <summary>
        /// Remove the given path from the directory.
        /// </summary>
        /// <param name="pathStr">The path to remove.</param>
        /// <param name="bIgnoreID">Ignore ID while trying to find the path.</param>
        public void RemovePath( string pathStr,
                                bool bIgnoreID )
        {
            Path path = new Path( pathStr );

            RemovePath( path, bIgnoreID );
        }


        /// <summary>
        /// Remove the given path from the directory.
        /// </summary>
        /// <param name="path">The path to remove.</param>
        /// <param name="bIgnoreID">Ignore ID while trying to find the path.</param>
        public void RemovePath( Path path,
                                bool bIgnoreID )
        {
            PathTreeNode rootNode = FindPathData( path, bIgnoreID );
            if ( rootNode==null )
                return;

            PathTreeNode nextNode = null;
            PathTreeNode currNode = Traverse( rootNode, null );
            while ( currNode!=null )
            {
                if ( currNode==this.Root )
                {
                    // Do not remove the root node.
                    return;
                }
                else
                {
                    nextNode = Traverse( rootNode, currNode );
                    currNode.Remove();
                }

                currNode = nextNode;
            }
        }


        /// <summary>
        /// Remove the given tree node and its children from the directory.
        /// </summary>
        /// <param name="node">The path tree node to remove.</param>
        public void RemovePath( PathTreeNode node )
        {
            if ( node==null )
                return;

            PathTreeNode nextNode = null;
            PathTreeNode currNode = Traverse( node, null );
            while ( currNode!=null )
            {
                if ( currNode==this.Root )
                {
                    // Do not remove the root node.
                    return;
                }
                else
                {
                    nextNode = Traverse( node, currNode );
                    currNode.Remove();
                }

                currNode = nextNode;
            }
        }


        /// <summary>
        /// Attach the given tree node and its children to the directory.
        /// </summary>
        /// <param name="node">The path tree node to attach.</param>
        /// <param name="bIgnoreID">Ignore ID while trying to find the path.</param>
        public void AttachSubTree( PathTreeNode node,
                                   bool bIgnoreID )
        {
            if ( node==null )
                return;

            Path path = node.FullPath;
            if ( path==null )
                return;

            PathTreeNode currNode = m_root;

            int  i;
            uint iID = 0;
            // Find the first non-matching child to start adding nodes
            for ( i=0;i<path.Length;++i )
            {
                PathNode pathNode = path[i];
                if ( pathNode==null )
                    continue;

                if ( bIgnoreID==false )
                    iID = pathNode.ID;

                // Find the first matching child
                PathTreeNode child =
                    currNode.FindFirstChild(pathNode.NameCRC, iID);

                // No match is found, that means we should start adding it!
                if ( child==null )
                {
                    // Start adding the path nodes
                    int j;
                    for ( j=i;j<path.Length-1;++j )
                    {
                        pathNode = path[j];
                        if ( pathNode==null )
                            continue;

                        // Add the path nodes
                        currNode = currNode.AddChild( pathNode,
                                                      null,
                                                      bIgnoreID );
                    }

                    // Add the given node as the last child
                    currNode.AddChild( node );

                    return;
                }

                // Advance to the child
                currNode = child;
            }
        }


        /// <summary>
        /// Detach the given tree node and its children from the directory.
        /// </summary>
        /// <param name="path">The path of the tree node to detach.</param>
        /// <param name="bIgnoreID">Ignore ID while trying to find the path.</param>
        public PathTreeNode DitachSubTree( Path path,
                                           bool bIgnoreID )
        {
            if ( path==null )
                return null;

            PathTreeNode node = FindPathData( path, bIgnoreID );
            if ( node==null )
                return null;

            node.Remove();

            return node;
        }


        /// <summary>
        /// Detach the given tree node and its children from the directory.
        /// </summary>
        /// <param name="node">The path tree node to remove.</param>
        public PathTreeNode DitachSubTree( PathTreeNode node )
        {
            if ( node==null )
                return null;

            node.Remove();

            return node;
        }


        /// <summary>
        /// Clean up the empty branches.
        /// </summary>
        public void CleanUpEmptyBranches()
        {
            PathTreeNode nextNode = null;
            PathTreeNode currNode = Traverse( this.Root, null );
            while ( currNode!=null )
            {
                nextNode = Traverse( this.Root, currNode );

                // Remove this node if this is a leaf and it does not have any data
                if ( currNode.FirstChild==null &&
                     currNode.Data==null &&
                     currNode!=this.Root )
                {
                    currNode.Remove();
                }

                currNode = nextNode;
            }
        }


        /// <summary>
        /// Traverse through all the children of the given root node.
        /// When this method is called first time, leave the second
        /// parameter null, this will initialize the traversal.
        /// Afterwards, call this method with the previously returned
        /// tree node.
        /// The first leaf child of the root will be returned first,
        /// then the leaf child of the next sibling if any, then the
        /// parent of the child.
        /// The root it self will be the last one visited.
        /// </summary>
        /// <param name="root">The root to traverse from.</param>
        /// <param name="lastTraversedNode">The last traversed child node.</param>
        /// <returns>
        /// The traversed child, null is returned if all the children are visited.
        /// </returns>
        public PathTreeNode Traverse( PathTreeNode root,
                                      PathTreeNode lastTraversedNode )
        {
            if ( root==lastTraversedNode )
                return null;

            PathTreeNode currNode = null;
            if ( lastTraversedNode==null )
            {
                currNode = root;

                // Internally, we start from the first leaf child of the root.
                while ( currNode.FirstChild!=null )
                {
                    currNode = currNode.FirstChild;
                }

                return currNode;
            }
            else
            {
                currNode = TraverseChildren( lastTraversedNode );
            }

            return currNode;
        }


        /// <summary>
        ///
        /// </summary>
        /// <param name="lastNode"></param>
        /// <returns></returns>
        private PathTreeNode TraverseChildren( PathTreeNode lastNode )
        {
            PathTreeNode currNode = null;

            // Find the next sibling
            if ( lastNode.NextSibling!=null )
            {
                currNode = lastNode.NextSibling;

                // Get to the first leaf child of the sibling
                if ( currNode.FirstChild!=null )
                {
                    currNode = currNode.FirstChild;
                    while ( currNode.FirstChild!=null )
                    {
                        currNode = currNode.FirstChild;
                    }
                }
            }

            // Find the parent
            else if ( lastNode.Parent!=null )
            {
                currNode = lastNode.Parent;
            }

            return currNode;
        }

        #endregion

        #region Member variables

        private PathTreeNode m_root = null;

        #endregion
    }

    #endregion
}
