﻿// ========================================================================
// <copyright file="Protocol4CAccessor.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.IO;
using System.Text;

namespace NWCore.Protocols
{
    /// <summary>
    /// 便利関数
    /// </summary>
    public static class ProtocolUtility
    {
        /// <summary>
        /// バイト並びを逆にします。（引数のワークを破壊します）
        /// </summary>
        public static byte[] ReverseBytes(byte[] bytes)
        {
            for (int i = 0; i < bytes.Length / 2; ++i)
            {
                byte dataT = bytes[i];
                byte dataB = bytes[bytes.Length - 1 - i];
                bytes[i] = dataB;
                bytes[bytes.Length - 1 - i] = dataT;
            }
            return bytes;
        }
    }

    /// <summary>
    /// Protocol4FReader
    /// short,ushort,int,uint,floatのエンディアンを変換し、読み込みます
    /// </summary>
    public abstract class Protocol4FReader : BinaryReader
    {
        protected Encoding m_encoding;

//         /// <summary>コンストラクタ</summary>
//         public Protocol4FReader(Stream input)
//             : base(input)
//         {
//         }
//
        /// <summary>コンストラクタ</summary>
        public Protocol4FReader(Stream input, Encoding encoding)
            : base(input, encoding)
        {
            m_encoding = encoding;
        }
    }

    /// <summary>
    /// Protocol4FWriter
    /// short,ushort,int,uint,floatのエンディアンを変換し、書き込みます
    /// </summary>
    public abstract class Protocol4FWriter : BinaryWriter
    {
        protected Encoding m_encoding;

//         /// <summary>コンストラクタ</summary>
//         public Protocol4FWriter()
//             : base()
//         {
//         }
//
//         /// <summary>コンストラクタ</summary>
//         public Protocol4FWriter(Stream input)
//             : base(input)
//         {
//         }
//
        /// <summary>コンストラクタ</summary>
        public Protocol4FWriter(Stream input, Encoding encoding)
            : base(input, encoding)
        {
            m_encoding = encoding;
        }

        /// <summary>
        /// 指定サイズで文字列を書き込みます
        /// 足りない場合は、０で埋めます。
        /// </summary>
        /// <param name="strData">文字列</param>
        /// <param name="szData">指定サイズ</param>
        public virtual void Write(string strData, int szData)
        {
            System.Diagnostics.Debug.Assert(szData > 0, "不正なサイズです。");

            // 文字列
            byte[] bytes = m_encoding.GetBytes(strData);
            byte[] buffer = new byte[szData];
            int endIndex = szData - 2;
            for (int i = 0; i < szData; ++i)
            {
                if (i < bytes.Length && i < endIndex)
                {
                    buffer[i] = bytes[i];
                }
                else if (i < szData)
                {
                    buffer[i] = (byte)0;
                }
            }
            this.Write(buffer, 0, szData);
        }
    }


    /// <summary>
    /// Protocol4FReader for Little endian
    /// </summary>
    public class LittleEndianProtocolReader : Protocol4FReader
    {
//         /// <summary>コンストラクタ</summary>
//         public LittleEndianProtocolReader(Stream input)
//             : base(input)
//         {
//         }
//
        /// <summary>コンストラクタ</summary>
        public LittleEndianProtocolReader(Stream input, Encoding encoding)
            : base(input, encoding)
        {
        }
    }

    /// <summary>
    /// Protocol4FWriter for Little endian
    /// </summary>
    public class LittleEndianProtocolWriter : Protocol4FWriter
    {
//         /// <summary>コンストラクタ</summary>
//         public LittleEndianProtocolWriter()
//             : base()
//         {
//         }
//
//         /// <summary>コンストラクタ</summary>
//         public LittleEndianProtocolWriter(Stream input)
//             : base(input)
//         {
//         }
//
        /// <summary>コンストラクタ</summary>
        public LittleEndianProtocolWriter(Stream input, Encoding encoding)
            : base(input, encoding)
        {
        }
    }

    /// <summary>
    /// Protocol4FReader for Big endian
    /// </summary>
    public class BigEndianProtocolReader : Protocol4FReader
    {
//         /// <summary>コンストラクタ</summary>
//         public BigEndianProtocolReader(Stream input)
//             : base(input)
//         {
//         }
//
        /// <summary>コンストラクタ</summary>
        public BigEndianProtocolReader(Stream input, Encoding encoding)
            : base(input, encoding)
        {
        }

        /// <summary>
        /// ReadDecimal は使用禁止です。
        /// </summary>
        public override decimal ReadDecimal()
        {
            // 使用禁止
            System.Diagnostics.Debug.Assert(false, "ERROR : ReadDecimal()は使用禁止です");
            return (decimal)0;
        }

        public override char  ReadChar()
        {
            return BitConverter.ToChar(
                ProtocolUtility.ReverseBytes(BitConverter.GetBytes(base.ReadChar())), 0);
        }

        public override double ReadDouble()
        {
            return BitConverter.ToDouble(
                ProtocolUtility.ReverseBytes(BitConverter.GetBytes(base.ReadDouble())), 0);
        }

        public override short ReadInt16()
        {
            return BitConverter.ToInt16(
                ProtocolUtility.ReverseBytes(BitConverter.GetBytes(base.ReadInt16())), 0);
        }

        public override int ReadInt32()
        {
            return BitConverter.ToInt32(
                ProtocolUtility.ReverseBytes(BitConverter.GetBytes(base.ReadInt32())), 0);
        }

        public override long ReadInt64()
        {
            return BitConverter.ToInt64(
                ProtocolUtility.ReverseBytes(BitConverter.GetBytes(base.ReadInt64())), 0);
        }

        public override float ReadSingle()
        {
            return BitConverter.ToSingle(
                ProtocolUtility.ReverseBytes(BitConverter.GetBytes(base.ReadSingle())), 0);
        }

        public override ushort ReadUInt16()
        {
            return BitConverter.ToUInt16(
                ProtocolUtility.ReverseBytes(BitConverter.GetBytes(base.ReadUInt16())), 0);
        }

        public override uint ReadUInt32()
        {
            return BitConverter.ToUInt32(
                ProtocolUtility.ReverseBytes(BitConverter.GetBytes(base.ReadUInt32())), 0);
        }

        public override ulong ReadUInt64()
        {
            return BitConverter.ToUInt64(
                ProtocolUtility.ReverseBytes(BitConverter.GetBytes(base.ReadUInt64())), 0);
        }
    }

    /// <summary>
    /// Protocol4FWriter for Big endian
    /// </summary>
    public class BigEndianProtocolWriter : Protocol4FWriter
    {
//         /// <summary>コンストラクタ</summary>
//         public BigEndianProtocolWriter()
//             : base()
//         {
//         }
//
//         /// <summary>コンストラクタ</summary>
//         public BigEndianProtocolWriter(Stream input)
//             : base(input)
//         {
//         }
//
        /// <summary>コンストラクタ</summary>
        public BigEndianProtocolWriter(Stream input, Encoding encoding)
            : base(input, encoding)
        {
        }

        /// <summary>
        /// Write(decimal value) は仕様禁止です
        /// </summary>
        /// <param name="value"></param>
        public override void Write(decimal value)
        {
            System.Diagnostics.Debug.Assert(false, "ERROR : Write(decimal value)は使用禁止です");
        }

        public override void Write(char value)
        {
            base.Write(ProtocolUtility.ReverseBytes(BitConverter.GetBytes(value)));
        }

        public override void Write(double value)
        {
            base.Write(ProtocolUtility.ReverseBytes(BitConverter.GetBytes(value)));
        }

        public override void Write(float value)
        {
            base.Write(ProtocolUtility.ReverseBytes(BitConverter.GetBytes(value)));
        }

        public override void Write(int value)
        {
            base.Write(ProtocolUtility.ReverseBytes(BitConverter.GetBytes(value)));
        }

        public override void Write(long value)
        {
            base.Write(ProtocolUtility.ReverseBytes(BitConverter.GetBytes(value)));
        }

        public override void Write(sbyte value)
        {
            base.Write(ProtocolUtility.ReverseBytes(BitConverter.GetBytes(value)));
        }

        public override void Write(short value)
        {
            base.Write(ProtocolUtility.ReverseBytes(BitConverter.GetBytes(value)));
        }

        public override void Write(uint value)
        {
            base.Write(ProtocolUtility.ReverseBytes(BitConverter.GetBytes(value)));
        }

        public override void Write(ulong value)
        {
            base.Write(ProtocolUtility.ReverseBytes(BitConverter.GetBytes(value)));
        }

        public override void Write(ushort value)
        {
            base.Write(ProtocolUtility.ReverseBytes(BitConverter.GetBytes(value)));
        }

    }
}
