﻿// ========================================================================
// <copyright file="ICommandManager.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.ComponentModel;

namespace NWCore.DataModel
{
    /// <summary>
    /// Event arguments sent before command is to be executed.
    /// </summary>
    public class BeforeCommandExecutionEventArgs : CancelEventArgs
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="command">The command to be executed.</param>
        /// <param name="executionType">The execution type.</param>
        public BeforeCommandExecutionEventArgs( ICommand command,
                                                CommandExecutionTypes executionType ) :
            this(false, command, executionType)
        {
        }


        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="bCancel">True to cancel the execution.</param>
        /// <param name="command">The command to be executed.</param>
        /// <param name="executionType">The execution type.</param>
        public BeforeCommandExecutionEventArgs( bool bCancel,
                                                ICommand command,
                                                CommandExecutionTypes executionType ) :
            base(bCancel)
        {
            this.Command       = command;
            this.ExecutionType = executionType;
        }


        /// <summary>Get the command to be executed.</summary>
        public ICommand Command { get; private set; }


        /// <summary>Get the execution type.</summary>
        public CommandExecutionTypes ExecutionType { get; private set; }
    }


    /// <summary>
    /// コマンドインターフェース
    /// </summary>
    public interface ICommand
    {
        /// <summary>
        /// Get or set the description of the command.
        /// </summary>
        string Description { get; set; }

        /// <summary>
        /// Get or set the data source description of the command.
        /// </summary>
        string DataSrcDesc { get; set; }

        /// <summary>
        /// Get or set the target document.
        /// </summary>
        IDocument TargetDocument { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether to keep the command on the stack.
        /// </summary>
        bool CanUndo { get; }

        /// <summary>
        /// Prepare command execution. This method is called before Execute(), Undo() and
        /// Redo().
        /// </summary>
        /// <param name="type">Type of the command execution.</param>
        /// <returns>True on success, otherwise the execution is aborted.</returns>
        bool PrepareExecution( CommandExecutionTypes type );

        /// <summary>
        /// Gets whether the command manager runs the command in the main UI thread context.
        /// </summary>
        bool RunOnUIThread { get; }

        /// <summary>
        /// Execute command. This method is called when the first time the command is
        /// executed.
        /// </summary>
        ICommand Execute();


        /// <summary>
        /// Undo command.
        /// </summary>
        ICommand Undo();


        /// <summary>
        /// Redo command.
        /// </summary>
        ICommand Redo();


        /// <summary>
        /// Finish the command execution. This method is called after Execute(), Undo()
        /// and Redo() to clean up the resources used for the execution.
        /// </summary>
        /// <param name="type">Type of the command execution.</param>
        void FinishExecution( CommandExecutionTypes type );

        /// <summary>
        /// Do proper process when the command is canceled.
        /// </summary>
        void ProcessCommandCanceled();
    }

    /// <summary>
    /// Post command update
    /// </summary>
    public interface IPostCommandUpdate
    {
        /// <summary>
        /// Update command
        /// </summary>
        String UpdateCommand();
    }

    /// <summary>
    /// コマンドセットインターフェース
    /// </summary>
    public interface ICommandSet : ICommand
    {
        /// <summary>
        /// 登録されているコマンド数
        /// </summary>
        int Count { get; }

        /// <summary>
        /// 追加
        /// </summary>
        void Add(ICommand command);

        /// <summary>
        /// コマンドリスト
        /// </summary>
        IList<ICommand> Commands { get; }
    }

    /// <summary>
    /// コマンドデリゲート
    /// </summary>
    public delegate void CommandDelegate( ICommand command,
                                          CommandExecutionTypes executionType );

    /// <summary>
    /// コマンドデリゲート
    /// </summary>
    public delegate void GlobalCommandHandler( ICommand command,
                                               CommandExecutionTypes executionType,
                                               IDocument targetDocument );

    /// <summary>
    /// Enumerator for the interested type of command events.
    /// </summary>
    public enum CmdEvtInterestTypes
    {
        EditedDataSource,
        EditedControl
    }

    /// <summary>
    /// Enumerator for the types of the command execution.
    /// </summary>
    public enum CommandExecutionTypes
    {
        Execute,
        Undo,
        Redo,
        ClearCommands
    }

    /// <summary>
    /// コマンドマネージャ用インターフェイスです。
    /// </summary>
    public interface ICommandManager : IDisposable
    {
        /// <summary>
        /// Get or set the flag whether to ignore new commands.
        /// </summary>
        bool IgnoreCommands { get; set; }

        bool BeginCommandSet( IDocument targetDocument,
                              ICommandSet commandSet );
        void EndCommandSet( bool bCanUndo = true );

        bool ExecuteIgnoreTargetDocument( ICommand command );
        bool Execute( string commandName,
                      object commandArguments,
                      string description = "",
                      string dataSrcPath = "",
                      IDocument targetDocument = null );
        bool Execute( ICommand command,
                      string description = "",
                      string dataSrcPath = "",
                      IDocument targetDocument = null );

        bool ScheduleExecIgnoreTargetDocument( ICommand command );
        bool ScheduleExec( string commandName,
                           object commandArguments,
                           string description = "",
                           string dataSrcPath = "",
                           IDocument targetDocument = null );
        bool ScheduleExec( ICommand command,
                           string description = "",
                           string dataSrcPath = "",
                           IDocument targetDocument = null );

        bool Redo();
        bool Undo();

        void NotifyValueChanging( string commandName,
                                  object commandArguments );

        bool CanRedo();
        bool CanUndo();

        void ClearActiveCommandStacks();
        void ClearAll();

        /// <summary>
        /// Add command listener for all the commands executed.
        /// </summary>
        /// <param name="handler">The event handler.</param>
        void AddCommandListener( GlobalCommandHandler handler );

        /// <summary>
        /// Add command listener for all the commands executed to the specified
        /// document.
        /// </summary>
        /// <param name="interestedDocument">The document.</param>
        /// <param name="handler">The event handler.</param>
        void AddCommandListener( IDocument interestedDocument,
                                 CommandDelegate handler );

        /// <summary>
        /// Add command listener for the interested type of command executions.
        /// </summary>
        /// <param name="interestType">
        /// Type of interest, data source or control.
        /// </param>
        /// <param name="path">Name of the interest.</param>
        /// <param name="handler">Event handler.</param>
        /// <returns>True on success.</returns>
        bool AddCommandListener( CmdEvtInterestTypes interestType,
                                 string path,
                                 CommandDelegate handler );

        /// <summary>
        /// Remove the command listener for all the commands executed.
        /// </summary>
        /// <param name="handler">The event handler.</param>
        void RemoveCommandListener( GlobalCommandHandler handler );

        /// <summary>
        /// Remove the command listener for all the commands executed to the specified
        /// document.
        /// </summary>
        /// <param name="interestedDocument">The document.</param>
        /// <param name="handler">The event handler.</param>
        void RemoveCommandListener( IDocument interestedDocument,
                                    CommandDelegate handler );

        /// <summary>
        /// Remove command listener.
        /// </summary>
        /// <param name="interestType">
        /// Type of interest, data source or control.
        /// </param>
        /// <param name="path">Name of the interest.</param>
        /// <param name="handler">Event handler.</param>
        void RemoveCommandListener( CmdEvtInterestTypes interestType,
                                    string path,
                                    CommandDelegate handler );

        /// <summary>
        /// Get command listeners registered to the data source.
        /// </summary>
        /// <param name="pathStr">The path of the data source.</param>
        /// <returns>A list of the command listeners.</returns>
        List<CommandDelegate> GetCommandDataSrcListeners( string pathStr );

        /// <summary>
        /// Get command listeners registered to the control.
        /// </summary>
        /// <param name="pathStr">The path of the control.</param>
        /// <returns>A list of the command listeners.</returns>
        List<CommandDelegate> GetCommandControlListeners( string pathStr );

        /// <summary>
        /// 保持できるコマンドの最大数を設定・取得します。
        /// </summary>
        int MaxUndoCount { get; set; }

        /// <summary>
        /// Get or set whether notification should be sent when commands being executed.
        /// </summary>
        bool EnableNotification { get; set; }

        /// <summary>
        /// Get the size of the active undo stack.
        /// </summary>
        int ActiveUndoStackSize { get; }

        /// <summary>
        /// Get the size of the active redo stack.
        /// </summary>
        int ActiveRedoStackSize { get; }

        /// <summary>
        /// Get the target document.
        /// </summary>
        IDocument TargetDocument { get; }

        /// <summary>
        /// Get command from undo stack.
        /// </summary>
        /// <param name="iIndex">Index of the command in the stack.</param>
        /// <returns>The command.</returns>
        ICommand GetCommandFromUndoStack( int iIndex );

        /// <summary>
        /// Get command from redo stack.
        /// </summary>
        /// <param name="iIndex">Index of the command in the stack.</param>
        /// <returns>The command.</returns>
        ICommand GetCommandFromRedoStack( int iIndex );

        /// <summary>
        /// Event being triggered before executing, undoing, redoing commands.
        /// </summary>
        event CancelEventHandler BeforeExecution;
    }
}
