﻿// ========================================================================
// <copyright file="PreviewModelData.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Xml.Serialization;
using NintendoWare.ToolDevelopmentKit;
using NWCore.DataModel.Major_1.Minor_6.Build_0.Revision_0;
using NWCore.Serializer.Major_1.Minor_6.Build_0.Revision_0;

namespace NWCore.DataModel.Major_1.Minor_6.Build_0.Revision_0
{
    /// <summary>
    /// プレビュー・モデル設定データ
    /// </summary>
    [Serializable]
    public sealed class PreviewModelData : BaseDataModel<PreviewModelData>
    {
        List<string> m_boneNameList = new List<string>();

        #region Constructors

        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        public PreviewModelData()
        {
            Initialize();
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        /// <param name="source">コピー元のベクトル</param>
        public PreviewModelData( PreviewModelData source ) :
            this()
        {
            this.Set( source );
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
            this.DisplayModel  = true;
            this.ModelName     = String.Empty;
            this.ModelFilePath = String.Empty;

            this.m_boneNameList.Clear();

            this.AnimationFolder = string.Empty;
            this.AnimationName   = string.Empty;

            this.AnimFileStartFrame = 0;
            this.AnimFileEndFrame   = -1;

            this.AnimPlaybackSpeed      = 1.0f;
            this.AnimPlaybackStartFrame = 0;

            this.UserFunctionList = new List<UserFunctionXML>();
        }

        /// <summary>
        /// ベクトルを設定します。
        /// </summary>
        /// <param name="source">設定するベクトルです。</param>
        public override void Set(PreviewModelData source)
        {
            Ensure.Argument.NotNull(source);

            this.DisplayModel  = source.DisplayModel;
            this.ModelName     = string.Copy(source.ModelName);
            this.ModelFilePath = string.Copy(source.ModelFilePath);

            SetBoneNameList( source.GetBoneNameList() );

            this.UserFunctionList.Clear();
            foreach (UserFunctionXML src in source.UserFunctionList)
            {
                UserFunctionXML dst = new UserFunctionXML(src.UserFuncName, src.UserFuncID);
                dst.Copy(src);
                this.UserFunctionList.Add(dst);
            }

            this.AnimationFolder = source.AnimationFolder;
            this.AnimationName   = source.AnimationName;

            this.AnimFileStartFrame = source.AnimFileStartFrame;
            this.AnimFileEndFrame   = source.AnimFileEndFrame;

            this.AnimPlaybackSpeed      = source.AnimPlaybackSpeed;
            this.AnimPlaybackStartFrame = source.AnimPlaybackStartFrame;
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public override PreviewModelData Clone()
        {
            return new PreviewModelData(this);
        }

        /// <summary>
        /// ボーン名を更新します。
        /// </summary>
        /// <param name="boneNames">Array of the bone names.</param>
        public void SetBoneNameList( string[] boneNames )
        {
            m_boneNameList.Clear();
            if ( boneNames!=null )
            {
                m_boneNameList.AddRange( boneNames );
            }
        }

        /// <summary>
        /// ボーン名リストを取得します。
        /// </summary>
        public string[] GetBoneNameList()
        {
            return m_boneNameList.ToArray();
        }

        /// <summary>
        /// ボーン数
        /// </summary>
        public int BoneCount
        {
            get { return m_boneNameList.Count; }
        }

        /// <summary>
        /// ボーン名を取得
        /// </summary>
        public string GetBoneName( int index )
        {
            return (m_boneNameList.Count > index ? m_boneNameList[index] : String.Empty);
        }

        /// <summary>
        /// ボーン名からインデックス取得。
        /// </summary>
        /// <returns>見つからないときは-1。</returns>
        public int GetBoneIndex( string boneName )
        {
            for ( int i=0;i<m_boneNameList.Count;++i )
            {
                if ( m_boneNameList[i]==boneName &&
                     m_boneNameList[i]==boneName )
                {
                    return i;
                }
            }
            return -1;
        }

        #endregion

        #region Properties

        /// <summary>
        /// 表示
        /// </summary>
        public bool DisplayModel { get; set; }

        /// <summary>
        /// モデル。
        /// </summary>
        public string ModelName { get; set; }

        /// <summary>
        /// モデルファイルパス。
        /// </summary>
        public string ModelFilePath { get; set; }

        /// <summary>
        /// キャラクタアニメフォルダ。
        /// </summary>
        public string AnimationFolder { get; set; }

        /// <summary>
        /// アニメーション名。
        /// </summary>
        public string AnimationName { get; set; }

        /// <summary>
        /// 再生範囲の開始フレーム。
        /// </summary>
        public int AnimFileStartFrame { get; set; }

        /// <summary>
        /// 再生範囲の終了フレーム。
        /// </summary>
        public int AnimFileEndFrame { get; set; }

        /// <summary>
        /// 再生速度。
        /// </summary>
        public float AnimPlaybackSpeed { get; set; }

        /// <summary>
        /// 再生開始フレーム。
        /// </summary>
        public int AnimPlaybackStartFrame { get; set; }

        /// <summary>
        /// The list of the user functions.
        /// </summary>
        [XmlArrayItem("UserFunction")]
        public List<UserFunctionXML> UserFunctionList { get; set; }

        #endregion
    }
}
