﻿// ========================================================================
// <copyright file="AnimTableData.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Xml.Serialization;
using System.Globalization;
using NintendoWare.ToolDevelopmentKit;
using NWCore.Utility;

namespace NWCore.DataModel
{
    /// <summary>
    /// Class for the data model for the animation table data.
    /// </summary>
    public class AnimTableData : BaseDataModel<AnimTableData>
    {
        #region Construtors

        /// <summary>
        /// Default constructor.
        /// </summary>
        public AnimTableData()
        {
            Initialize();
        }


        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="source">The source to copy from.</param>
        public AnimTableData( AnimTableData source ) :
            this()
        {
            this.Set(source);
        }


        /// <summary>
        /// Initialize the data.
        /// </summary>
        public override void Initialize()
        {
            this.TimeBaseType              = AnimTimeBaseTypes.Emitter;
            this.Name                      = AnimTableTargetTypes.unknown;
            this.IsEnabled                 = false;
            this.IsLoop                    = false;
            this.InterpolationType         = AnimInterpolationTypes.Linear;
            this.NumLoopFrames             = DEFAULT_NUM_LOOP_FRAMES;
            this.IsRandomStartFrameEnabled = false;

            if ( this.KeyFrames==null )
                this.KeyFrames = new List<AnimTableKeyFrameData>();

            this.KeyFrames.Clear();
        }

        #endregion

        #region Properties

        /// <summary>
        /// The type of the animation playback timing.
        /// </summary>
        public AnimTimeBaseTypes TimeBaseType { get; set; }


        /// <summary>
        /// The applied target type name of the animation table.
        /// </summary>
        public AnimTableTargetTypes Name { get; set; }


        /// <summary>
        /// Is the animation table enabled?
        /// </summary>
        public bool IsEnabled { get; set; }


        /// <summary>
        /// Is loop enabled for this animation.
        /// </summary>
        public bool IsLoop { get; set; }


        /// <summary>
        /// The type of the interpolation mode between key frames.
        /// </summary>
        public AnimInterpolationTypes InterpolationType { get; set; }


        /// <summary>
        /// Get or set the number of the loop frames.
        /// </summary>
        public int NumLoopFrames { get; set; }


        /// <summary>
        /// Get or set the flag indicating whether the animation start frame should be randomized.
        /// </summary>
        public bool IsRandomStartFrameEnabled { get; set; }


        /// <summary>
        /// The list of animation key frames.
        /// </summary>
        public List<AnimTableKeyFrameData> KeyFrames
        {
            get { return m_keyFrames; }
            set
            {
                if ( value==null )
                {
                    if ( m_keyFrames!=null )
                        m_keyFrames.Clear();
                    return;
                }

                if ( m_keyFrames==null )
                    m_keyFrames = new List<AnimTableKeyFrameData>( value.Count );

                m_keyFrames.Clear();
                foreach ( AnimTableKeyFrameData data in value )
                {
                    m_keyFrames.Add( data.Clone() );
                }

                SortKeyFrames();
            }
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// Set the data from the source.
        /// </summary>
        /// <param name="source">The source to set data from.</param>
        public override void Set( AnimTableData source )
        {
            Ensure.Argument.NotNull(source);

            this.TimeBaseType              = source.TimeBaseType;
            this.Name                      = source.Name;
            this.IsEnabled                 = source.IsEnabled;
            this.IsLoop                    = source.IsLoop;
            this.InterpolationType         = source.InterpolationType;
            this.NumLoopFrames             = source.NumLoopFrames;
            this.IsRandomStartFrameEnabled = source.IsRandomStartFrameEnabled;

            this.KeyFrames.Clear();
            this.KeyFrames.Capacity = source.KeyFrames.Count;
            foreach ( AnimTableKeyFrameData data in source.KeyFrames )
            {
                this.KeyFrames.Add( data.Clone() );
            }
        }


        /// <summary>
        /// Clone the data model.
        /// </summary>
        /// <returns>The new data model.</returns>
        public override AnimTableData Clone()
        {
            return new AnimTableData( this );
        }


        /// <summary>
        /// Compare the animation table with the given one.
        /// </summary>
        /// <param name="table">The animation table to compare to.</param>
        /// <returns>True on equal.</returns>
        public bool IsEqual( AnimTableData table )
        {
            if ( table==null )
                return false;

            if ( this.Name!=table.Name )
                return false;

            if ( this.TimeBaseType!=table.TimeBaseType )
                return false;

            if ( this.IsEnabled!=table.IsEnabled )
                return false;

            if ( this.IsLoop!=table.IsLoop )
                return false;

            if ( this.InterpolationType!=table.InterpolationType )
                return false;

            if ( this.NumLoopFrames!=table.NumLoopFrames )
                return false;

            if ( this.IsRandomStartFrameEnabled!=table.IsRandomStartFrameEnabled )
                return false;

            if ( this.KeyFrames.Count!=table.KeyFrames.Count )
                return false;

            int iNumKeyFrames = this.KeyFrames.Count;
            for ( int i=0;i<iNumKeyFrames;++i )
            {
                if ( this.KeyFrames[i].IsEqual(table.KeyFrames[i])==false )
                    return false;
            }

            return true;
        }


        /// <summary>
        /// Sort the key frames by the key frame number.
        /// </summary>
        public void SortKeyFrames()
        {
            if ( m_keyFrames==null ||
                 m_keyFrames.Count<=1 )
            {
                return;
            }

            QuickSortInternal( 0, m_keyFrames.Count-1 );
        }


        /// <summary>
        /// Internal key frame sorting method implementing the quick sort algorithm.
        /// </summary>
        /// <param name="iLeft">The left pivot index.</param>
        /// <param name="iRight">The right pivot index.</param>
        private void QuickSortInternal( int iLeft,
                                        int iRight )
        {
            int l     = iLeft;
            int r     = iRight;
            int pivot = m_keyFrames[(iLeft+iRight)/2].KeyFrame;

            do
            {
                while ( m_keyFrames[l].KeyFrame<pivot &&
                        l<iRight )
                {
                    ++l;
                }

                while ( m_keyFrames[r].KeyFrame>pivot &&
                        iLeft<r )
                {
                    --r;
                }

                if ( l<=r )
                {
                    AnimTableKeyFrameData tmp = m_keyFrames[l];
                    m_keyFrames[l] = m_keyFrames[r];
                    m_keyFrames[r] = tmp;

                    ++l;
                    --r;
                }
            } while ( l<=r );

            if ( iLeft<r )
                QuickSortInternal( iLeft, r );

            if ( l<iRight )
                QuickSortInternal( l, iRight );
        }

        #endregion

        #region Member variables

        public const int DEFAULT_NUM_LOOP_FRAMES = 100;

        private List<AnimTableKeyFrameData> m_keyFrames = null;

        #endregion
    }

    #region Base key frame data

    /// <summary>
    /// Class for storing the key frame data of the animation table.
    /// </summary>
    public abstract class AnimTableKeyFrameData : BaseDataModel<AnimTableKeyFrameData>
    {
        #region Initialize

        /// <summary>
        /// Initialize the data.
        /// </summary>
        public override void Initialize()
        {
            this.KeyFrame = 0;
        }

        #endregion

        #region Properties

        /// <summary>
        /// The frame number of the key frame data.
        /// </summary>
        public int KeyFrame { get; set; }


        /// <summary>
        /// The key frame value in string formats.
        /// </summary>
        public abstract string StrValue { get; set; }

        #endregion

        #region Utility Functions

        /// <summary>
        /// Set the data from the source.
        /// </summary>
        /// <param name="source">The source to set data from.</param>
        public override void Set( AnimTableKeyFrameData source )
        {
        }


        /// <summary>
        /// Clone the data model.
        /// </summary>
        /// <returns>The new data model.</returns>
        public override AnimTableKeyFrameData Clone()
        {
            return null;
        }


        /// <summary>
        /// Compare the data with the given key frame data.
        /// </summary>
        /// <param name="data">The key frame to compare to.</param>
        /// <returns>True on equal.</returns>
        public abstract bool IsEqual( AnimTableKeyFrameData data );

        #endregion
    }

    #endregion

    #region Integer value key frame data

    /// <summary>
    /// Class for storing the key frame data of the animation table.
    /// </summary>
    public class IntAnimTableKeyFrameData : AnimTableKeyFrameData
    {
        #region Construtors

        /// <summary>
        /// Default constructor.
        /// </summary>
        public IntAnimTableKeyFrameData()
        {
            Initialize();
        }

        /// <summary>
        /// Default constructor.
        /// </summary>
        public IntAnimTableKeyFrameData( int iDefaultValue )
        {
            m_iDefaultValue = iDefaultValue;
            Initialize();
        }


        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="source">The source to copy from.</param>
        public IntAnimTableKeyFrameData( IntAnimTableKeyFrameData source ) :
            this()
        {
            this.Set( source );
        }


        /// <summary>
        /// Initialize the data.
        /// </summary>
        public override void Initialize()
        {
            base.Initialize();

            this.Value = m_iDefaultValue;
        }

        #endregion

        #region Properties

        /// <summary>
        /// The animation value for the key frame.
        /// </summary>
        public int Value { get; set; }


        /// <summary>
        /// The key frame value in string formats.
        /// </summary>
        public override string StrValue
        {
            get { return this.Value.ToString( CultureInfo.InvariantCulture ); }
            set
            {
                int iValue;
                if ( int.TryParse( value,
                                   NumberStyles.Number,
                                   CultureInfo.InvariantCulture,
                                   out iValue )==true )
                    this.Value = iValue;
            }
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// Set the data from the source.
        /// </summary>
        /// <param name="source">The source to set data from.</param>
        public override void Set( AnimTableKeyFrameData source )
        {
            Ensure.Argument.NotNull( source );

            this.KeyFrame = source.KeyFrame;
            if ( source is IntAnimTableKeyFrameData )
                this.Value = (source as IntAnimTableKeyFrameData).Value;
        }


        /// <summary>
        /// Clone the data model.
        /// </summary>
        /// <returns>The new data model.</returns>
        public override AnimTableKeyFrameData Clone()
        {
            return new IntAnimTableKeyFrameData( this );
        }


        /// <summary>
        /// Compare the data with the given key frame data.
        /// </summary>
        /// <param name="data">The key frame to compare to.</param>
        /// <returns>True on equal.</returns>
        public override bool IsEqual( AnimTableKeyFrameData data )
        {
            IntAnimTableKeyFrameData castedData = data as IntAnimTableKeyFrameData;
            if ( castedData==null )
                return false;

            return ( this.Value==castedData.Value );
        }

        #endregion

        #region Member variables

        private int m_iDefaultValue = 0;

        #endregion
    }

    #endregion

    #region Float value key frame data

    /// <summary>
    /// Class for storing the key frame data of the animation table.
    /// </summary>
    public class FloatAnimTableKeyFrameData : AnimTableKeyFrameData
    {
        #region Construtors

        /// <summary>
        /// Default constructor.
        /// </summary>
        public FloatAnimTableKeyFrameData()
        {
            Initialize();
        }

        /// <summary>
        /// Default constructor.
        /// </summary>
        public FloatAnimTableKeyFrameData( float fDefaultValue )
        {
            m_fDefaultValue = fDefaultValue;
            Initialize();
        }


        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="source">The source to copy from.</param>
        public FloatAnimTableKeyFrameData( FloatAnimTableKeyFrameData source ) :
            this()
        {
            this.Set( source );
        }


        /// <summary>
        /// Initialize the data.
        /// </summary>
        public override void Initialize()
        {
            base.Initialize();

            this.Value = m_fDefaultValue;
        }

        #endregion

        #region Properties

        /// <summary>
        /// The animation value for the key frame.
        /// </summary>
        public float Value { get; set; }


        /// <summary>
        /// The key frame value in string formats.
        /// </summary>
        public override string StrValue
        {
            get
            {
                // By default, float.ToString() does not preserve the maximum precision
                // of a float, so we have to specify ToString() with that precision.
                return this.Value.ToString( "G9", CultureInfo.InvariantCulture );
            }
            set
            {
                float fValue;
                if ( float.TryParse( value,
                                     NumberStyles.Float,
                                     CultureInfo.InvariantCulture,
                                     out fValue )==true )
                    this.Value = fValue;
            }
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// Set the data from the source.
        /// </summary>
        /// <param name="source">The source to set data from.</param>
        public override void Set( AnimTableKeyFrameData source )
        {
            Ensure.Argument.NotNull( source );

            this.KeyFrame = source.KeyFrame;
            if ( source is FloatAnimTableKeyFrameData )
                this.Value = (source as FloatAnimTableKeyFrameData).Value;
        }


        /// <summary>
        /// Clone the data model.
        /// </summary>
        /// <returns>The new data model.</returns>
        public override AnimTableKeyFrameData Clone()
        {
            return new FloatAnimTableKeyFrameData( this );
        }


        /// <summary>
        /// Compare the data with the given key frame data.
        /// </summary>
        /// <param name="data">The key frame to compare to.</param>
        /// <returns>True on equal.</returns>
        public override bool IsEqual( AnimTableKeyFrameData data )
        {
            FloatAnimTableKeyFrameData castedData = data as FloatAnimTableKeyFrameData;
            if ( castedData==null )
                return false;

            return (this.Value==castedData.Value);
        }

        #endregion

        #region Member variables

        private float m_fDefaultValue = 0.0f;

        #endregion
    }

    #endregion
}
