﻿// ========================================================================
// <copyright file="Timer.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using System.Diagnostics;

namespace App.Utility
{
    /// <summary>
    /// タイマーの種類
    /// </summary>
    public enum TimerType
    {
        TIMER_TOTAL = 0,              /// トータル
        TIMER_ESET_LOAD,              /// エミッターセットの読み込み
        TIMER_TEXTURE_LOAD,           /// テクスチャの読み込み
        TIMER_DATAMODEL_DESERIALIZE,  ///
        TIMER_DATAMODEL_UPDATE,       ///
        TIMER_MAX
    }

    /// <summary>
    /// タイマー
    /// </summary>
    public class ConvertTimer
    {
        /// <summary>
        /// 静的コンストラクタ
        /// </summary>
        static ConvertTimer()
        {
            for ( int i = 0; i < (int)(TimerType.TIMER_MAX); ++i)
            {
                s_stopWatch[i] = new Stopwatch();
            }
        }

        /// <summary>
        /// 時間計測を開始します。
        /// </summary>
        public static void Start( TimerType type )
        {
            s_stopWatch[(int)type].Start();
        }

        /// <summary>
        /// 時間計測を停止します。
        /// </summary>
        public static void Stop( TimerType type )
        {
            s_stopWatch[(int)type].Stop();
        }

        /// <summary>
        /// 経過時間を取得します。
        /// </summary>
        public static float GetTime( TimerType type )
        {
            return (float)s_stopWatch[(int)type].ElapsedMilliseconds / 1000.0f;
        }

        /// <summary>
        /// 経過時間をログに表示します。
        /// </summary>
        public static void Output()
        {
            Console.Write("[EftConv] Total Time                       : {0:f2} (s)\n", GetTime( TimerType.TIMER_TOTAL ) );
            Console.Write("[EftConv]  - Emitter Set Load Time         : {0:f2} (s)\n", GetTime( TimerType.TIMER_ESET_LOAD ) );
            Console.Write("[EftConv]     - Texture Load Time          : {0:f2} (s)\n", GetTime( TimerType.TIMER_TEXTURE_LOAD ) );
            Console.Write("[EftConf]     - Datamodel Deserialize Time : {0:f2} (s)\n", GetTime( TimerType.TIMER_DATAMODEL_DESERIALIZE ) );
            Console.Write("[EftConf]     - Datamodel Update Time      : {0:f2} (s)\n", GetTime( TimerType.TIMER_DATAMODEL_UPDATE ) );
        }

        private static Stopwatch[]  s_stopWatch = new Stopwatch[(int)(TimerType.TIMER_MAX)];
    }
}
