﻿// ========================================================================
// <copyright file="ColorBlender.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Drawing;
using System.Drawing.Imaging;

namespace App.Utility
{
    /// <summary>
    /// カラーブレンダクラス。
    /// </summary>
    public sealed class ColorBlender
    {
        /// <summary>
        /// 半々の比率でブレンド。
        /// </summary>
        public static Color Blend(Color color1, Color color2)
        {
            return Blend(color1, color2, 0.5f);
        }

        /// <summary>
        /// 混合率を指定してブレンド。
        /// </summary>
        public static Color Blend(Color color1, Color color2, float ratio)
        {
            byte r = ToElement(ToRatio(color1.R) * (1.0f - ratio) + ToRatio(color2.R) * ratio);
            byte g = ToElement(ToRatio(color1.G) * (1.0f - ratio) + ToRatio(color2.G) * ratio);
            byte b = ToElement(ToRatio(color1.B) * (1.0f - ratio) + ToRatio(color2.B) * ratio);

            return Color.FromArgb(r, g, b);
        }

        /// <summary>
        /// アルファカラーのブレンド。
        /// </summary>
        public static Color AlphaBlend(Color color, Color alphaColor)
        {
            return Blend(color, alphaColor, ToRatio(alphaColor.A));
        }

        /// <summary>
        /// 行列を指定してブレンド。
        /// </summary>
        public static Color MatrixBlend(Color color, ColorMatrix cm)
        {
            float rf = ToRatio(color.R);
            float gf = ToRatio(color.G);
            float bf = ToRatio(color.B);
            float af = ToRatio(color.A);

            byte r = ToElement(rf * cm.Matrix00 + gf * cm.Matrix10 + bf * cm.Matrix20 + af * cm.Matrix30 + 1.0f * cm.Matrix40);
            byte g = ToElement(rf * cm.Matrix01 + gf * cm.Matrix11 + bf * cm.Matrix21 + af * cm.Matrix31 + 1.0f * cm.Matrix41);
            byte b = ToElement(rf * cm.Matrix02 + gf * cm.Matrix12 + bf * cm.Matrix22 + af * cm.Matrix32 + 1.0f * cm.Matrix42);
            byte a = ToElement(rf * cm.Matrix03 + gf * cm.Matrix13 + bf * cm.Matrix23 + af * cm.Matrix33 + 1.0f * cm.Matrix43);

            return Color.FromArgb(a, r, g, b);
        }

        /// <summary>
        /// 要素（0-255）を割合率（0.0-1.0）に変換。
        /// </summary>
        public static float ToRatio(byte element)
        {
            return (float)element / 255.0f;
        }

        /// <summary>
        /// 割合率（0.0-1.0）を要素（0-255）に変換。
        /// </summary>
        public static byte ToElement(float ratio)
        {
            if      (ratio < 0.0f) { return 0;                      }
            else if (ratio > 1.0f) { return 255;                    }
            else                   { return (byte)(ratio * 255.0f); }
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        private ColorBlender() {}
    }
}
