﻿// ========================================================================
// <copyright file="DocumentOpener.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Diagnostics;
using System.IO;
//using App.Controls;
using App.Data;
using App.Utility;
using NWCore.DataModel;

namespace App.IO
{
    /// <summary>
    /// ドキュメントオープナ。
    /// </summary>
    public class DocumentOpener
    {
        //---------------------------------------------------------------------
        /// <summary>
        /// ファイルの存在チェック。
        /// </summary>
        protected bool CheckFileExists(string filePath)
        {
            if (!File.Exists(filePath))
            {
                ErrorFileNotFound(filePath);
                return false;
            }
            return true;
        }

        /// <summary>
        /// ドキュメントを開く。
        /// </summary>
        protected IDocument OpenDocument(
            string filePath, DocumentComponentChangedEventArgs changed)
        {
            //TheApp.MainFrame.StatusMessage =
            //	StringResource.Get("IO_LOADING_MSG",
            //                       Name.GetFileName(filePath));

            IDocument result = null;
            DateTime start = DateTime.Now;
            try
            {
                using (WaitCursor wait = new WaitCursor())
                {
                    result = OpenDocumentImplement(filePath, changed);
                }
            }
            catch (Exception ex)
            {
                DebugConsole.WriteLine(ex.Message);
            }

            TimeSpan time = DateTime.Now - start;
            DebugConsole.WriteLine("Open    {0,4:f1}ms {1,4:f1}sec {2}",
                                   time.TotalMilliseconds,
                                   time.TotalSeconds,
                                   Path.GetFileName(filePath));

            return result;
        }

        /// <summary>
        /// ドキュメントを開く（TemplateMethod）。
        /// </summary>
        protected virtual IDocument OpenDocumentImplement(
            string filePath, DocumentComponentChangedEventArgs changed)
        {
            return null;
        }

        /// <summary>
        /// 未対応エラー。
        /// </summary>
        protected void ErrorUnknown(Exception error)
        {
            string message = error.Message;
#if DEBUG
            message += "\n\n" + error.StackTrace;
#endif
            /*
            ThreadSafeMsgBox.Show( message,
                                   res.Strings.ERROR_CAPTION,
                                   System.Windows.Forms.MessageBoxButtons.OK,
                                   System.Windows.Forms.MessageBoxIcon.Error );
            */
            Debug.Assert(false);
        }

        /// <summary>
        /// ファイルが見つからないエラー。
        /// </summary>
        protected void ErrorFileNotFound(string fileName)
        {
            string message = string.Format(res.Strings.IO_FILE_NOT_FOUND_MSG, fileName);
            /*
            ThreadSafeMsgBox.Show( message,
                                   res.Strings.ERROR_CAPTION,
                                   System.Windows.Forms.MessageBoxButtons.OK,
                                   System.Windows.Forms.MessageBoxIcon.Error );
            */
        }

        /// <summary>
        /// ファイル名重複エラー。
        /// </summary>
        protected void ErrorFileNameCollision(string filePath)
        {
            string message = string.Format(res.Strings.IO_FILENAME_COLLISION_MSG, Path.GetFileName(filePath));
            /*
            ThreadSafeMsgBox.Show( message,
                                   res.Strings.WARNING_CAPTION,
                                   System.Windows.Forms.MessageBoxButtons.OK,
                                   System.Windows.Forms.MessageBoxIcon.Warning );
            */
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// 無効なTGAエラー。
        /// </summary>
        protected void ErrorInvalidTga(string fileName)
        {
            string message = string.Format(res.Strings.IO_INVALID_TGA, fileName);
            /*
            ThreadSafeMsgBox.Show( message,
                                   res.Strings.ERROR_CAPTION,
                                   System.Windows.Forms.MessageBoxButtons.OK,
                                   System.Windows.Forms.MessageBoxIcon.Error );
            */
        }

        /// <summary>
        /// データ上書きエラー。
        /// </summary>
        protected void ErrorDataOverwrite(string name)
        {
            string message = string.Format(res.Strings.IO_DATA_OVERWRITE_FAILED_MSG, name);
            /*
            ThreadSafeMsgBox.Show( message,
                                   res.Strings.WARNING_CAPTION,
                                   System.Windows.Forms.MessageBoxButtons.OK,
                                   System.Windows.Forms.MessageBoxIcon.Warning );
            */
        }
    }
}
