﻿// ========================================================================
// <copyright file="DocumentClipboardData.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.Windows.Forms;
//using App.Command;
using App.Data;
using NintendoWare.ToolDevelopmentKit.Xml;
using NWCore.DataModel;
using NWCore.Serializer;

namespace App.IO
{
    [Serializable]
    public class DocumentClipboardData
    {
        #region Constructor

        /// <summary>
        /// Internal default constructor.
        /// </summary>
        protected DocumentClipboardData()
        {
            //this.MainFrameHanlde = TheApp.MainFrame.Handle;
        }


        /// <summary>
        /// Constructor for setting an emitter set document.
        /// </summary>
        /// <param name="document">The emitter set document.</param>
        public DocumentClipboardData( EmitterSetDocument document ) :
            this()
        {
            if ( document.Name==null )
                this.DocumentName = string.Empty;
            else
                this.DocumentName = string.Copy( document.Name );

            this.DocumentID = document.DocumentID;

            this.ChildDocuments = new List<DocumentClipboardData>();

            SerializeDocument( document );
        }


        /// <summary>
        /// Constructor for setting an emitter document.
        /// </summary>
        /// <param name="document">The emitter document.</param>
        public DocumentClipboardData( EmitterDocument document ) :
            this()
        {
            if ( document.Name==null )
                this.DocumentName = string.Empty;
            else
                this.DocumentName = string.Copy( document.Name );

            this.DocumentID = document.DocumentID;

            this.ChildDocuments = new List<DocumentClipboardData>();

            SerializeDocument( document );
        }


        /// <summary>
        /// Constructor for setting an game settings document.
        /// </summary>
        /// <param name="document">The game settings document.</param>
        public DocumentClipboardData( GameSettingsDocument document ) :
            this()
        {
            if ( document.Name==null )
                this.DocumentName = string.Empty;
            else
                this.DocumentName = string.Copy( document.Name );

            this.DocumentID = document.DocumentID;

            this.ChildDocuments = new List<DocumentClipboardData>();

            SerializeDocument( document );
        }


        /// <summary>
        /// Constructor for setting an field document.
        /// </summary>
        /// <param name="document">The game settings document.</param>
        public DocumentClipboardData(FieldDocument document)
        {
            if (document.Name == null)
                this.DocumentName = string.Empty;
            else
                this.DocumentName = string.Copy(document.Name);

            this.DocumentID = document.DocumentID;

            this.ChildDocuments = new List<DocumentClipboardData>();

            SerializeDocument(document);
        }

        /// <summary>
        /// Constructor for setting an field document.
        /// </summary>
        /// <param name="document">The game settings document.</param>
        public DocumentClipboardData(FluctuationDocument document)
        {
            if (document.Name == null)
                this.DocumentName = string.Empty;
            else
                this.DocumentName = string.Copy(document.Name);

            this.DocumentID = document.DocumentID;

            this.ChildDocuments = new List<DocumentClipboardData>();

            SerializeDocument(document);
        }

        /// <summary>
        /// Constructor for setting an field document.
        /// </summary>
        /// <param name="document">The game settings document.</param>
        public DocumentClipboardData(ChildDocument document)
        {
            if (document.Name == null)
                this.DocumentName = string.Empty;
            else
                this.DocumentName = string.Copy(document.Name);

            this.DocumentID = document.DocumentID;

            this.ChildDocuments = new List<DocumentClipboardData>();

            SerializeDocument(document);
        }

        /// <summary>
        /// Constructor for setting an game settings document.
        /// </summary>
        /// <param name="docData">The game settings document data.</param>
        public DocumentClipboardData( GameConfigDataXml docData ) :
            this()
        {
            this.DocumentName   = string.Copy( docData.Name );
            this.ChildDocuments = new List<DocumentClipboardData>();
            this.DocumentType   = GuiObjectID.GameSettings;

            this.DocumentID = 0;

            // Serialize the XML package to a string.
            string serializedData;
            XmlUtility.SerializeToString( docData, out serializedData );

            // Save the serialized data.
            this.Data = serializedData;
        }

        #endregion

        #region Properties

        /// <summary>
        /// Get or set the document type.
        /// </summary>
        public GuiObjectID DocumentType { get; private set; }


        /// <summary>
        /// Get or set the document name.
        /// </summary>
        public string DocumentName { get; private set; }


        /// <summary>
        /// Get the ID of the document.
        /// </summary>
        public int DocumentID { get; private set; }


        /// <summary>
        /// Get the handle to the EffectMaker instance where the data was copied from.
        /// </summary>
        public IntPtr MainFrameHanlde { get; private set; }


        /// <summary>
        /// Get or set the serialized string data.
        /// </summary>
        public string Data { get; private set; }


        /// <summary>
        /// Get or set the child document data list.
        /// </summary>
        public List<DocumentClipboardData> ChildDocuments { get; private set; }

        #endregion

        #region Serialize

        /// <summary>
        /// Serialize the given document to a XML string.
        /// </summary>
        /// <param name="document">The document to serialize.</param>
        /// <returns>True on success.</returns>
        public bool SerializeDocument( EmitterSetDocument document )
        {
            this.DocumentType = GuiObjectID.EmitterSet;

            if ( document==null )
            {
                this.Data = string.Empty;
                return false;
            }

            // Collect the emitters to set to the serializer
            List<EmitterData> emitters = new List<EmitterData>();
            foreach ( IEmitterDocument item in document.EmitterDocuments )
            {
                emitters.Add( item.EmitterData );
            }

            // Create the XML package for serializing the
            // emitter set document to system clipboard.
            EmitterSetDataXml xml = new EmitterSetDataXml( document.EmitterSetData,
                                                           emitters,
                                                           false );

            if ( xml==null )
                return false;

            // Serialize the XML package to a string.
            string serializedData;
            XmlUtility.SerializeToString( xml, out serializedData );

            // Save the serialized data.
            this.Data = serializedData;

            // Serialize game settings documents
            foreach ( GameSettingsDocument childDoc in document.GameSettingsDocuments )
            {
                if ( childDoc==null )
                    continue;

                this.ChildDocuments.Add( new DocumentClipboardData( childDoc ) );
            }

            return true;
        }


        /// <summary>
        /// Serialize the given document to a XML string.
        /// </summary>
        /// <param name="document">The document to serialize.</param>
        /// <returns>True on success.</returns>
        public bool SerializeDocument( EmitterDocument document )
        {
            this.DocumentType = GuiObjectID.Emitter;

            if ( document==null ||
                 document.EmitterData==null )
            {
                this.Data = string.Empty;
                return false;
            }

            // Create the XML package for serializing the
            // emitter data to system clipboard.
            ComplexEmitterDataXml xml = new ComplexEmitterDataXml( document.EmitterData, false );

            // Serialize the XML package to a string.
            string serializedData;
            XmlUtility.SerializeToString( xml, out serializedData );

            // Save the serialized data.
            this.Data = serializedData;

            return true;
        }


        /// <summary>
        /// Serialize the given document to a XML string.
        /// </summary>
        /// <param name="document">The document to serialize.</param>
        /// <returns>True on success.</returns>
        public bool SerializeDocument( GameSettingsDocument document )
        {
            this.DocumentType = GuiObjectID.GameSettings;

            if ( document==null ||
                 document.GameConfigData==null )
            {
                this.Data = string.Empty;
                return false;
            }

            // Create the XML package for serializing the
            // game config data to system clipboard.
            GameConfigDataXml xml = new GameConfigDataXml( string.Empty,
                                                           document.GameConfigData );

            // Set up the name and comment
            if ( document.Name!=null )
                xml.Name = string.Copy( document.Name );

            if ( document.Comment==null )
                xml.Comment = string.Empty;
            else
                xml.Comment = string.Copy( document.Comment );

            xml.ModelLinkagePageEnabled = document.ModelLinkagePageEnabled;

            // User functions
            IList<IDataModelSerializeList> dataModels =
                document.DataModelProxy.GetAllDataModelsForSerialize();
            foreach ( DataModelSerializeList dataModel in dataModels )
            {
                UserFunctionXML userFunc =
                        new UserFunctionXML( dataModel.Name, dataModel.ID );

                // Fields in the user function
                foreach ( DataModelSerializeItem item in dataModel )
                {
                    uint iNameCRC = TheApp.CRC32Helper.ComputeCRC32Str( item.Name );
                    userFunc.Add( item.Name, iNameCRC, item.Value );
                }

                xml.UserFunctionList.Add( userFunc );
            }

            // Process pre-serialize event.
            xml.PreSerialize();

            // Serialize the XML package to a string.
            string serializedData;
            XmlUtility.SerializeToString( xml, out serializedData );

            // Save the serialized data.
            this.Data = serializedData;

            return true;
        }

        /// <summary>
        /// Serialize the given document to a XML string.
        /// </summary>
        /// <param name="document">The document to serialize.</param>
        /// <returns>True on success.</returns>
        public bool SerializeDocument( FieldDocument document )
        {
            SerializeDocument(document.EmitterDocument);

            this.DocumentType = GuiObjectID.Field;

            return true;
        }

        /// <summary>
        /// Serialize the given document to a XML string.
        /// </summary>
        /// <param name="document">The document to serialize.</param>
        /// <returns>True on success.</returns>
        public bool SerializeDocument( FluctuationDocument document )
        {
            SerializeDocument(document.EmitterDocument);

            this.DocumentType = GuiObjectID.Fluctuation;

            return true;
        }

        /// <summary>
        /// Serialize the given document to a XML string.
        /// </summary>
        /// <param name="document">The document to serialize.</param>
        /// <returns>True on success.</returns>
        public bool SerializeDocument( ChildDocument document )
        {
            SerializeDocument(document.EmitterDocument);

            this.DocumentType = GuiObjectID.Child;

            return true;
        }

        #endregion

        #region Deserialize

        /// <summary>
        /// Deserialize the data to emitter set document.
        /// </summary>
        /// <param name="project">The project for the emitter set.</param>
        /// <param name="commandDesc">The description for the command.</param>
        /// <returns>The deserialized document, or null on failed.</returns>
        public EmitterSetDocument DeserializeEmitterSetDocument( IProjectDocument project,
                                                                 string commandDesc = "",
                                                                 string newName = "" )
        {
            return null;

            /*
            if ( project==null )
                return null;

            if ( this.DocumentType!=GuiObjectID.EmitterSet )
                return null;

            EmitterSetDataXml xml =
                XmlUtility.DeserializeFromString<EmitterSetDataXml>( this.Data );
            if ( xml==null )
                return null;

            // Create the document
            EmitterSetDocument document = EmitterSetDocument.CreateFromDeserializer( project, xml );

            // Compose document name
            string docName      = newName;
            string extendedName = string.Empty;
            if ( string.IsNullOrEmpty(docName)==false )
            {
                if ( project.FindChild(newName)!=null )
                {
                    App.Controls.ThreadSafeMsgBox.Show( TheApp.MainFrame,
                                                        res.Strings.WARNING_CREATE_ESET_NAME_CONFLICT,
                                                        res.Strings.WARNING_CAPTION,
                                                        MessageBoxButtons.OK,
                                                        MessageBoxIcon.Exclamation );
                    return null;
                }
            }
            else
            {
                docName =
                    DocumentCreator.ComposeCopiedEmitterSetName( project,
                                                                 this.DocumentName );

                // Find the extended substring in the composed document name
                if ( docName.Length>this.DocumentName.Length )
                    extendedName = docName.Substring( this.DocumentName.Length );
            }
            if (docName.Length > DocumentIO.s_MaxFileNameLength)
            {
                DocumentIO.ErrorFileNameLengthCopy(docName);
                return null;
            }

            // Set the name to the document.
            document.Name      = docName;
            document.FileTitle = docName;

            using ( new NWCore.Viewer.MCSDisableBlock() )
            {
                foreach ( IEmitterDocument child in document.EmitterDocuments )
                {
                    EmitterDocument emitter = child as EmitterDocument;
                    if ( emitter!=null )
                        emitter.UpdateDrawPathID( true );
                }
            }

            // Add the emitter set to the given project.
            EmitterSetDocumentAddRemoveCommand command =
                new EmitterSetDocumentAddRemoveCommand( document,
                                                        project );

            command.CanUndo = false;

            // Execute the command
            TheApp.CommandManager.Execute( command, commandDesc );

            #region Deserialize game settings documents

            // Make sure the UI definition has been loaded.
            TheApp.UIManager.UpdatePropertyPanel( PropertyEdit.PropertyPanelID.GameSettingsPropertyPanel );

            foreach ( DocumentClipboardData childData in this.ChildDocuments )
            {
                if ( childData.DocumentType!=GuiObjectID.GameSettings )
                    return null;

                // .prevファイル読み込み
                GameConfigDataXml content = null;
                try
                {
                    content = XmlUtility.DeserializeFromString<GameConfigDataXml>( childData.Data );

                    // Process post-deserialize event.
                    content.PostDeserialize();
                }
                catch ( Exception ex )
                {
                    DebugConsole.WriteLine( ex.Message );
                    return null;
                }

                // Create the game settings document.
                GameSettingsDocumentCreationCommand cmd =
                    new GameSettingsDocumentCreationCommand( childData.DocumentName + extendedName,
                                                             document );

                // Execute the command
                TheApp.CommandManager.ExecuteIgnoreTargetDocument( cmd );

                // Get the created game settings document
                GameSettingsDocument doc =
                    document.FindGameSettingsDocument( childData.DocumentName );
                if ( doc==null )
                    continue;

                // Load the game settings documents.
                doc.LoadFromDeserializer( content );
            }

            // Refresh the page content by showing the first property page of the game settings panel.
            TheApp.UIManager.UpdatePropertyPanel( PropertyEdit.PropertyPanelID.GameSettingsPropertyPanel );

            #endregion

            return document;
            */
        }


        /// <summary>
        /// Deserialize the data to emitter document.
        /// </summary>
        /// <param name="parent">The parent emitter set document.</param>
        /// <param name="bAddToProject">True to add the deserialized document to project.</param>
        /// <param name="bSetModifyFlag">True to set modify mark to the emitter set.</param>
        /// <param name="commandDesc">The description for the command.</param>
        /// <param name="iIndex">The index to insert the new document to.</param>
        /// <param name="bUseSrcDocName">Use the name of the source document.</param>
        /// <returns>The deserialized document, or null on failed.</returns>
        public EmitterDocument DeserializeEmitterDocument( EmitterSetDocument parent,
                                                           bool bAddToProject = true,
                                                           bool bSetModifyFlag = false,
                                                           string commandDesc = "",
                                                           int iIndex = -1,
                                                           bool bUseSrcDocName = false )
        {
            if ( parent==null )
                return null;

            if ( this.DocumentType!=GuiObjectID.Emitter &&
                 this.DocumentType!=GuiObjectID.Field   &&
                 this.DocumentType!=GuiObjectID.Fluctuation &&
                 this.DocumentType!=GuiObjectID.Child )
                return null;

            ComplexEmitterDataXml xml =
                XmlUtility.DeserializeFromString<ComplexEmitterDataXml>( this.Data );
            if ( xml==null )
                return null;

            DocumentIO.ReadTextureFile( xml );

            EmitterDocument document = new EmitterDocument( parent, xml.EditData );
            if ( document==null )
                return null;

            // Compose document name
            string docName = xml.name;
            if ( bUseSrcDocName==false )
            {
                docName =
                    DocumentCreator.ComposeCopiedEmitterName( parent,
                                                              this.DocumentName );
                if (docName.Length > DocumentIO.s_MaxFileNameLength)
                {
                    DocumentIO.ErrorFileNameLengthCopy(docName);
                    return null;
                }
            }

            // Set the name to the document.
            document.Name = docName;

            using ( new NWCore.Viewer.MCSDisableBlock() )
            {
                document.UpdateDrawPathID( true );
            }

            // Add the deserialized document to project.
            if ( bAddToProject==true )
            {
                /*
                EmitterDocumentAddCommand command =
                    new EmitterDocumentAddCommand( parent,
                                                   document,
                                                   iIndex );

                command.SetModifyFlag = bSetModifyFlag;

                // Execute the command
                TheApp.CommandManager.Execute( command,
                                               commandDesc,
                                               "",
                                               parent );
                */
            }

            return document;
        }


        /// <summary>
        /// Deserialize the data to game settings document.
        /// </summary>
        /// <param name="parent">The parent emitter set document.</param>
        /// <param name="bAddToProject">True to add the deserialized document to project.</param>
        /// <param name="commandDesc">The description for the command.</param>
        /// <param name="iIndex">The index to insert the new document to.</param>
        /// <param name="bUseSrcDocName">Use the name of the source document.</param>
        /// <returns>The deserialized document, or null on failed.</returns>
        public GameSettingsDocument DeserializeGameSettingsDocument( EmitterSetDocument parent,
                                                                     bool bAddToProject = true,
                                                                     string commandDesc = "",
                                                                     int iIndex = -1,
                                                                     bool bUseSrcDocName = false )
        {
            return null;

            /*
            if ( parent==null )
                return null;

            if ( this.DocumentType!=GuiObjectID.GameSettings )
                return null;

            // .prevファイル読み込み
            GameConfigDataXml content = null;
            try
            {
                content = XmlUtility.DeserializeFromString<GameConfigDataXml>( this.Data );

                // Process post-deserialize event.
                content.PostDeserialize();
            }
            catch ( Exception ex )
            {
                DebugConsole.WriteLine( ex.Message );
                return null;
            }

            // Compose document name
            string docName = content.Name;
            if ( bUseSrcDocName==false )
            {
                docName =
                    DocumentCreator.ComposeCopiedGameSettingsDocName( parent,
                                                                      this.DocumentName );
                if (docName.Length > DocumentIO.s_MaxFileNameLength)
                {
                    DocumentIO.ErrorFileNameLengthCopy(docName);
                    return null;
                }
            }

            // Create the game settings document.
            GameSettingsDocument doc = null;
            if ( bAddToProject==true )
            {
                GameSettingsDocumentCreationCommand command =
                    new GameSettingsDocumentCreationCommand( docName, parent, iIndex );

                // Execute the command
                TheApp.CommandManager.Execute( command, commandDesc );

                // Get the created game settings document
                doc = parent.FindGameSettingsDocument( docName );
                if ( doc==null )
                    return null;
            }
            else
            {
                doc = new GameSettingsDocument( parent, docName );
            }

            // Make sure the UI definition has been loaded.
            TheApp.UIManager.UpdatePropertyPanel( PropertyEdit.PropertyPanelID.GameSettingsPropertyPanel );

            // Load the game settings document.
            doc.LoadFromDeserializer( content );

            // Refresh the page content by showing the first property page of the game settings panel.
            TheApp.UIManager.UpdatePropertyPanel( PropertyEdit.PropertyPanelID.GameSettingsPropertyPanel );

            return doc;
            */
        }


        /// <summary>
        /// Deserialize the data to game settings document.
        /// </summary>
        /// <param name="target">The game settings document to deserialize to.</param>
        /// <returns>True on success.</returns>
        public bool DeserializeGameSettingsDocument( GameSettingsDocument target )
        {
            if ( target==null )
                return false;

            if ( this.DocumentType!=GuiObjectID.GameSettings )
                return false;

            // .prevファイル読み込み
            GameConfigDataXml content = null;
            try
            {
                content = XmlUtility.DeserializeFromString<GameConfigDataXml>( this.Data );

                // Process post-deserialize event.
                content.PostDeserialize();
            }
            catch ( Exception ex )
            {
                DebugConsole.WriteLine( ex.Message );
                return false;
            }

            // Make sure the UI definition has been loaded.
            //TheApp.UIManager.UpdatePropertyPanel( PropertyEdit.PropertyPanelID.GameSettingsPropertyPanel );

            // Load the game settings document.
            target.LoadFromDeserializer( content );

            // Refresh the page content by showing the first property page of the game settings panel.
            //TheApp.UIManager.UpdatePropertyPanel( PropertyEdit.PropertyPanelID.GameSettingsPropertyPanel );

            return true;
        }

        #endregion

        #region Utility methods

        /// <summary>
        /// Determine if the given document is the source document.
        /// </summary>
        /// <param name="doc">The document to check for.</param>
        /// <returns>True if the document is the source.</returns>
        public bool IsSourceDocument( Document doc )
        {
            // Do we have a document ID to check for?
            if ( this.DocumentID==0 )
                return false;

            if ( doc==null )
                return false;

            // Is the source copied from this instance of EffectMaker?
            /*
            if ( this.MainFrameHanlde==IntPtr.Zero ||
                 TheApp.MainFrame.Handle!=this.MainFrameHanlde )
            {
                return false;
            }
            */

            // Same instance of EffectMaker, so check the document ID.
            if ( this.DocumentID==doc.DocumentID )
                return true;

            return false;
        }


        /// <summary>
        /// Determine if the clipboard data can be pasted and create a new document
        /// when the given document is currently selected.
        /// </summary>
        /// <param name="activeDoc">The active document.</param>
        /// <returns>True if the clipboard data can paste.</returns>
        public bool CanPasteToNewNode( Document activeDoc )
        {
            return false;
        }


        /// <summary>
        /// Determine if the clipboard data can be pasted to the given
        /// active document.
        /// </summary>
        /// <param name="activeDoc">The active document.</param>
        /// <returns>True if the clipboard data can paste.</returns>
        public bool CanPasteToExistingNode( Document activeDoc )
        {
            if ( IsSourceDocument( activeDoc )==true )
                return false;

            if ( this.DocumentType==GuiObjectID.Emitter &&
                 activeDoc is EmitterDocument )
            {
                return true;
            }
            else if ( this.DocumentType==GuiObjectID.GameSettings &&
                      activeDoc is GameSettingsDocument )
            {
                return true;
            }
            else if (this.DocumentType == GuiObjectID.Field &&
                     activeDoc is FieldDocument)
            {
                return true;
            }
            else if (this.DocumentType == GuiObjectID.Fluctuation &&
                     activeDoc is FluctuationDocument)
            {
                return true;
            }
            else if (this.DocumentType == GuiObjectID.Child &&
                     activeDoc is ChildDocument)
            {
                return true;
            }

            return false;
        }


        /// <summary>
        /// Determine if the clipboard data can be pasted to the given
        /// active document as a new child node of it.
        /// </summary>
        /// <param name="activeDoc">The active document.</param>
        /// <returns>True if the clipboard data can paste.</returns>
        public bool CanPasteAsNewChild( Document activeDoc )
        {
            if ( this.DocumentType==GuiObjectID.Emitter &&
                 activeDoc is EmitterSetDocument )
            {
                var eset = activeDoc as EmitterSetDocument;
                if ( eset.Count<TheApp.MAX_EMITTER_COUNT )
                    return true;
            }
            else if ( this.DocumentType==GuiObjectID.GameSettings &&
                      activeDoc is EmitterSetDocument )
            {
                return true;
            }

            return false;
        }

        #endregion
    }
}
