﻿// ========================================================================
// <copyright file="DocumentBinarySaver.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Diagnostics;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Windows.Forms;

using NWCore.DataModel;
//using NWCore.Converter;

//using App.Controls;
using App.Data;
using App.Utility;

namespace App.IO
{
    /// <summary>
    /// バイナリドキュメントセーバ。
    /// </summary>
    public class DocumentBinarySaver
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public DocumentBinarySaver()
        {
            // デフォルトは最適化をします。
            this.IsOptimize = true;
        }

        #region イベント

        //---------------------------------------------------------------------
        /// <summary>
        /// 中間保存イベント。
        /// </summary>
        public event EventHandler TemporarySaved = null;

        /// <summary>
        /// 中間保存ハンドラ。
        /// </summary>
        protected virtual void OnTemporarySaved(EventArgs e)
        {
            if (this.TemporarySaved != null)
            {
                this.TemporarySaved(this, e);
            }
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// バイナリ保存イベント。
        /// </summary>
        public event EventHandler BinarySaved = null;

        /// <summary>
        /// バイナリ保存ハンドラ。
        /// </summary>
        protected virtual void OnBinarySaved(EventArgs e)
        {
            if (this.BinarySaved != null)
            {
                this.BinarySaved(this, e);
            }
        }

        #endregion

        /// <summary>
        /// 最適化をかけて保存するか？
        /// </summary>
        public bool IsOptimize { get; set; }

        /// <summary>
        /// 保存。
        /// </summary>
        /*
        public bool Save( List<IDocument> docList,
                          string filePath,
                          bool bBigEndian,
                          bool bUseNativeFormat )
        {
            using ( WaitCursor wait = new WaitCursor() )
            {
                List<IEmitterSetDocument> esetList =
                    new List<IEmitterSetDocument>( docList.Count );
                foreach ( IDocument doc in docList )
                {
                    IEmitterSetDocument esetDoc = doc as IEmitterSetDocument;
                    if ( esetDoc!=null )
                        esetList.Add( esetDoc );
                }

                // Create the binary converter.
                PtclConverter converter =
                    new PtclConverter( new ConverterContext(bBigEndian,
                                                            bUseNativeFormat,
                                                            Config.Data.Application.IsEftCombinerEditorEnabled,
                                                            TheApp.G3dXsdBasePath,
                                                            Config.Data.Option.ShaderDefPath) );
                converter.LogEvent += new NWCore.LogHandler( TheApp.OutputLogMsg );

                // Convert the emitter sets for Windows.
                MemoryStream stream = new MemoryStream();
                converter.Convert( stream, esetList );

                // Get the buffer.
                byte[] buffer = stream.GetBuffer();

                // Write the result to file.
                FileStream file = new FileStream( filePath, FileMode.Create );
                file.Write(buffer, 0, (int)stream.Length);
                file.Close();

                converter.Dispose();
                return true;
            }
        }
        */

        /// <summary>
        /// NW4F_ROOTの取得。
        /// </summary>
        protected string GetNW4FRoot()
        {
            // NW4F_ROOTの取得
            if (DocumentConstants.NW4FRoot == null)
            {
                ErrorNW4FRootNotFound();
                return null;
            }
            return Path.GetFullPath(DocumentConstants.NW4FRoot);
        }

        /// <summary>
        /// テンポラリパスの取得。
        /// </summary>
        protected string GetTemporallyPath(string NW4FRoot)
        {
            string tempPath =
                string.Format("{0}\\EffectMaker\\work\\temp\\{1:d4}\\", NW4FRoot, _saveCounter);
            _saveCounter++;
            if (_saveCounter == 10000) { _saveCounter = 0; }
            if (!Directory.Exists(tempPath))
            {
                Directory.CreateDirectory(tempPath);
            }
            return tempPath;
        }

        // セーブカウンタ
        private static int _saveCounter = 0;

        /// <summary>
        /// コンバート引数の構築（TemplateMethod）。
        /// </summary>
        protected virtual string BuildConvertArgs(string filePath, List<string> sourcePaths)
        {
            string args = "-o=\"" + filePath + "\"";
            foreach (string source in sourcePaths)
            {
                args += (" \"" + source + "\"");
            }
            return args;
        }

        /// <summary>
        /// バイナリコンバータの取得。
        /// </summary>
        private string GetG3DConverter()
        {
            string NW4FRoot = GetNW4FRoot();
            if (NW4FRoot == null) { return null; }
//TODO:コンバーターはいらない
#if false
            string g3dcvtr = NW4FRoot + "\\CommandLineTools\\bin\\NW4F_g3dcvtr.exe";
            if (!File.Exists(g3dcvtr))
            {
                ErrorBinaryConverterNotFound();
                return null;
            }
            return g3dcvtr;
#else
            return null;
#endif
        }

        /// <summary>
        /// コンバート。
        /// </summary>
        private bool Convert(string filePath, string args)
        {
//TODO:コンバーターはいらない
#if false
            string g3dcvtr = GetG3DConverter();
            if (g3dcvtr == null)
            {
                return false;
            }
            Process process = new Process();
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.CreateNoWindow = true;
            process.StartInfo.FileName = g3dcvtr;

            // 引数のオーバーフローを防ぐためにファイル化する
            string NW4FRoot = GetNW4FRoot();
            if (NW4FRoot == null) { return false; }
            string tempPath = NW4FRoot + "\\EffectMaker\\work\\temp\\";
            if (!Directory.Exists(tempPath)) { Directory.CreateDirectory(tempPath); }
            string argsFilePath = tempPath + "g3dcvtr_args.txt";
            using (StreamWriter writer = new StreamWriter(argsFilePath, false, Encoding.Default))
            {
                writer.Write(args);
            }
            process.StartInfo.Arguments = "@\"" + argsFilePath + "\"";

            // 標準出力の取得
            process.StartInfo.RedirectStandardOutput = true;
            StringBuilder standardOutput = new StringBuilder();
            process.OutputDataReceived +=
                delegate(object sendingProcess, DataReceivedEventArgs outLine)
                {
                    if (!string.IsNullOrEmpty(outLine.Data))
                    {
                        standardOutput.Append(outLine.Data);
                    }
                };

            // 標準エラーの取得
            process.StartInfo.RedirectStandardError = true;
            StringBuilder standardError = new StringBuilder();
            process.ErrorDataReceived +=
                delegate(object sendingProcess, DataReceivedEventArgs outLine)
                {
                    if (!string.IsNullOrEmpty(outLine.Data))
                    {
                        standardError.Append(outLine.Data);
                    }
                };

            TheApp.MainFrame.StatusMessage = GetStatusbarMessage(filePath);
            DateTime start = DateTime.Now;
            {
                process.Start();
                process.BeginOutputReadLine();
                process.BeginErrorReadLine();
                process.WaitForExit();
            }
            DebugConsole.WriteLine("Convert {0,4:f1}sec {1}",
                (DateTime.Now - start).TotalSeconds, Path.GetFileName(filePath));

            // 標準出力処理
            ProcessStandardOutput(standardOutput.ToString());

            // エラー処理
            string error = standardError.ToString();
            if (error != "") { ErrorBinarySave(filePath, error, process.ExitCode); }

            // 結果
            if (process.ExitCode == 0)
            {
                // イベント発行
                OnBinarySaved(EventArgs.Empty);
                return true;
            }
            else
            {
                return false;
            }
#else
            return true;
#endif
        }

        /// <summary>
        /// 標準出力を処理。
        /// </summary>
        protected virtual void ProcessStandardOutput(string standardOutput)
        {
        }

        /// <summary>
        /// 標準出力を処理。
        /// </summary>
        protected virtual string GetStatusbarMessage(string filePath)
        {
            return StringResource.Get("IO_CONVERTING_MSG", Path.GetFileName(filePath));
        }

        //---------------------------------------------------------------------
        // NW4FRootが未定義
        private void ErrorNW4FRootNotFound()
        {
            /*
            ThreadSafeMsgBox.Show( res.Strings.APP_NW4F_NOT_FOUND_MSG,
                                   res.Strings.ERROR_CAPTION,
                                   System.Windows.Forms.MessageBoxButtons.OK,
                                   System.Windows.Forms.MessageBoxIcon.Error );
            */
        }

        // バイナリコンバータが見つからない
        private void ErrorBinaryConverterNotFound()
        {
            /*
            ThreadSafeMsgBox.Show( res.Strings.IO_BINARY_SAVE_G3DCVTR_NOT_FOUND_MSG,
                                   res.Strings.ERROR_CAPTION,
                                   System.Windows.Forms.MessageBoxButtons.OK,
                                   System.Windows.Forms.MessageBoxIcon.Error );
            */
        }

        // バイナリセーブエラー
        private void ErrorBinarySave(string filePath, string message, int exitCode)
        {
            /*
            IOErrorDialog dialog = new IOErrorDialog();
            if (exitCode == 0)
            {
                dialog.ShowWarningDialog(
                    StringResource.Get("IO_BINARY_SAVE_WARNING_MSG", filePath), message);
            }
            else
            {
                dialog.ShowErrorDialog(
                    StringResource.Get("IO_BINARY_SAVE_ERROR_MSG", filePath), message);
            }
            */
        }

    }

    //=========================================================================
    /// <summary>
    /// プレビューバイナリセーバ。
    /// </summary>
    public sealed class PreviewBinarySaver : DocumentBinarySaver
    {

        /// <summary>
        /// コンバート引数の構築（TemplateMethod）。
        /// </summary>
        protected override string BuildConvertArgs(string filePath, List<string> sourcePaths)
        {
#if DEBUG
            string args = "-viewer -o=\"" + filePath + "\"";
#else
            string args = "-disable_schema -viewer -o=\"" + filePath + "\"";
#endif
            foreach (string source in sourcePaths) { args += (" \"" + source + "\""); }
            return args;
        }
    }

    //=========================================================================
    /// <summary>
    /// バイナリドキュメントアナライザ。
    /// </summary>
    public sealed class BinaryDocumentAnalyzer : DocumentBinarySaver
    {
        private string _summary = string.Empty;
        private long _fileSize = 0;

        /// <summary>
        /// コンバート引数の構築（TemplateMethod）。
        /// </summary>
        protected override string BuildConvertArgs(string filePath, List<string> sourcePaths)
        {
            string args = "-disable_schema -summary -o=\"" + filePath + "\"";
            foreach (string source in sourcePaths) { args += (" \"" + source + "\""); }
            return args;
        }

        /// <summary>
        /// 標準出力を処理。
        /// </summary>
        protected override void ProcessStandardOutput(string standardOutput)
        {
            _summary = standardOutput;
        }

        /// <summary>
        /// 標準出力を処理。
        /// </summary>
        protected override string GetStatusbarMessage(string filePath)
        {
            return StringResource.Get("IO_ANALYZING_MSG", Path.GetFileName(filePath));
        }

        /// <summary>
        /// ファイルサイズの取得。
        /// </summary>
        public long FileSize { get { return _fileSize; } }

        /// <summary>
        /// 概要の取得。
        /// </summary>
        public string Summary { get { return _summary; } }
    }

    //=========================================================================
    /// <summary>
    /// ビューアバイナリセーバ。
    /// </summary>
    public sealed class ViewerBinarySaver : DocumentBinarySaver
    {
        /// <summary>
        /// ビューア用バイナリ保存。
        /// </summary>
        /*
        public bool SaveForViewer()
        {
            // 出力先の指定
            FolderBrowserDialog dlg = new FolderBrowserDialog();
            dlg.Description = StringResource.Get("IO_SAVE_DIRECTORY_SPECIFY_MSG");

            if (GetNW4FRoot() != null)
            {
                string selectedPath = Path.GetFullPath(GetNW4FRoot() + "\\Viewer\\dvdroot");
                if (Directory.Exists(selectedPath)) { dlg.SelectedPath = selectedPath; }
            }

            if (dlg.ShowDialog(TheApp.MainFrame) != DialogResult.OK) { return false; }

            return SaveForViewer(dlg.SelectedPath + "\\");
        }
        */

        /// <summary>
        /// ビューア用バイナリ保存。
        /// </summary>
        public bool SaveForViewer(string basePath)
        {
            bool result = true;
#if fasle   //TODO:CUT
            Debug.Assert((ModelManager.Models.Count + SceneManager.Scenes.Count) > 0);

            // モデルの出力
            foreach (Model model in ModelManager.Models)
            {
                List<IDocument> modelSet = new List<IDocument>();
                modelSet.Add(model);
                foreach (Texture texture in model.GetTextures()) { modelSet.Add(texture); }
                foreach (Palette palette in model.GetPalettes()) { modelSet.Add(palette); }
                string modelPath = basePath + model.Name + DocumentConstants.DotBrres;
                if (!Save(modelSet, modelPath)) { result = false; }

                // アニメーションの出力
                foreach (string action in model.GetActionNames())
                {
                    List<IDocument> actionSet = new List<IDocument>();
                    List<ModelAnimation> animations = model.GetActionAnimations(action);
                    foreach (ModelAnimation animation in animations)
                    {
                        actionSet.Add(animation);
                        // パターンアニメならテクスチャとパレット追加
                        TexPatternAnimation texAnim = animation as TexPatternAnimation;
                        if (texAnim != null)
                        {
                            foreach (Texture texture in texAnim.Textures)
                            {
                                actionSet.Add(texture);
                            }
                            foreach (Palette palette in texAnim.Palettes)
                            {
                                actionSet.Add(palette);
                            }
                        }
                    }
                    string actionPath =
                        basePath + model.Name + "__" + action + DocumentConstants.DotBrres;
                    if (!Save(actionSet, actionPath)) { result = false; }
                }
            }

            // シーンの出力
            foreach (Scene scene in SceneManager.Scenes)
            {
                string scenePath = basePath + scene.BinaryFileName;
                if (!Save(scene, scenePath)) { result = false; }
            }
#endif
            return result;
        }
    }

    //=========================================================================
    /// <summary>
    /// バイナリドキュメントセーバステータスインジケータクラス。
    /// </summary>
    public sealed class DocumentBinarySaverStatusIndicator : StatusIndicator
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DocumentBinarySaverStatusIndicator( DocumentBinarySaver saver,
                                                   int stepCount) :
            base(stepCount)
        {
            m_saver = saver;
            if ( m_saver!=null )
            {
                m_saver.TemporarySaved += new EventHandler( OnTemporaryFileSaved );
                m_saver.BinarySaved    += new EventHandler( OnBinaryFileSaved );
            }
        }


        /// <summary>
        /// 破棄。
        /// </summary>
        public override void Dispose()
        {
            if ( m_saver!=null )
            {
                m_saver.TemporarySaved -= new EventHandler( OnTemporaryFileSaved );
                m_saver.BinarySaved    -= new EventHandler( OnBinaryFileSaved );
            }

            base.Dispose();
        }


        /// <summary>
        /// Handle TemporarySaved event sent from the saver.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnTemporaryFileSaved( object sender,
                                           EventArgs e )
        {
            AdvanceProgress();
        }


        /// <summary>
        /// Handle BinarySaved event sent from the saver.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnBinaryFileSaved( object sender,
                                        EventArgs e )
        {
            AdvanceProgress();
        }


        private DocumentBinarySaver m_saver = null;
    }
}
