﻿// ========================================================================
// <copyright file="Vector.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System.Diagnostics;
using System.Xml.Serialization;

namespace App.Data
{
    /// <summary>
    /// ベクタクラス。
    /// </summary>
    public sealed class Vector
    {
        private float _x = 0.0f;
        private float _y = 0.0f;
        private float _z = 0.0f;

        #region コンストラクタ
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public Vector() { }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public Vector(Vector src) { Set(src); }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public Vector(float x, float y, float z) { Set(x, y, z); }
        #endregion

        #region アトリビュート
        /// <summary>
        /// Ｘ成分。
        /// </summary>
        [XmlElement("x")]
        public float X
        {
            get { return _x; }
            set { _x = value; }
        }

        /// <summary>
        /// Ｙ成分。
        /// </summary>
        [XmlElement("y")]
        public float Y
        {
            get { return _y; }
            set { _y = value; }
        }

        /// <summary>
        /// Ｚ成分。
        /// </summary>
        [XmlElement("z")]
        public float Z
        {
            get { return _z; }
            set { _z = value; }
        }

        /// <summary>
        /// 成分（インデクサ）。
        /// 各成分をインデクス指定でアクセスします。
        /// </summary>
        [XmlIgnore]
        public float this[int index]
        {
            get
            {
                Debug.Assert(0 <= index && index < 3);
                switch (index)
                {
                    case 0: return _x;
                    case 1: return _y;
                    case 2: return _z;
                    default: break;
                }
                return 0;
            }
            set
            {
                Debug.Assert(0 <= index && index < 3);
                switch (index)
                {
                    case 0: this.X = value; break;
                    case 1: this.Y = value; break;
                    case 2: this.Z = value; break;
                    default: break;
                }
            }
        }
        #endregion

        #region 設定
        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(Vector src)
        {
            _x = src._x;
            _y = src._y;
            _z = src._z;
        }

        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(float x, float y, float z)
        {
            _x = x;
            _y = y;
            _z = z;
        }
        #endregion

        #region 型変換

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override string ToString()
        {
            return string.Format("X={0},Y={1},Z={2}", _x, _y, _z);
        }
        #endregion

        #region 比較
        /// <summary>
        /// 等値比較。
        /// </summary>
        public bool Equals(Vector src)
        {
            if (src == null) return false;
            if (src == this) return true;

            return
                _x == src._x &&
                _y == src._y &&
                _z == src._z;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override bool Equals(object obj)
        {
            return Equals(obj as Vector);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override int GetHashCode()
        {
            return
                _x.GetHashCode() ^
                _y.GetHashCode() ^
                _z.GetHashCode();
        }
        #endregion
    }
}
