﻿// ========================================================================
// <copyright file="Hsv.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Diagnostics;
using System.Xml.Serialization;

namespace App.Data
{
    /// <summary>
    /// ＨＳＶカラークラス。
    /// </summary>
    public sealed class Hsv
    {
        private int _h = 0; // 0 - 360
        private int _s = 0; // 0 - 255
        private int _v = 0; // 0 - 255

        #region コンストラクタ
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public Hsv() { }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public Hsv(Hsv src) { Set(src); }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public Hsv(int h, int s, int v) { Set(h, s, v); }
        #endregion

        #region アトリビュート
        /// <summary>
        /// Hue成分。
        /// </summary>
        public int H
        {
            get { return _h; }
            set
            {
                Debug.Assert(0 <= value && value <= 360);
                _h = value;
            }
        }

        /// <summary>
        /// Saturation成分。
        /// </summary>
        public int S
        {
            get { return _s; }
            set
            {
                Debug.Assert(0 <= value && value <= 255);
                _s = value;
            }
        }

        /// <summary>
        /// Value成分。
        /// </summary>
        public int V
        {
            get { return _v; }
            set
            {
                Debug.Assert(0 <= value && value <= 255);
                _v = value;
            }
        }

        /// <summary>
        /// 成分（インデクサ）。
        /// 各成分をインデクス指定でアクセスします。
        /// </summary>
        [XmlIgnore]
        public int this[int index]
        {
            get
            {
                Debug.Assert(0 <= index && index < 3);
                switch (index)
                {
                    case 0: return _h;
                    case 1: return _s;
                    case 2: return _v;
                    default: break;
                }
                return 0;
            }
            set
            {
                Debug.Assert(0 <= index && index < 3);
                switch (index)
                {
                    case 0: this.H = value; break;
                    case 1: this.S = value; break;
                    case 2: this.V = value; break;
                    default: break;
                }
            }
        }
        #endregion

        #region 設定
        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(Hsv src)
        {
            _h = src._h;
            _s = src._s;
            _v = src._v;
        }

        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(int h, int s, int v)
        {
            _h = h;
            _s = s;
            _v = v;
        }
        #endregion

        #region 型変換
        /// <summary>
        /// Rgba型に変換。
        /// </summary>
        public Rgba ToRgba()
        {
            if (_s == 0)
            {
                return new Rgba(_v, _v, _v);
            }

            int dif = (_h * 6) % 360;
            int num1 = (int)(_v * (255 - _s) / 255.0f);
            int num2 = (int)(_v * (255 - _s * dif / 360.0f) / 255.0f);
            int num3 = (int)(_v * (255 - _s * (360 - dif) / 360.0f) / 255.0f);

            switch (_h / 60)
            {
                case 0: return new Rgba(_v, num3, num1);
                case 1: return new Rgba(num2, _v, num1);
                case 2: return new Rgba(num1, _v, num3);
                case 3: return new Rgba(num1, num2, _v);
                case 4: return new Rgba(num3, num1, _v);
                default: return new Rgba(_v, num1, num2);
            }
        }
        #endregion

        #region 比較
        /// <summary>
        /// 等値比較。
        /// </summary>
        public bool Equals(Hsv src)
        {
            if (src == null) return false;
            if (src == this) return true;

            return
                _h == src._h &&
                _s == src._s &&
                _v == src._v;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override bool Equals(object obj)
        {
            return Equals(obj as Hsv);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override int GetHashCode()
        {
            return
                _h.GetHashCode() ^
                _s.GetHashCode() ^
                _v.GetHashCode();
        }
        #endregion
    }
}
