﻿// ========================================================================
// <copyright file="TextureManager.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

//#define TM_LOGGING

using System;
using System.Linq;
using System.IO;
using System.Drawing;
using System.Drawing.Imaging;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using System.Windows.Forms;

using NWCore.DataModel;

//using App.Controls;

//using nw.g3d.iflib;
//using nw.g3d.nw4f_3dif;

//using NintendoWare.Tga;
//using NintendoWare.ColorPicker;

namespace App.Data
{
    #region Enumerator for the results of the texture loading

    /// <summary>
    /// Enumerator for the results of the texture loading.
    /// </summary>
    public enum LoadTextureResults
    {
        Success,

        ErrCannotAccessFile,
        ErrFileNotFound,
        ErrInvalidFilePath,
        ErrNW4FRootNotFound,
        ErrFailLoadingFTX,
        ErrUnknowPixelFormat,
        ErrFailLoadingTGA,
        ErrUnknowTexType,

        ErrUnknown
    }

    #endregion

    /// <summary>
    /// Class for managing the textures used in the application.
    /// </summary>
    public class TextureManager
    {
        private class Logger
        {
            private static readonly string logFile = TheApp.ApplicationPath + "\\TextureManager.log";
            private static readonly object threadLock = new object();
            private static DateTime start;
            private static int NextId = 0;
            private int id;

            public Logger()
            {
                lock (threadLock)
                {
                    if (NextId == 0)
                    {
                        start = DateTime.Now;
                        if (File.Exists(logFile))
                            File.Delete(logFile);
                    }

                    id = NextId++;
                    using (var sw = new StreamWriter(logFile, true))
                    {
                        sw.Write(string.Format("{0:000}  Logging start date: {1:yyyy-MM-dd HH:mm:ss.fff}" + Environment.NewLine, id, DateTime.Now));
                    }
                }
            }

            private void Write(string info)
            {
                lock (threadLock)
                {
                    if (this == null)
                        return;

                    using (var sw = new StreamWriter(logFile, true))
                    {
                        sw.Write(string.Format("{0:000}  {1:dd HH:mm:ss.fff}: {2}" + Environment.NewLine, id, DateTime.Now, info));
                    }
                }
            }

            public static void Write(Logger logger, string info)
            {
                if (logger != null) logger.Write(info);
            }

            public static string DumpException(Exception ex)
            {
                if (ex == null)
                    throw new ArgumentNullException("ex");
                return DumpExceptionRecursive(ex, 0);
            }

            private static string DumpExceptionRecursive(Exception ex, int level)
            {
                var sb = new System.Text.StringBuilder();

                var indent = new string(' ', level * 4);

                sb.AppendLine(string.Format("{0}{1}: {2}", indent, "Type", ex.GetType().FullName));
                sb.AppendLine(string.Format("{0}{1}: {2}", indent, "Message", ex.Message));

                if (ex.Data.Keys.Count > 0)
                {
                    sb.AppendLine(string.Format("{0}{1}:", indent, "Data"));
                    foreach (var key in ex.Data.Keys)
                    {
                        sb.AppendLine(string.Format("{0}  {1}: {2}, {3}: {4}", indent,
                            "Key", key,
                            "Value", ex.Data[key]));
                    }
                }

                if (ex is System.Reflection.ReflectionTypeLoadException)
                {
                    foreach (var loaderException in ((System.Reflection.ReflectionTypeLoadException)ex).LoaderExceptions)
                        sb.AppendLine(DumpExceptionRecursive(loaderException, level + 1));
                }
                else if (ex is AggregateException)
                {
                    foreach (var innerException in ((AggregateException)ex).InnerExceptions)
                        sb.AppendLine(DumpExceptionRecursive(innerException, level + 1));
                }

                if (ex.StackTrace != null)
                {
                    sb.AppendLine(string.Format("{0}{1}:", indent, "Call stack"));
                    DumpStackTrace(level, ex.StackTrace, sb);
                }

                if (ex.InnerException != null)
                {
                    sb.AppendLine();
                    sb.AppendLine(DumpExceptionRecursive(ex.InnerException, level + 1));
                }

                return sb.ToString();
            }

            private static void DumpStackTrace(int level, string stackTrace, System.Text.StringBuilder sb)
            {
                var indent = new string(' ', 2 + level * 4);

                foreach (var line in stackTrace.Split('\n').Select(l => l.Trim()))
                    sb.AppendLine(string.Format("{0}{1}", indent, line));
            }
        }


        #region Class for texture loading message box block

        /// <summary>
        /// Class for showing warning message boxes when texture loading.
        /// </summary>
        public sealed class TextureLoadingMsgBoxBlock : IDisposable
        {
            /// <summary>
            /// Constructor.
            /// </summary>
            public TextureLoadingMsgBoxBlock()
            {
                m_bOrigFlag = TheApp.TextureManager.m_bShowLoadTexWarningMsgBox;

                TheApp.TextureManager.m_bShowLoadTexWarningMsgBox = true;
            }


            /// <summary>
            /// Dispose.
            /// </summary>
            public void Dispose()
            {
                TheApp.TextureManager.m_bShowLoadTexWarningMsgBox = m_bOrigFlag;
            }


            private bool m_bOrigFlag = false;
        }

        #endregion

        #region Class for storing texture data

        /// <summary>
        /// Class for storing texture data.
        /// </summary>
        public class TextureData
        {
            #region Constructor

            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="filePath">File path of the texture.</param>
            /// <param name="fileModTime">The latest modify time of the texture file.</param>
            /// <param name="texInfo">The texture information.</param>
            /// <param name="colorImgs">The RGB channel mip maps.</param>
            /// <param name="alphaImgs">The alpha channel mip maps.</param>
            /// <param name="pixelFormat">Pixel format.</param>
            /// <param name="compSelector">The color component selector.</param>
            /*
            public TextureData( string filePath,
                                DateTime fileModTime,
                                TextureInfo texInfo,
                                Bitmap[] colorImgs,
                                Bitmap[] alphaImgs,
                                Enum pixelFormat,
                                texture_info_comp_selValue[] compSelector)
            {
                if ( string.IsNullOrEmpty(filePath)==true )
                {
                    m_filePath     = string.Empty;
                    m_iFilePathCRC = 0;
                }
                else
                {
                    m_filePath     = string.Copy( filePath.ToLower() );
                    m_iFilePathCRC = TheApp.CRC32Helper.ComputeCRC32Str( m_filePath );
                }

                SetTexture( fileModTime,
                            texInfo,
                            colorImgs,
                            alphaImgs,
                            pixelFormat,
                            compSelector);
            }
            */

            #endregion

            #region Destructor

            /// <summary>
            /// Destructor.
            /// </summary>
            ~TextureData()
            {
                this.m_texData.NativeImage   = null;
                this.m_texData.OriginalImage = null;

                this.m_colorImgs = null;
                this.m_alphaImgs = null;
            }

            #endregion

            #region Properties

            /// <summary>
            /// Get the file path of the texture.
            /// </summary>
            public string FilePath
            {
                get { return m_filePath; }
            }


            /// <summary>
            /// Get the CRC32 hash code of the file path.
            /// </summary>
            public uint FilePathCRC
            {
                get { return m_iFilePathCRC; }
            }


            /// <summary>
            /// Get the loaded texture information.
            /// </summary>
            public TextureInfo TextureInfo
            {
                get { return m_texData; }
            }


            /// <summary>
            /// Approximate size of the texture data.
            /// The size consists of the size of the image buffers ( native & original )
            /// and the size of the bitmaps ( alpha & color ).
            /// </summary>
            public long DataSize
            {
                get { return m_iDataSize; }
            }


            /// <summary>
            /// Get the last modify time of the texture file.
            /// </summary>
            public DateTime ModifyTime
            {
                get { return m_fileModTime; }
            }


            /// <summary>
            /// Get the loaded texture data.
            /// </summary>
            public TextureInfo Data
            {
                get { return m_texData; }
            }


            /// <summary>
            /// Get the RGB channel mip map images.
            /// </summary>
            public Bitmap[] ColorImages
            {
                get { return m_colorImgs; }
            }


            /// <summary>
            /// Get the alpha channel mip map images.
            /// </summary>
            public Bitmap[] AlphaImages
            {
                get { return m_alphaImgs; }
            }

            /// <summary>
            /// Get the RGB channel mip map images.
            /// </summary>
            public Bitmap[] GammaCorrectedColorImages
            {
                get { return m_GammaCorrectedColorImgs; }
            }


            /// <summary>
            /// Get the alpha channel mip map images.
            /// </summary>
            public Bitmap[] GammaCorrectedAlphaImages
            {
                get { return m_GammaCorrectedAlphaImgs; }
            }

            /// <summary>
            /// Get the pixel format of the image.
            /// </summary>
            /*
            public Enum PixelFormat
            {
                get { return m_pixelFormat; }
            }
            */

            /// <summary>
            /// Get the color component selector.
            /// </summary>
            /*
            public texture_info_comp_selValue[] ComponentSelector
            {
                get { return m_compSelector; }
            }
            */

            #endregion

            #region Set texture

            /// <summary>
            /// Set the texture.
            /// </summary>
            /// <param name="fileModTime">The latest modify time of the texture file.</param>
            /// <param name="texInfo">The texture information.</param>
            /// <param name="colorImgs">The RGB channel mip maps.</param>
            /// <param name="alphaImgs">The alpha channel mip maps.</param>
            /// <param name="pixelFormat">Pixel format.</param>
            /// <param name="compSelector">The color component selector.</param>
            /*
            public void SetTexture( DateTime fileModTime,
                                    TextureInfo texInfo,
                                    Bitmap[] colorImgs,
                                    Bitmap[] alphaImgs,
                                    Enum pixelFormat,
                                    texture_info_comp_selValue[] compSelector )
            {
                m_fileModTime  = fileModTime;
                m_texData      = texInfo;
                m_colorImgs    = colorImgs;
                m_alphaImgs    = alphaImgs;
                m_pixelFormat  = pixelFormat;
                m_compSelector = compSelector;

                // Compute the approximate size the texture data takes.
                m_iDataSize = 0;
                if ( m_texData!=null )
                    m_iDataSize += m_texData.OriginalDataSize + m_texData.NativeDataSize;

                bool bNeedInvGamma =
                    TheApp.TextureManager.IsGammaCorrectedImage( m_texData.NativeDataFormat );
                if ( m_colorImgs!=null && m_colorImgs.Length>0 )
                {
                    foreach ( Bitmap img in m_colorImgs )
                    {
                        m_iDataSize += img.Width * img.Height * 3; // Rgb24 format

                        // Apply inverse gamma correction to the color image.
                        if ( bNeedInvGamma==true )
                            ColorUtil.ApplyInvRGBImageGamma( img );
                    }
                }

                if ( m_alphaImgs!=null && m_alphaImgs.Length>0 )
                {
                    foreach ( Bitmap img in m_alphaImgs )
                    {
                        m_iDataSize += img.Width * img.Height * 3; // Rgb24 format
                    }
                }

                m_GammaCorrectedColorImgs = null;
                m_GammaCorrectedAlphaImgs = null;
            }
            */

            #endregion

            #region Utility methods

            /// <summary>
            /// Generate gamma corrected images
            /// </summary>
            public void GenerateGammaCorrectedImages()
            {
                // Swizzle linear flag.
                bool[] linearSwizzled = new bool[4];
#if false
                for (int i = 0; i < 4; ++i)
                {
                    switch (m_compSelector[i])
                    {
                        case texture_info_comp_selValue.r:
                            linearSwizzled[i] = m_texData.Linear[0];
                            break;
                        case texture_info_comp_selValue.g:
                            linearSwizzled[i] = m_texData.Linear[1];
                            break;
                        case texture_info_comp_selValue.b:
                            linearSwizzled[i] = m_texData.Linear[2];
                            break;
                        case texture_info_comp_selValue.a:
                            linearSwizzled[i] = m_texData.Linear[3];
                            break;
                        default:
                            linearSwizzled[i] = false;
                            break;
                    }
                }
#else
                // EffectMaker上ではリニアフラグに依存せず
                // リニアモードならガンマ補正をかけるので全部true
                linearSwizzled[0] = true;
                linearSwizzled[1] = true;
                linearSwizzled[2] = true;
                linearSwizzled[3] = false;
#endif

                if ( m_GammaCorrectedColorImgs==null &&
                     m_colorImgs!=null)
                {
                    // Apply gamma correction
                    m_GammaCorrectedColorImgs = new Bitmap[m_colorImgs.Length];

                    for ( int i=0;i<m_colorImgs.Length;++i )
                    {
                        // Copy the bitmap.
                        Bitmap img = new Bitmap( m_colorImgs[i].Width,
                                                 m_colorImgs[i].Height,
                                                 System.Drawing.Imaging.PixelFormat.Format24bppRgb );
                        App.Utility.GraphicsUtility.CopyBitmap( m_colorImgs[i], img );

                        m_GammaCorrectedColorImgs[i] = img;

                        // Image data size.
                        m_iDataSize += img.Width * img.Height * 3; // Rgb24 format

                        // Apply gamma correction to the image.
                        /*
                        ColorUtil.ApplyRGBImageGamma( img,
                                                      true,
                                                      linearSwizzled[0],
                                                      linearSwizzled[1],
                                                      linearSwizzled[2] );
                        */
                    }
                }

                if ( m_GammaCorrectedAlphaImgs==null &&
                     m_alphaImgs!=null )
                {
                    // Apply gamma correction
                    m_GammaCorrectedAlphaImgs = new Bitmap[m_alphaImgs.Length];

                    for ( int i=0;i<m_alphaImgs.Length;++i )
                    {
                        // Copy the bitmap.
                        Bitmap img = new Bitmap( m_alphaImgs[i].Width,
                                                 m_alphaImgs[i].Height,
                                                 System.Drawing.Imaging.PixelFormat.Format24bppRgb );
                        App.Utility.GraphicsUtility.CopyBitmap( m_alphaImgs[i], img );

                        m_GammaCorrectedAlphaImgs[i] = img;

                        // Image data size.
                        m_iDataSize += img.Width * img.Height * 3; // Rgb24 format

                        if ( TheApp.TextureManager.IsGammaCorrectedImage(m_texData.NativeDataFormat)==false )
                        {
                            // Apply gamma correction to the image.
                            /*
                            ColorUtil.ApplyRGBImageGamma( img,
                                                          true,
                                                          linearSwizzled[3],
                                                          linearSwizzled[3],
                                                          linearSwizzled[3] );
                            */
                        }
                    }
                }
            }

            #endregion

            #region Member variables

            private string      m_filePath       = string.Empty;
            private uint        m_iFilePathCRC   = 0;
            private long        m_iDataSize      = 0;
            private DateTime    m_fileModTime    = DateTime.Now;
            private TextureInfo m_texData        = null;
            private Bitmap[]    m_colorImgs      = null;
            private Bitmap[]    m_alphaImgs      = null;
            private Bitmap[]    m_GammaCorrectedColorImgs      = null;
            private Bitmap[]    m_GammaCorrectedAlphaImgs      = null;

            //private texture_info_comp_selValue[] m_compSelector = null;
            //private Enum                         m_pixelFormat  = TextureFormat.Unknown;

            #endregion
        }

        #endregion

        #region Constructor

        /// <summary>
        /// Constructor.
        /// </summary>
        public TextureManager()
        {
#if TM_LOGGING
                logger = new Logger();
#endif

            // We need to apply inverse gamma correction to these image formats.
            // Compute their CRC32 hash code to avoid the string comparison of the
            // image format strings.
            m_invGammaCorrImgFormatCRCs = new uint[4];

            /*
            m_invGammaCorrImgFormatCRCs[0] = TheApp.CRC32Helper.ComputeCRC32Str( texture_info_quantize_typeType.srgb_8_8_8_8.ToString() );
            m_invGammaCorrImgFormatCRCs[1] = TheApp.CRC32Helper.ComputeCRC32Str( texture_info_quantize_typeType.srgb_bc1.ToString() );
            m_invGammaCorrImgFormatCRCs[2] = TheApp.CRC32Helper.ComputeCRC32Str( texture_info_quantize_typeType.srgb_bc2.ToString() );
            m_invGammaCorrImgFormatCRCs[3] = TheApp.CRC32Helper.ComputeCRC32Str( texture_info_quantize_typeType.srgb_bc3.ToString() );
            */
        }

        #endregion

        #region Properties

        /// <summary>
        /// Get or set the flag indicating whether to reload
        /// the cached texture if it was modified.
        /// </summary>
        public bool IsCheckModTimeEnabled
        {
            get { return m_bCheckModTime; }
            set { m_bCheckModTime = value; }
        }


        /// <summary>
        /// Get or set the flag indicating whether to create
        /// mip map bitmaps when the textures are loaded.
        /// </summary>
        public bool IsCreateBitmapEnabled
        {
            get { return m_bCreateBitmaps; }
            set
            {
                /*
                if ( value==true &&
                     m_texConverter==null )
                {
                    m_texConverter = new TexUtils.Converter();
                    m_texConverter.Initialize();
                    m_texConverter.IsGammaCorrection = false;
                }
                */

                m_bCreateBitmaps = value;
            }
        }


        /// <summary>
        /// Get or set the maximum size of all the texture cache data.
        /// </summary>
        public long MaxTextureCacheSize
        {
            get { return m_iMaxCacheSize; }
            set
            {
                m_iMaxCacheSize = value;
                CheckCacheSizeLimit();
            }
        }

        #endregion

        #region Load texture

        /// <summary>
        /// Load texture without caching
        /// </summary>
        /// <param name="filePath"></param>
        /// <param name="texInfo"></param>
        /// <param name="colorImgs"></param>
        /// <param name="alphaImgs"></param>
        /// <param name="pixelFormat"></param>
        /// <param name="compSelector"></param>
        /// <returns></returns>
        /*
        public LoadTextureResults LoadTextureDirect(string filePath,
                                               out TextureInfo texInfo,
                                               out Bitmap[] colorImgs,
                                               out Bitmap[] alphaImgs,
                                               out Enum pixelFormat,
                                               out texture_info_comp_selValue[] compSelector)
        {
            texInfo = null;
            colorImgs = null;
            alphaImgs = null;
            pixelFormat = TextureFormat.Unknown;
            compSelector = null;

            // Get the file extension.
            string fileExt = Path.GetExtension(filePath).ToLower();

            LoadTextureResults result;
            if (fileExt.IndexOf(App.IO.DocumentConstants.DotFtx) == 0)
            {
                // Load FTX texture file.
                result = LoadTextureFTX(filePath,
                                         out texInfo,
                                         out colorImgs,
                                         out alphaImgs,
                                         out pixelFormat,
                                         out compSelector);
            }
            else if (fileExt.IndexOf(App.IO.DocumentConstants.DotTga) == 0)
            {
                // Load TGA texture file.
                result = LoadTextureTGA(filePath,
                                         out texInfo,
                                         out colorImgs,
                                         out alphaImgs,
                                         out pixelFormat);
            }
            else
            {
                // Unknow texture file format.
                result = LoadTextureResults.ErrUnknowTexType;
            }

            return result;
        }
        */

        /// <summary>
        /// Gumma correction
        /// </summary>
        /// <param name="textureInfo"></param>
        /// <param name="color"></param>
        /// <returns></returns>
        /*
        public Color CorrectGamma(texture_infoType textureInfo,
                                  Color color)
        {
            // ガンマ補正を行います。以下、中間ファイルドキュメントのtexture_infoページからの引用です
            // > 基本的には実機用イメージに対して、true を指定したチャンネルのガンマを sRGB からリニアに補正します。
            // > ただし quantize_type が srgb_8_8_8_8、srgb_bc1、srgb_bc2、srgb_bc3 の場合は補正しません。
            // > srgb_8_8_8_8、srgb_bc1、srgb_bc2、srgb_bc3 の場合は "true true true false" を指定します。

            Func<byte, byte> correctGamma = (byte col) => { return (byte)(255 * Math.Pow((float)col / 255, 1.0f / 2.2f)); };

            byte r = color.R, g = color.G, b = color.B, a = color.A;
            if ( IsGammaCorrectedImage(textureInfo.quantize_type.ToString())==false )
            {
                if (textureInfo.linear[0]) r = correctGamma(color.R); else r = color.R;
                if (textureInfo.linear[1]) g = correctGamma(color.G); else g = color.G;
                if (textureInfo.linear[2]) b = correctGamma(color.B); else b = color.B;
                if (textureInfo.linear[3]) a = correctGamma(color.A); else a = color.A;
            }

            // ガンマ補正の匿名メソッドです
            return Color.FromArgb((int)a, (int)r, (int)g, (int)b);
        }
        */


        /// <summary>
        /// Texture infoを取得します。まだ読み込まれていなければNULLを返します。
        /// </summary>
        /// <param name="filepath"></param>
        /// <param name="?"></param>
        public void GetTextureInfo( string filePath, out TextureInfo texInfo)
        {
            texInfo = null;

            // Check if the texture has already been loaded.
            TextureData data = FindTextureData( filePath );
            if ( data!=null )
            {
                // Do not check the last modify time, just return the texture data.
                texInfo         = data.Data;
            }
        }

        /// <summary>
        /// Load the specified texture file and raise the TextureLoaded event.
        /// </summary>
        /// <param name="filePath">The file path of the texture.</param>
        /// <param name="texInfo">The information about the loaded texture.</param>
        /// <param name="colorImgs">The RGB channel mip maps.</param>
        /// <param name="alphaImgs">The alpha channel mip maps.</param>
        /// <param name="pixelFormat">The pixel format of the image.</param>
        /// <param name="compSelector">The color component selector.</param>
        /// <returns>True on success.</returns>
        /*
        public LoadTextureResults LoadTexture(string filePath,
                                               out TextureInfo texInfo,
                                               out Bitmap[] colorImgs,
                                               out Bitmap[] alphaImgs,
                                               out Enum pixelFormat,
                                               out texture_info_comp_selValue[] compSelector)
        {
            var result = LoadTextureInternal(filePath, out texInfo, out colorImgs, out alphaImgs, out pixelFormat, out compSelector);
            Logger.Write(logger, string.Format("Texture loading result: {0}", result.ToString()));

            NotifyTextureLoaded(texInfo);

            return result;
        }
        */

        /// <summary>
        /// Load the specified texture file. This function doesn't raise the TextureLoaded event.
        /// </summary>
        /// <param name="filePath">The file path of the texture.</param>
        /// <param name="texInfo">The information about the loaded texture.</param>
        /// <param name="colorImgs">The RGB channel mip maps.</param>
        /// <param name="alphaImgs">The alpha channel mip maps.</param>
        /// <param name="pixelFormat">The pixel format of the image.</param>
        /// <param name="compSelector">The color component selector.</param>
        /// <returns>True on success.</returns>
        /*
        private LoadTextureResults LoadTextureInternal( string filePath,
                                               out TextureInfo texInfo,
                                               out Bitmap[] colorImgs,
                                               out Bitmap[] alphaImgs,
                                               out Enum pixelFormat,
                                               out texture_info_comp_selValue[] compSelector)
        {
            texInfo         = null;
            colorImgs       = null;
            alphaImgs       = null;
            pixelFormat     = TextureFormat.Unknown;
            compSelector    = null;

            #region Check if the texture has already been loaded

            // Check if the texture has already been loaded.
            TextureData data = FindTextureData( filePath );
            if ( data!=null )
            {
                if ( this.IsCheckModTimeEnabled==false )
                {
                    // Do not check the last modify time, just return the texture data.
                    texInfo         = data.Data;
                    pixelFormat     = data.PixelFormat;
                    compSelector    = data.ComponentSelector;

                    if ( TheApp.IsGammaCorrectionEnabled==true )
                    {
                        if (data.GammaCorrectedColorImages == null
                         || data.GammaCorrectedAlphaImages == null)
                        {
                            data.GenerateGammaCorrectedImages();
                            UpdateTotalCacheSize();
                        }

                        colorImgs = data.GammaCorrectedColorImages;
                        alphaImgs = data.GammaCorrectedAlphaImages;
                    }
                    else
                    {
                        colorImgs = data.ColorImages;
                        alphaImgs = data.AlphaImages;
                    }

                    DebugOutputTextureCache();

                    return LoadTextureResults.Success;
                }
                else
                {
                    // We do want to check the modify time.
                    if ( File.Exists(filePath)==false )
                    {
                        // The texture file was removed, loading failed.
                        return LoadTextureResults.ErrFileNotFound;
                    }
                    else if ( data.ModifyTime.Equals(File.GetLastWriteTimeUtc(filePath))==true )
                    {
                        // The texture file is not modified, return the texture data.
                        texInfo         = data.Data;

                        if ( TheApp.IsGammaCorrectionEnabled==true )
                        {
                            if (data.GammaCorrectedColorImages == null
                             || data.GammaCorrectedAlphaImages == null)
                            {
                                data.GenerateGammaCorrectedImages();
                                UpdateTotalCacheSize();
                            }

                            colorImgs = data.GammaCorrectedColorImages;
                            alphaImgs = data.GammaCorrectedAlphaImages;
                        }
                        else
                        {
                            colorImgs = data.ColorImages;
                            alphaImgs = data.AlphaImages;
                        }

                        pixelFormat = data.PixelFormat;
                        compSelector    = data.ComponentSelector;

                        DebugOutputTextureCache();

                        return LoadTextureResults.Success;
                    }
                }
            }

            #endregion

            // Either the texture has not been loaded or the file has been modified.
            // Reload the texture.
            #region Load texture file

            LoadTextureResults result = LoadTextureDirect(filePath,
                                                          out texInfo,
                                                          out colorImgs,
                                                          out alphaImgs,
                                                          out pixelFormat,
                                                          out compSelector);


            if (result == LoadTextureResults.Success)
                TheApp.Logger.Info.AddMessage(filePath + res.Strings.IO_FileDoesImport);
            else
                TheApp.Logger.Error.FormatMessage(res.Strings.IO_LOADFILE_FAILED_MSG, filePath);

            // Cache the loaded data.
            if ( result==LoadTextureResults.Success )
            {
                DateTime modTime = File.GetLastWriteTimeUtc( filePath );
                if ( data==null )
                {
                    data = new TextureData( filePath,
                                            modTime,
                                            texInfo,
                                            colorImgs,
                                            alphaImgs,
                                            pixelFormat,
                                            compSelector);

                    AddTextureData( data );

                    CustomFileSystemWatcher watcher = new CustomFileSystemWatcher();
                    watcher.DelayToWait = 1000; // 1000 ms

                    watcher.Path         = Path.GetDirectoryName(filePath);
                    watcher.Filter       = Path.GetFileName(filePath);
                    watcher.NotifyFilter = NotifyFilters.CreationTime |
                                           NotifyFilters.LastWrite |
                                           NotifyFilters.LastAccess |
                                           NotifyFilters.FileName;
                    watcher.SynchronizingObject = TheApp.MainFrame;

                    watcher.Changed += DocumentManager.OnTextureChanged;
                    watcher.Deleted += DocumentManager.OnTextureChanged;
                    watcher.Renamed += DocumentManager.OnTextureChanged;

                    watcher.EnableRaisingEvents = true;

                    lock (s_SyncObject)
                    {
                        s_WatcherList.Add(watcher);
                    }
                }
                else
                {
                    data.SetTexture( modTime,
                                     texInfo,
                                     colorImgs,
                                     alphaImgs,
                                     pixelFormat,
                                     compSelector);
                }

                // Copy gamma corrected images if needed.
                if ( TheApp.IsGammaCorrectionEnabled==true )
                {
                    if (data.GammaCorrectedColorImages == null
                     || data.GammaCorrectedAlphaImages == null)
                    {
                        data.GenerateGammaCorrectedImages();
                        UpdateTotalCacheSize();
                    }

                    colorImgs = data.GammaCorrectedColorImages;
                    alphaImgs = data.GammaCorrectedAlphaImages;
                }

                // Check the cache size.
                // Remove old cache data if the size exceeds the limits.
                CheckCacheSizeLimit();

                DebugOutputTextureCache();
            }

            #endregion

            return result;
        }
        */

        /// <summary>
        /// Load the specified texture file.
        /// </summary>
        /// <param name="filePath">The file path of the texture.</param>
        /// <param name="texInfo">The information about the loaded texture.</param>
        /// <param name="colorImgs">The RGB channel mip maps.</param>
        /// <param name="alphaImgs">The alpha channel mip maps.</param>
        /// <param name="pixelFormat">The pixel format of the image.</param>
        /// <param name="compSelector">The color component selector.</param>
        /// <returns>True on success.</returns>
        /*
        private LoadTextureResults LoadTextureFTX( string filePath,
                                                   out TextureInfo texInfo,
                                                   out Bitmap[] colorImgs,
                                                   out Bitmap[] alphaImgs,
                                                   out Enum pixelFormat,
                                                   out texture_info_comp_selValue[] compSelector)
        {
            Logger.Write(logger, string.Format("Load FTX texture {0}", filePath));

            texInfo      = null;
            colorImgs    = null;
            alphaImgs    = null;
            pixelFormat  = TextureFormat.Unknown;
            compSelector = null;

            if ( String.IsNullOrEmpty(filePath)==true )
                return LoadTextureResults.ErrInvalidFilePath;

            if ( Path.IsPathRooted(filePath)==false )
                return LoadTextureResults.ErrInvalidFilePath;

            if ( File.Exists(filePath)==false )
                return LoadTextureResults.ErrFileNotFound;

            #region Load the .ftx file

            // Load the .ftx file
            nw4f_3difType   loadedObj = null;
            List<G3dStream> streams   = new List<G3dStream>();
            try
            {
                loadedObj = IfReadUtility.Read( streams,
                                                filePath.ToLower(),
                                                TheApp.G3dXsdBasePath );
            }
            catch (Exception e)
            {
                Logger.Write(logger, string.Format("Exception while reading: {0}", Logger.DumpException(e)));

                return LoadTextureResults.ErrFailLoadingFTX;
            }

            #endregion

            #region Check if the texture is correctly loaded

            // Check the texture object is correctly loaded
            if ( (loadedObj==null) ||
                 (loadedObj.Item is textureType)==false )
            {
                return LoadTextureResults.ErrFailLoadingFTX;
            }

            #endregion

            #region Load the texture information

            textureType texture = loadedObj.Item as textureType;

            texInfo = new TextureInfo();

            // Basic information
            texInfo.FileName  = filePath;
            texInfo.Width     = (uint)texture.texture_info.width;
            texInfo.Height    = (uint)texture.texture_info.height;

            texInfo.TileMode  = texture.texture_info.tile_mode.ToString();
            texInfo.Swizzle   = texture.texture_info.initial_swizzle;
            texInfo.Alignment = texture.texture_info.alignment;
            texInfo.Pitch     = texture.texture_info.pitch;
            texInfo.MipLevel  = (uint)texture.texture_info.mip_level;
            texInfo.CompSel   = ConvertComponentSelector( texture.texture_info.comp_sel );
            texInfo.Linear    = texture.texture_info.linear;
            compSelector = texture.texture_info.comp_sel;

            #if BUILD_FOR_CTR
            texInfo.IsNativeTextureType = false;
            #else
            texInfo.IsNativeTextureType = true;
            #endif

            // Mipmap data offset
            if ( texture.texture_info.mip_offset==null ||
                    texture.texture_info.mip_offset.Length<=0 )
            {
                texInfo.MipOffset = new uint[0];
            }
            else
            {
                texInfo.MipOffset = new uint[texture.texture_info.mip_offset.Length];
                Buffer.BlockCopy( texture.texture_info.mip_offset,
                                  0,
                                  texInfo.MipOffset,
                                  0,
                                  texture.texture_info.mip_offset.Length );
            }

            // Native image data
            texInfo.NativeDataFormat = texture.texture_info.quantize_type.ToString();
            texInfo.NativeDataSize   = texture.texture_info.size;

            G3dStream nativeStream = streams[texture.texture_info.stream_index];

            // Load the native image
            texInfo.NativeImage = nativeStream.ByteData.ToArray();

            // Original image data
            original_imageType origImage = texture.original_image_array.original_image[0];
            if ( origImage==null )
            {
                texInfo.OriginalDataFormat = string.Empty;
                texInfo.OriginalDataSize   = 0;
            }
            else
            {
                texInfo.OriginalDataFormat = origImage.format.ToString();
                texInfo.OriginalDataSize   = origImage.size;

                G3dStream origStream = streams[origImage.stream_index];

                bool isGammaCorrectedImage = IsGammaCorrectedImage(texture.texture_info.quantize_type.ToString());

                bool[] linear;
                if (isGammaCorrectedImage)
                {
                    linear = new bool[4]{true, true, true, false};
                }
                else
                {
                    linear = texture.texture_info.linear;
                }

                switch (origImage.format)
                {
                    case original_image_formatType.rgb8:
                        {
                            for (int j = 0; j < 3; ++j)
                            {
                                if (linear[j])
                                {
                                    for (int i = 0; i < texInfo.OriginalDataSize / 3; ++i)
                                    {
                                        origStream.ByteData[i * 3 + j] =
                                            ColorUtil.ApplyColorInvGamma( origStream.ByteData[i * 3 + j],
                                                                            true );
                                    }
                                }
                            }
                            break;
                        }
                    case original_image_formatType.rgba8:
                        {
                            for (int j = 0; j < 4; ++j)
                            {
                                if (linear[j])
                                {
                                    for (int i = 0; i < texInfo.OriginalDataSize / 4; ++i)
                                    {
                                        origStream.ByteData[i * 4 + j] =
                                            ColorUtil.ApplyColorInvGamma( origStream.ByteData[i * 4 + j],
                                                                            true );
                                    }
                                }
                            }
                            break;
                        }
                    case original_image_formatType.rgb32f:
                        {
                            for (int j = 0; j < 3; ++j)
                            {
                                if (linear[j])
                                {
                                    for (int i = 0; i < texInfo.OriginalDataSize / 12; ++i)
                                    {
                                        origStream.FloatData[i * 3 + j] =
                                            ColorUtil.ApplyColorInvGamma( origStream.FloatData[i * 3 + j],
                                                                            true );
                                    }
                                }
                            }
                            break;
                        }
                    case original_image_formatType.rgba32f:
                        {
                            for (int j = 0; j < 4; ++j)
                            {
                                if (linear[j])
                                {
                                    for (int i = 0; i < texInfo.OriginalDataSize / 16; ++i)
                                    {
                                        origStream.FloatData[i * 4 + j] =
                                            ColorUtil.ApplyColorInvGamma( origStream.FloatData[i * 4 + j],
                                                                            true );
                                    }
                                }
                            }
                            break;
                        }
                }

                // Load the original image
                texInfo.OriginalImage = origStream.ByteData.ToArray();
            }


            #endregion

            #region Convert the mipmaps to bitmaps

            if ( m_bCreateBitmaps==true )
            {
                Bitmap[] mipmaps = m_texConverter.ConvertTo1d2dBitmap( texture, streams );

                colorImgs = new Bitmap[mipmaps.Length];
                alphaImgs = new Bitmap[mipmaps.Length];

                Rectangle rect = new Rectangle( 0, 0, 0, 0 );

                App.Utility.FTXPixelFormatInfo pixelInfo =
                    TheApp.PixelFormatUtil.GetPixelFormatInfo( texture.texture_info.quantize_type );
                if ( pixelInfo==null )
                    return LoadTextureResults.ErrUnknowPixelFormat;

                bool bEnableR  = pixelInfo.EnableR;
                bool bEnableG  = pixelInfo.EnableG;
                bool bEnableB  = pixelInfo.EnableB;
                bool bEnableA  = pixelInfo.EnableA;
                byte iDefaultR = pixelInfo.DefaultR;
                byte iDefaultG = pixelInfo.DefaultG;
                byte iDefaultB = pixelInfo.DefaultB;
                byte iDefaultA = pixelInfo.DefaultA;

                if ( bEnableA==true )
                    pixelFormat = TextureFormat.Rgba8;
                else
                    pixelFormat = TextureFormat.Rgb8;

                for ( int i=0;i<mipmaps.Length;++i )
                {
                    Bitmap srcImg = mipmaps[i];

                    rect.Width  = srcImg.Width;
                    rect.Height = srcImg.Height;

                    // Create the bitmaps
                    Bitmap colorImg = new Bitmap( rect.Width,
                                                  rect.Height,
                                                  PixelFormat.Format24bppRgb );

                    Bitmap alphaImg = new Bitmap( rect.Width,
                                                  rect.Height,
                                                  PixelFormat.Format24bppRgb );

                    colorImgs[i] = colorImg;
                    alphaImgs[i] = alphaImg;

                    // Lock the bitmaps for write
                    BitmapData colorLockData =
                        colorImg.LockBits( rect,
                                           ImageLockMode.ReadWrite,
                                           PixelFormat.Format24bppRgb );
                    BitmapData alphaLockData =
                        alphaImg.LockBits( rect,
                                           ImageLockMode.ReadWrite,
                                           PixelFormat.Format24bppRgb );

                    #region Copy the pixels of the mipmap

                    unsafe
                    {
                        byte* colorBuffer = (byte*)(void*)colorLockData.Scan0;
                        byte* alphaBuffer = (byte*)(void*)alphaLockData.Scan0;

                        int   x, y;
                        int   iIndex = 0;
                        Color color;

                        int iExtraStrideBytes = colorLockData.Stride - (colorLockData.Width * 3);

                        for ( y=0;y<rect.Height;++y )
                        {
                            for ( x=0;x<rect.Width;++x )
                            {
                                color = srcImg.GetPixel( x, y );
                                colorBuffer[iIndex]     = GetSwizzled(texture.texture_info.comp_sel[2], color);
                                colorBuffer[iIndex + 1] = GetSwizzled(texture.texture_info.comp_sel[1], color);
                                colorBuffer[iIndex + 2] = GetSwizzled(texture.texture_info.comp_sel[0], color);
                                alphaBuffer[iIndex]     = GetSwizzled(texture.texture_info.comp_sel[3], color);
                                alphaBuffer[iIndex + 1] = alphaBuffer[iIndex];
                                alphaBuffer[iIndex + 2] = alphaBuffer[iIndex];

                                iIndex += 3;
                            }

                            iIndex += iExtraStrideBytes;
                        }

                        if ( i == 0 )
                        {
                            uint[] trimSize = new uint[4];
                            trimSize[0] = trimSize[2] = uint.MaxValue;
                            trimSize[1] = trimSize[3] = uint.MinValue;

                            for (y = 0; y < rect.Height; ++y)
                            {
                                for (x = 0; x < rect.Width; ++x)
                                {
                                    var index = y * rect.Width * 3 + x * 3;
                                    if ( alphaBuffer[index] == 0 ) continue;

                                    if ( x < trimSize[0] ) trimSize[0] = (uint)x;
                                    if ( x > trimSize[1] ) trimSize[1] = (uint)x;
                                    if ( y < trimSize[2] ) trimSize[2] = (uint)y;
                                    if ( y > trimSize[3] ) trimSize[3] = (uint)y;
                                }
                            }

                            if ( trimSize[0] > trimSize[1] || trimSize[2] > trimSize[3] )
                            {
                                trimSize[0] = trimSize[1] = trimSize[2] = trimSize[3] = 0;
                            }

                            texInfo.TrimSize = trimSize;
                        }
                    }

                    #endregion

                    colorImg.UnlockBits( colorLockData );
                    alphaImg.UnlockBits( alphaLockData );
                }
            }

            #endregion

            return LoadTextureResults.Success;
        }
        */

        /// <summary>
        /// Load the specified texture file.
        /// </summary>
        /// <param name="filePath">The file path of the texture.</param>
        /// <param name="texInfo">The information about the loaded texture.</param>
        /// <param name="colorImgs">The RGB channel mip maps.</param>
        /// <param name="alphaImgs">The alpha channel mip maps.</param>
        /// <param name="pixelFormat">The pixel format of the image.</param>
        /// <returns>True on success.</returns>
        /*
        private LoadTextureResults LoadTextureTGA( string filePath,
                                                   out TextureInfo texInfo,
                                                   out Bitmap[] colorImgs,
                                                   out Bitmap[] alphaImgs,
                                                   out Enum pixelFormat )
        {
            Logger.Write(logger, string.Format("Load TGA texture {0}", filePath));

            texInfo     = null;
            colorImgs   = null;
            alphaImgs   = null;
            pixelFormat = TextureFormat.Unknown;

            MemoryStream stream = null;
            try
            {
                if ( String.IsNullOrEmpty(filePath)==true )
                    return LoadTextureResults.ErrInvalidFilePath;

                if ( Path.IsPathRooted(filePath)==false )
                    return LoadTextureResults.ErrInvalidFilePath;

                if ( File.Exists(filePath)==false )
                    return LoadTextureResults.ErrFileNotFound;

                // Load the file to memory.
                stream =
                    new MemoryStream( File.ReadAllBytes(filePath) );
            }
            catch ( Exception )
            {
                return LoadTextureResults.ErrCannotAccessFile;
            }

            // Create the reader.
            TgaReader reader = new TgaReader();

            // Read the texture from the stream.
            try
            {
                reader.Read( stream );
            }
            catch ( UnsupportedTGAFormatException ex )
            {
                if ( m_bShowLoadTexWarningMsgBox==true )
                {
                    ThreadSafeMsgBox.Show( StringResource.Get("ERR_UNSUPPORTED_TGA_FILE_FORMAT", ex.FormatName),
                                           res.Strings.WARNING_CAPTION,
                                           MessageBoxButtons.OK,
                                           MessageBoxIcon.Exclamation,
                                           MessageBoxDefaultButton.Button1,
                                           MessageBoxOptions.DefaultDesktopOnly );
                }
                return LoadTextureResults.ErrFailLoadingTGA;
            }
            catch ( UnsupportedTextureTypeException ex )
            {
                if ( ex.TextureType==TextureTypes.TexCube &&
                     m_bShowLoadTexWarningMsgBox==true )
                {
                    ThreadSafeMsgBox.Show( res.Strings.ERR_UNSUPPORTED_CUBEMAP,
                                           res.Strings.WARNING_CAPTION,
                                           MessageBoxButtons.OK,
                                           MessageBoxIcon.Exclamation,
                                           MessageBoxDefaultButton.Button1,
                                           MessageBoxOptions.DefaultDesktopOnly );
                }
                return LoadTextureResults.ErrFailLoadingTGA;
            }
            catch ( Exception ex )
            {
                if ( m_bShowLoadTexWarningMsgBox==true )
                {
                    ThreadSafeMsgBox.Show( StringResource.Get("ERR_UNSUPPORTED_TGA_FILE_FORMAT", ex.Message),
                                           res.Strings.WARNING_CAPTION,
                                           MessageBoxButtons.OK,
                                           MessageBoxIcon.Exclamation,
                                           MessageBoxDefaultButton.Button1,
                                           MessageBoxOptions.DefaultDesktopOnly );
                }
                return LoadTextureResults.ErrFailLoadingTGA;
            }

            if ( reader.TextureDataSize<=0 )
                return LoadTextureResults.ErrFailLoadingTGA;

            if (reader.ColorBitmap == null || reader.AlphaBitmap == null) // TODO: is it useless ?
                return LoadTextureResults.ErrFailLoadingTGA;

            texInfo = new TextureInfo();

            texInfo.FileName = filePath;

            var colorBitmap = reader.ColorBitmap;
            var alphaBitmap = reader.AlphaBitmap;

            var colorBitmapData = colorBitmap.LockBits(new Rectangle(Point.Empty, colorBitmap.Size), ImageLockMode.ReadOnly, PixelFormat.Format24bppRgb);
            var alphaBitmapData = alphaBitmap.LockBits(new Rectangle(Point.Empty, alphaBitmap.Size), ImageLockMode.ReadOnly, PixelFormat.Format24bppRgb);

            var stride = colorBitmapData.Stride;

            try
            {
                var colorBuffer = new byte[stride * colorBitmap.Height];
                var alphaBuffer = new byte[stride * alphaBitmap.Height];

                Marshal.Copy(colorBitmapData.Scan0, colorBuffer, 0, colorBuffer.Length);
                Marshal.Copy(alphaBitmapData.Scan0, alphaBuffer, 0, alphaBuffer.Length);

                var resultBuffer = new byte[colorBitmap.Width * colorBitmap.Height * 4];

                int height = colorBitmap.Height;
                int width = colorBitmap.Width;
                int k = 0;
                for (int y = 0; y < height; y++)
                {
                    for (int x = 0; x < width; x++)
                    {
                        var index = y * stride + x * 3;
                        resultBuffer[k + 0] = colorBuffer[index + 0];
                        resultBuffer[k + 1] = colorBuffer[index + 1];
                        resultBuffer[k + 2] = colorBuffer[index + 2];
                        resultBuffer[k + 3] = alphaBuffer[index + 0];
                        k += 4;
                    }
                }

                texInfo.OriginalDataFormat = original_image_formatType.rgba8.ToString();
                texInfo.OriginalDataSize = resultBuffer.Length;
                texInfo.OriginalImage = resultBuffer;

                uint[] trimSize = new uint[4];
                trimSize[0] = trimSize[2] = uint.MaxValue;
                trimSize[1] = trimSize[3] = uint.MinValue;

                for (int y = 0; y < height; y++)
                {
                    for (int x = 0; x < width; x++)
                    {
                        var index = y * stride + x * 3;
                        if ( alphaBuffer[index] == 0 ) continue;

                        if (x < trimSize[0]) trimSize[0] = (uint)x;
                        if (x > trimSize[1]) trimSize[1] = (uint)x;
                        if (y < trimSize[2]) trimSize[2] = (uint)y;
                        if (y > trimSize[3]) trimSize[3] = (uint)y;
                    }
                }

                if ( trimSize[0] > trimSize[1] || trimSize[2] > trimSize[3] )
                {
                    trimSize[0] = trimSize[1] = trimSize[2] = trimSize[3] = 0;
                }

                texInfo.TrimSize = trimSize;
            }
            finally
            {
                colorBitmap.UnlockBits(colorBitmapData);
                alphaBitmap.UnlockBits(alphaBitmapData);
            }

            texInfo.NativeDataFormat = reader.TextureFormat.ToString();
            texInfo.NativeImage = reader.NativeImageData;
            texInfo.NativeDataSize = reader.NativeDataSize;

            texInfo.Width = (uint)reader.Size.Width;
            texInfo.Height = (uint)reader.Size.Height;

            texInfo.MipOffset = reader.MipmapOffsets;

            #if BUILD_FOR_CTR
            texInfo.IsNativeTextureType = true;
            texInfo.Alignment           = 128;
            #else
            texInfo.IsNativeTextureType = false;
            #endif

            pixelFormat = reader.TextureFormat;

            texture_info_comp_selValue[] compSel =
            {
                texture_info_comp_selValue.b,
                texture_info_comp_selValue.g,
                texture_info_comp_selValue.r,
                texture_info_comp_selValue.a
            };

            texInfo.CompSel  = ConvertComponentSelector( compSel );
            texInfo.MipLevel = Convert.ToUInt32( reader.MipmapLevel );

            if ( m_bCreateBitmaps==true )
            {
                colorImgs = reader.ColorBitmaps;
                alphaImgs = reader.AlphaBitmaps;
            }

            return LoadTextureResults.Success;
        }
        */

        #endregion

        #region Texture cache data

        /// <summary>
        /// Add the loaded texture data to the cache.
        /// </summary>
        /// <param name="data">The texture data.</param>
        private void AddTextureData( TextureData data )
        {
            // Add the texture data to the cache map.
            m_textures.Add( data.FilePathCRC, data );

            // We need this linked list to keep track of the old cache data.
            m_accessOrderCacheList.AddLast( data );
        }


        /// <summary>
        /// Find the cached texture data with the given file path.
        /// </summary>
        /// <param name="filePath">The file path of the texture file.</param>
        /// <returns>The found texture data or null if not found.</returns>
        private TextureData FindTextureData( string filePath )
        {
            if ( string.IsNullOrEmpty(filePath)==true )
                return null;

            // Compute the hash code of the file path.
            uint iPathCRC = TheApp.CRC32Helper.ComputeCRC32Str( filePath.ToLower() );

            return FindTextureData( iPathCRC );
        }


        /// <summary>
        /// Find the cached texture data with the given file path.
        /// </summary>
        /// <param name="iPathCRC">The CRC hash code of the texture file path.</param>
        /// <returns>The found texture data or null if not found.</returns>
        private TextureData FindTextureData( uint iPathCRC )
        {
            // Find the texture data with the hash code of the file path.
            TextureData data;
            if ( m_textures.TryGetValue(iPathCRC, out data)==false )
                return null;

            // Pull the cache to the latest accessed cache on the list.
            LinkedListNode<TextureData> node = m_accessOrderCacheList.Find( data );
            if ( node!=null )
            {
                m_accessOrderCacheList.Remove( node );
                m_accessOrderCacheList.AddLast( node );
            }

            return data;
        }


        /// <summary>
        /// Check if the cached texture data already reached the size limits.
        /// Remove old cache data if it does.
        /// </summary>
        private void CheckCacheSizeLimit()
        {
            #if DEBUG

                // Update the current total cache size.
                UpdateTotalCacheSize();

                // Is there cache size limitation and is there more than one cache?
                if ( m_iMaxCacheSize<=0 || m_accessOrderCacheList.Count<=1 )
                    return;

            #else

                // Is there cache size limitation and is there more than one cache?
                if ( m_iMaxCacheSize<=0 || m_accessOrderCacheList.Count<=1 )
                    return;

                // Update the current total cache size.
                UpdateTotalCacheSize();

            #endif

            // Is the cache size exceeding the maximum size?
            if ( m_iMaxCacheSize>=m_iTotalCacheSize )
                return;

            // Keep removing the oldest cache data until the
            // total cache size is smaller than the limits.
            LinkedListNode<TextureData> node = m_accessOrderCacheList.First;
            while ( node!=null )
            {
                if ( node.Value!=null )
                {
                    m_iTotalCacheSize -= node.Value.DataSize;

                    // Remove the cached data from our cache map.
                    m_textures.Remove( node.Value.FilePathCRC );
                }

                // Remove the linked list node.
                m_accessOrderCacheList.RemoveFirst();

                // Is the cache size still exceeding the maximum size?
                if ( m_iMaxCacheSize>=m_iTotalCacheSize )
                    return;

                node = m_accessOrderCacheList.First;
            }
        }


        /// <summary>
        /// Update the total texture cache size.
        /// </summary>
        private void UpdateTotalCacheSize()
        {
            m_iTotalCacheSize = 0;

            LinkedListNode<TextureData> node = m_accessOrderCacheList.First;
            while ( node!=null )
            {
                if ( node.Value==null )
                    continue;

                m_iTotalCacheSize += node.Value.DataSize;

                node = node.Next;
            }
        }


        /// <summary>
        /// Output texture cache contents for debugging.
        /// </summary>
        public void DebugOutputTextureCache()
        {
            #if DEBUG

            DebugConsole.WriteLine( "========== TextureManager Cache Output ==========" );
            DebugConsole.WriteLine( "Cache Size : {0} bytes", m_iTotalCacheSize );
            DebugConsole.WriteLine( "=================================================" );
            LinkedListNode<TextureData> node = m_accessOrderCacheList.First;
            while ( node!=null )
            {
                DebugConsole.WriteLine( "    > {0} : {1} bytes",
                                        Path.GetFileName(node.Value.FilePath),
                                        node.Value.DataSize );
                node = node.Next;
            }
            DebugConsole.WriteLine( "=================================================" );

            #endif
        }

        #endregion

        #region Utility methods

        /// <summary>
        /// Check if the texture version matches the version NW4F_nw4f_3dif.dll requires.
        /// </summary>
        /// <param name="filePath">The file path of the texture.</param>
        /// <returns>True if version matches.</returns>
        public bool CheckTextureVersion( string filePath,
                                         out string strDllVersion,
                                         out string strTexVersion )
        {
            // Compose the DLL file path.
            string dllPath = Path.Combine( TheApp.ApplicationPath, "/NW4F_nw4f_3dif.dll" );

            // Get the version of the DLL file.
            System.Diagnostics.FileVersionInfo dllVersion =
                System.Diagnostics.FileVersionInfo.GetVersionInfo( dllPath );

            strDllVersion = dllVersion.FileVersion;
            strTexVersion = string.Empty;
            {
                byte[] ftxBinary = null;
                using ( FileStream fr = new FileStream(filePath, FileMode.Open) )
                {
                    using ( BinaryReader br = new BinaryReader(fr) )
                    {
                        ftxBinary = br.ReadBytes(128);
                    }
                }

                //strTexVersion = IfReadUtility.GetVersion( ftxBinary );
            }

            if ( string.IsNullOrEmpty(strTexVersion)==true ||
                 String.Compare(strDllVersion, 0,
                                strTexVersion, 0,
                                strTexVersion.Length)!=0 )
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// Return the swizzled color.
        /// </summary>
        /// <param name="compSelValue">Value of component selector.</param>
        /// <param name="color">Color to be swizzled.</param>
        /*
        private byte GetSwizzled( texture_info_comp_selValue compSelValue,
                                  Color color )
        {
            switch ( compSelValue )
            {
                case texture_info_comp_selValue.r:
                    return color.R;
                case texture_info_comp_selValue.g:
                    return color.G;
                case texture_info_comp_selValue.b:
                    return color.B;
                case texture_info_comp_selValue.a:
                    return color.A;
                case texture_info_comp_selValue.Item0:
                    return (byte)0;
                case texture_info_comp_selValue.Item1:
                    return (byte)255;
            }

            return 0;
        }
        */


        /// <summary>
        /// Return the converted component selector.
        /// </summary>
        /// <param name="compSelValues">Values of component selector.</param>
        /*
        private uint ConvertComponentSelector( texture_info_comp_selValue[] compSelValues )
        {
            uint iResult = 0;
            if ( compSelValues==null )
                return 0;

            foreach ( texture_info_comp_selValue val in compSelValues )
            {
                switch ( val )
                {
                    case texture_info_comp_selValue.r :
                        iResult = (iResult << 8) | 0x00000000;
                        break;

                    case texture_info_comp_selValue.g :
                        iResult = (iResult << 8) | 0x00000001;
                        break;

                    case texture_info_comp_selValue.b :
                        iResult = (iResult << 8) | 0x00000002;
                        break;

                    case texture_info_comp_selValue.a :
                        iResult = (iResult << 8) | 0x00000003;
                        break;

                    case texture_info_comp_selValue.Item0 :
                        iResult = (iResult << 8) | 0x00000004;
                        break;

                    case texture_info_comp_selValue.Item1 :
                        iResult = (iResult << 8) | 0x00000005;
                        break;
                }
            }

            return iResult;
        }
        */


        /// <summary>
        /// Determine if the image of the given image format is already gamma corrected.
        /// </summary>
        /// <param name="imageFormat">The image format.</param>
        /// <returns>True if gamma corrected.</returns>
        public bool IsGammaCorrectedImage( string imageFormat )
        {
            if ( m_invGammaCorrImgFormatCRCs==null )
                return false;

            uint iFormatCRC = TheApp.CRC32Helper.ComputeCRC32Str( imageFormat );

            foreach ( uint iNeededFormat in m_invGammaCorrImgFormatCRCs )
            {
                if ( iFormatCRC==iNeededFormat )
                    return true;
            }

            return false;
        }

        /// <summary>
        /// 編集モードとテクスチャタイプが一致しているかの結果です
        /// </summary>
        public enum TextureTypeCheckResult
        {
            /// <summary>
            /// 編集モードとテクスチャタイプが一致しています
            /// </summary>
            OK,

            /// <summary>
            /// 編集モードはリニアではありませんがテクスチャがリニアです
            /// </summary>
            LinearTexture,

            /// <summary>
            /// 編集モードはリニアですがテクスチャがリニアではありません
            /// </summary>
            NonlinearTexture,

            /// <summary>
            /// The texture to be checked has not been loaded.
            /// </summary>
            TextureNotLoaded
        };

        /// <summary>
        /// Check if the texture linear flags matches the application linear edit mode settings.
        ///
        /// This method only check for textures that are already being loaded to cache.
        /// If the given file path is not yet loaded, the result will be
        /// TextureTypeCheckResult.TextureNotLoaded
        /// </summary>
        /// <param name="filePath">The file path of the texture.</param>
        /// <param name="bShowDialog">
        /// True to show the warning dialog if the linear mode is different from
        /// the application settings.
        /// </param>
        /// <returns>The check result.</returns>
        public TextureTypeCheckResult CheckTextureLinearEditMode( string filePath,
                                                                  bool bShowDialog )
        {
            TextureData data = FindTextureData( filePath );
            if ( data==null )
                return TextureTypeCheckResult.TextureNotLoaded;

            return CheckTextureLinearEditMode( data.TextureInfo,
                                               bShowDialog );
        }

        /// <summary>
        /// Check if the texture linear flags matches the application linear edit mode settings.
        ///
        /// This method only check for textures that are already being loaded to cache.
        /// If the given file path is not yet loaded, the result will be
        /// TextureTypeCheckResult.TextureNotLoaded
        /// </summary>
        /// <param name="iPathCRC">
        /// The CRC32 hash code of the (lower cased) texture file path.
        /// </param>
        /// <param name="bShowDialog">
        /// True to show the warning dialog if the linear mode is different from
        /// the application settings.
        /// </param>
        /// <returns>The check result.</returns>
        public TextureTypeCheckResult CheckTextureLinearEditMode( uint iPathCRC,
                                                                  bool bShowDialog )
        {
            TextureData data = FindTextureData( iPathCRC );
            if ( data==null )
                return TextureTypeCheckResult.TextureNotLoaded;

            return CheckTextureLinearEditMode( data.TextureInfo,
                                               bShowDialog );
        }

        /// <summary>
        /// Check if the texture linear flags matches the application linear edit mode settings.
        /// </summary>
        /// <param name="texInfo">The texture information.</param>
        /// <param name="bShowDialog">
        /// True to show the warning dialog if the linear mode is different from
        /// the application settings.
        /// </param>
        /// <returns>The check result.</returns>
        public TextureTypeCheckResult CheckTextureLinearEditMode( TextureInfo texInfo,
                                                                  bool bShowDialog )
        {
            if ( texInfo==null )
                return TextureTypeCheckResult.OK;

            // Is any of the linear flag of the texture true?
            bool bLinearTexture = texInfo.Linear != null && texInfo.Linear.Any(x => x);

            // Is the texture any of the srgb pixel formats?
            if ( TheApp.TextureManager.IsGammaCorrectedImage(texInfo.NativeDataFormat)==true )
                bLinearTexture = true;

            TextureTypeCheckResult result = TextureTypeCheckResult.OK;

            // Is the linear edit mode settings different from the texture?
            if ( bLinearTexture!=TheApp.IsGammaCorrectionEnabled )
            {
                string msg;
                if ( bLinearTexture==true )
                {
                    msg = res.Strings.WARNING_TEXTURE_TYPE_MISMATCH + "\n" + res.Strings.WARNING_TEXTURE_LINEAR;
                    result = TextureTypeCheckResult.LinearTexture;
                }
                else
                {
                    msg = res.Strings.WARNING_TEXTURE_TYPE_MISMATCH + "\n" + res.Strings.WARNING_TEXTURE_NONLINEAR;
                    result = TextureTypeCheckResult.NonlinearTexture;
                }

                if ( TheApp.ConsoleMode==false ||
                     TheApp.EnableConsoleModeWarningDialogs==true )
                {
                    if (bShowDialog)
                    {
                        /*
                        using (OptionalMessageBoxContext context = new OptionalMessageBoxContext("LinearEditTextureWarning"))
                        {
                            UIMessageBox.Show(msg,
                                               App.Controls.UIMessageBoxButtons.OK,
                                               MessageBoxIcon.Warning,
                                               MessageBoxDefaultButton.Button1);
                        }
                        */
                    }
                }
                else
                {
                    TheApp.OutputLogMsg( NWCore.LogLevels.Warning, msg );
                }
            }

            return result;
        }

        #endregion

        #region Events

        /// <summary>
        /// Delegation for document event handler.
        /// </summary>
        /// <param name="document">The document for the event.</param>
        public delegate void TextureEventHandler(TextureInfo textureInfo);

        /// <summary>
        /// Event triggered when a texture is loaded being created.
        /// </summary>
        public event TextureEventHandler TextureLoaded = null;

        private void NotifyTextureLoaded(TextureInfo texInfo)
        {
            var handler = TextureLoaded;
            if (handler != null) handler(texInfo);
        }

        #endregion

        #region Member variables

        //private TexUtils.Converter m_texConverter   = null;
        private bool               m_bCreateBitmaps = false;
        private bool               m_bCheckModTime  = false;

        private bool               m_bShowLoadTexWarningMsgBox = false;

        private long               m_iTotalCacheSize = 0;
        private long               m_iMaxCacheSize   = -1; // No limitations by default.

        static private object s_SyncObject = new object();

        //static private List<CustomFileSystemWatcher> s_WatcherList = new List<CustomFileSystemWatcher>();

        private Dictionary<uint, TextureData> m_textures =
            new Dictionary<uint, TextureData>();

        private LinkedList<TextureData> m_accessOrderCacheList =
            new LinkedList<TextureData>();

        private uint[] m_invGammaCorrImgFormatCRCs = null;

        //private Logger logger = null;

        #endregion
    }
}
