﻿// ========================================================================
// <copyright file="SerializableNameValueGroup.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

//#define DEBUG_NAME_VALUE_GROUP

using System;
using System.Collections.Generic;
using System.Diagnostics;

using NWCore.DataModel;
using NWCore.Serializer;

namespace App.Data
{
    /// <summary>
    /// Class for storing serializable name / value pairs.
    /// This class can be used as a general purpose storage
    /// for copy / paste.
    /// </summary>
    [Serializable]
    public class SerializableNameValueGroup
    {
        #region Class for value information

        /// <summary>
        /// Value information.
        /// </summary>
        [Serializable]
        private class ValueInfo
        {
            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="value">Value.</param>
            public ValueInfo( string name,
                              object value )
            {
                #if DEBUG_NAME_VALUE_GROUP
                this.Name = name;
                #endif

                this.Value = value;
                if ( value!=null )
                    this.ValueType = value.GetType();
            }


            /// <summary>
            /// Constructor for manually assign the value type.
            /// </summary>
            /// <param name="value">Value.</param>
            /// <param name="valueType">Value type.</param>
            public ValueInfo( string name,
                              object value,
                              Type valueType )
            {
                #if DEBUG_NAME_VALUE_GROUP
                this.Name = name;
                #endif

                this.Value     = value;
                this.ValueType = valueType;
            }

            #if DEBUG_NAME_VALUE_GROUP
            public string Name      { get; private set; }
            #endif

            public object Value     { get; private set; }
            public Type   ValueType { get; private set; }
        }

        #endregion

        #region Constructor

        /// <summary>
        /// Default constructor.
        /// </summary>
        public SerializableNameValueGroup()
        {
        }

        #endregion

        #region Properties

        /// <summary>
        /// Get or set the flag indicating whether should assert
        /// while not being able to access the data.
        /// </summary>
        public static bool ShouldAssert
        {
            get { return s_bShouldAssert; }
            set { s_bShouldAssert = value; }
        }


        /// <summary>Get the number of elements the group has.</summary>
        public int Count { get { return m_dictionary.Count; } }

        #endregion

        #region Add / get value

        /// <summary>
        /// Check if a value with the given name exists.
        /// </summary>
        /// <param name="name">The name of the value.</param>
        /// <returns>True if the value exists.</returns>
        public bool Contains( string name )
        {
            // Compute CRC32 hash code of the name to speed up the search.
            uint iNameCRC = TheApp.CRC32Helper.ComputeCRC32Str( name );

            // Try to get the value.
            return m_dictionary.ContainsKey( iNameCRC );
        }


        /// <summary>
        /// Get the value with the given name.
        /// </summary>
        /// <param name="name">Name of the value to get.</param>
        /// <returns>
        /// The value found with the given name, or null if the
        /// given name is not found.
        /// </returns>
        public object Get( string name )
        {
            // Compute CRC32 hash code of the name to speed up the search.
            uint iNameCRC = TheApp.CRC32Helper.ComputeCRC32Str( name );

            // Try to get the value.
            ValueInfo info;
            bool bResult = m_dictionary.TryGetValue( iNameCRC, out info );
            if ( bResult==false )
            {
                ShowWarning( "The requested name/value pair does not exist." );
                return null;
            }

            if ( info.ValueType!=null &&
                 info.ValueType.GetInterface("NintendoWare.ToolDevelopmentKit.IVector")!=null )
            {
                float[] array = info.Value as float[];
                if ( array.Length==1 )
                    return new NintendoWare.ToolDevelopmentKit.Vector1( array );
                else if ( array.Length==2 )
                    return new NintendoWare.ToolDevelopmentKit.Vector2( array );
                else if ( array.Length==3 )
                    return new NintendoWare.ToolDevelopmentKit.Vector3( array );
                else if ( array.Length==4 )
                    return new NintendoWare.ToolDevelopmentKit.Vector4( array );
            }
            else if ( info.ValueType!=null &&
                      info.ValueType.GetInterface("NWCore.DataModel.IVectori")!=null )
            {
                int[] array = info.Value as int[];
                if ( array.Length==1 )
                    return new NWCore.DataModel.Vector1i( array );
                else if ( array.Length==2 )
                    return new NWCore.DataModel.Vector2i( array );
                else if ( array.Length==3 )
                    return new NWCore.DataModel.Vector3i( array );
                else if ( array.Length==4 )
                    return new NWCore.DataModel.Vector4i( array );
            }
            else if ( info.ValueType!=null &&
                      info.ValueType.Equals(typeof(NintendoWare.ToolDevelopmentKit.RgbaColor))==true )
            {
                float[] array = info.Value as float[];
                if ( array.Length==4 )
                {
                    return new NintendoWare.ToolDevelopmentKit.RgbaColor( array[0],
                                                                          array[1],
                                                                          array[2],
                                                                          array[3] );
                }
            }
            else if ( info.ValueType!=null &&
                      info.ValueType.Equals(typeof(System.Drawing.Color))==true )
            {
                byte[] array = info.Value as byte[];
                if ( array.Length==4 )
                {
                    return System.Drawing.Color.FromArgb( array[0],
                                                          array[1],
                                                          array[2],
                                                          array[3] );
                }
            }
            else if ( info.ValueType!=null &&
                      info.ValueType.Equals(typeof(EmitterAnimTableListXml))==true )
            {
                EmitterAnimTableListXml serializer = new EmitterAnimTableListXml();

                System.IO.MemoryStream stream = new System.IO.MemoryStream( info.Value as byte[] );
                System.Xml.XmlReader   reader = new System.Xml.XmlTextReader( stream );

                reader.MoveToElement();

                serializer.ReadXml( reader );

                stream.Close();

                if ( serializer!=null &&
                     serializer.AnimationList!=null &&
                     serializer.AnimationList.Animations!=null )
                {
                    return serializer.AnimationList.Animations.ToArray();
                }
                else
                {
                    return null;
                }
            }

            return info.Value;
        }


        /// <summary>
        /// [ DEPRECATED ] Get the value with the given name.
        /// </summary>
        /// <param name="name">Name of the value to get.</param>
        /// <returns>
        /// The value found with the given name, or null if the
        /// given name is not found.
        /// </returns>
        public object GetVector( string name )
        {
            return Get( name );
        }


        /// <summary>
        /// [ DEPRECATED ] Get the value with the given name.
        /// </summary>
        /// <param name="name">Name of the value to get.</param>
        /// <returns>
        /// The value found with the given name, or null if the
        /// given name is not found.
        /// </returns>
        public object GetColor( string name )
        {
            return Get( name );
        }


        /// <summary>
        /// Add a name/value pair to the group.
        /// </summary>
        /// <param name="name">Name.</param>
        /// <param name="value">Value.</param>
        /// <returns>True on success.</returns>
        public bool Add( string name,
                         object value )
        {
            // Compute CRC32 hash code of the name to speed up the search.
            uint iNameCRC = TheApp.CRC32Helper.ComputeCRC32Str( name );

            // Does the name already exist?
            if ( m_dictionary.ContainsKey(iNameCRC)==true )
            {
                ShowWarning( "SerializableNameValueGroup.AddVector : The name \"" +
                             name +
                             "\" already exists." );
                return false;
            }

            // Is the value serializable?
            if ( value!=null )
            {
                Type valueType = value.GetType();
                if ( value is NintendoWare.ToolDevelopmentKit.IVector )
                {
                    return AddVector( name, iNameCRC, value as NintendoWare.ToolDevelopmentKit.IVector );
                }
                else if ( value is NWCore.DataModel.IVectori )
                {
                    return AddVector( name, iNameCRC, value as NWCore.DataModel.IVectori );
                }
                else if ( value is NintendoWare.ToolDevelopmentKit.RgbaColor )
                {
                    return AddRgbaColor( name, iNameCRC, value as NintendoWare.ToolDevelopmentKit.RgbaColor );
                }
                else if ( value is System.Drawing.Color )
                {
                    return AddColor( name, iNameCRC, (System.Drawing.Color)value );
                }
                else if ( value is BaseKeyFrameList )
                {
                    return AddAnimationKeyFrameList( name, iNameCRC, value as BaseKeyFrameList );
                }

                if ( valueType.IsSerializable==false )
                {
                    ShowWarning( "SerializableNameValueGroup.Add : The value (" +
                                 valueType.ToString() +
                                 ") is not serializable." );
                    return false;
                }
            }

            // Add the name/value pair
            m_dictionary.Add( iNameCRC, new ValueInfo( name, value ) );

            return true;
        }


        /// <summary>
        /// Add a name/value pair to the group.
        /// </summary>
        /// <param name="name">Name.</param>
        /// <param name="vec">Vector of float values.</param>
        /// <returns>True on success.</returns>
        private bool AddVector( string name,
                                uint iNameCRC,
                                NintendoWare.ToolDevelopmentKit.IVector vec )
        {
            // Add the converted float array
            m_dictionary.Add( iNameCRC, new ValueInfo( name, vec.ToArray(), vec.GetType() ) );

            return true;
        }


        /// <summary>
        /// Add a name/value pair to the group.
        /// </summary>
        /// <param name="name">Name.</param>
        /// <param name="vec">Vector of integer values.</param>
        /// <returns>True on success.</returns>
        private bool AddVector( string name,
                                uint iNameCRC,
                                NWCore.DataModel.IVectori vec )
        {
            // Add the converted float array
            m_dictionary.Add( iNameCRC, new ValueInfo( name, vec.ToArray(), vec.GetType() ) );

            return true;
        }


        /// <summary>
        /// Add a name/value pair to the group.
        /// </summary>
        /// <param name="name">Name.</param>
        /// <param name="color">Color object to add.</param>
        /// <returns>True on success.</returns>
        private bool AddRgbaColor( string name,
                                   uint iNameCRC,
                                   NintendoWare.ToolDevelopmentKit.RgbaColor color )
        {
            // Store the color channels in a float array
            float[] array = new float[] { color.R,
                                          color.G,
                                          color.B,
                                          color.A };

            // Add the converted float array
            m_dictionary.Add( iNameCRC, new ValueInfo( name, array, color.GetType() ) );

            return true;
        }


        /// <summary>
        /// Add a name/value pair to the group.
        /// </summary>
        /// <param name="name">Name.</param>
        /// <param name="color">Color object to add.</param>
        /// <returns>True on success.</returns>
        private bool AddColor( string name,
                               uint iNameCRC,
                               System.Drawing.Color color )
        {
            // Store the color channels in a float array
            byte[] array = new byte[] { color.A,
                                        color.R,
                                        color.G,
                                        color.B };

            // Add the converted float array
            m_dictionary.Add( iNameCRC, new ValueInfo( name, array, color.GetType() ) );

            return true;
        }


        /// <summary>
        /// Add a name/value pair to the group.
        /// </summary>
        /// <param name="name">Name.</param>
        /// <param name="data">The animation key frame list to add.</param>
        /// <returns>True on success.</returns>
        private bool AddAnimationKeyFrameList( string name,
                                               uint iNameCRC,
                                               BaseKeyFrameList data )
        {
            AnimTableListXml animations =
                TheApp.AnimationTableManager.ConvertAnimation( data );

            System.IO.MemoryStream stream = new System.IO.MemoryStream();
            using ( System.Xml.XmlWriter writer = System.Xml.XmlWriter.Create( stream ) )
            {
                writer.WriteStartDocument( true );
                writer.WriteStartElement( "animations" );

                animations.WriteXml( writer );

                writer.WriteEndElement();
                writer.WriteEndDocument();
            }

            byte[] buffer = stream.GetBuffer();
            stream.Close();

            // Add the converted animations
            m_dictionary.Add( iNameCRC, new ValueInfo( name, buffer, typeof(EmitterAnimTableListXml) ) );

            return true;
        }


        #endregion

        #region Warning

        /// <summary>
        /// Show warning message.
        /// </summary>
        /// <param name="msg">The warning message.</param>
        private void ShowWarning( string msg )
        {
            if ( SerializableNameValueGroup.ShouldAssert==true )
                Debug.Assert( false, msg );
        }

        #endregion

        #region Member variables

        private static bool s_bShouldAssert = true;

        Dictionary<uint, ValueInfo> m_dictionary = new Dictionary<uint, ValueInfo>();

        #endregion
    }

    #region NameValueGroupAssertionBlock

    /// <summary>
    /// Block assertions from the SerializableNameValueGroup.
    /// Usage :
    ///     using ( NameValueGroupAssertionBlock block = new NameValueGroupAssertionBlock() )
    ///     {
    ///         // The assertions will be blocked within this block.
    ///     }
    /// </summary>
    public class NameValueGroupAssertionBlock : IDisposable
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public NameValueGroupAssertionBlock()
        {
            m_bOriginalValue = SerializableNameValueGroup.ShouldAssert;
            SerializableNameValueGroup.ShouldAssert = false;
        }


        /// <summary>
        /// Dispose.
        /// </summary>
        public void Dispose()
        {
            SerializableNameValueGroup.ShouldAssert = m_bOriginalValue;
        }


        private bool m_bOriginalValue = true;
    }

    #endregion
}
