﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Xml;

using EffectMaker.DataModel.DataModels;
using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModel.Serializer
{
    /// <summary>
    /// Manages data model serializers.
    /// </summary>
    public static class DataModelSerializerManager
    {
        /// <summary>Constant string representing the attribute name for data model version.</summary>
        private const string VersionAttributeName = "Version";

        /// <summary>Constant string representing the attribute name for use XmlDocSerializer flag.</summary>
        private const string UseXmlDocSerializerAttributeName = "UseXmlDocSerializer";

        /// <summary>The serializers.</summary>
        private static List<DataModelSerializerBase> serializers =
            new List<DataModelSerializerBase>();

        /// <summary>
        /// Add all the data model serializers in the assembly.
        /// </summary>
        /// <param name="assembly">The assembly.</param>
        /// <returns>The number of added data model serializers.</returns>
        public static int AddDataModelSerializers(Assembly assembly)
        {
            // Collect all the serializers.
            var serializerTypes =
                from t in assembly.GetTypes()
                where t.IsSubclassOf(typeof(DataModelSerializerBase)) == true
                select t;

            // Save the singleton instance of the serializers.
            int count = 0;
            foreach (Type type in serializerTypes)
            {
                var singletonInstanceProperty = type.GetProperty("Instance");
                var serializer = singletonInstanceProperty.GetValue(null) as DataModelSerializerBase;

                serializers.Add(serializer);
                ++count;
            }

            // Sort the serializers by their version.
            serializers.Sort((a, b) => a.Version.CompareTo(b.Version));

            return count;
        }

        /// <summary>
        /// Serialize the data model to the given stream.
        /// </summary>
        /// <typeparam name="T">The data model type.</typeparam>
        /// <param name="dataModel">The data model to serialize.</param>
        /// <param name="stream">The stream.</param>
        /// <returns>True on success.</returns>
        public static bool Serialize<T>(T dataModel, Stream stream) where T : DataModelBase
        {
            const string XmlNamespaceUri = "http://www.w3.org/2000/xmlns/";
            const string XsiNamespaceUri = "http://www.w3.org/2001/XMLSchema-instance";
            const string XsdNamespaceUri = "http://www.w3.org/2001/XMLSchema";

            // Get the latest serializer.
            var serializer = serializers[serializers.Count - 1];

            // Create the XmlDocument for serialization.
            XmlDocument doc = new XmlDocument();

            // Serialize the data model to XmlDocument.
            serializer.Serialize(dataModel, doc);
            if (doc.DocumentElement == null)
            {
                return false;
            }

            // Inject version and serializer flag to the root element.
            doc.DocumentElement.SetAttribute(VersionAttributeName, serializer.Version.ToString());
            doc.DocumentElement.SetAttribute(UseXmlDocSerializerAttributeName, bool.TrueString);

            // Add xsi and xsd namespaces to the root element.
            var xsiNamespaceDeclAttr = doc.CreateAttribute("xmlns", "xsi", XmlNamespaceUri);
            xsiNamespaceDeclAttr.Value = XsiNamespaceUri;

            var xsdNamespaceDeclAttr = doc.CreateAttribute("xmlns", "xsd", XmlNamespaceUri);
            xsdNamespaceDeclAttr.Value = XsdNamespaceUri;

            doc.DocumentElement.SetAttributeNode(xsiNamespaceDeclAttr);
            doc.DocumentElement.SetAttributeNode(xsdNamespaceDeclAttr);

            // Insert XML declaration.
            var xmldecl = doc.CreateXmlDeclaration("1.0", string.Empty, string.Empty);
            doc.InsertBefore(xmldecl, doc.DocumentElement);

            // Save the XML document to the stream.
            doc.Save(stream);

            return true;
        }

        /// <summary>
        /// Deserialize the specified data model type from the provided stream.
        /// The type should be of the latest version.
        /// </summary>
        /// <param name="stream">The stream.</param>
        /// <typeparam name="type">The data model type.</typeparam>
        /// <returns>The deserialized data model.</returns>
        public static DataModelBase Deserialize(Stream stream, Type type)
        {
            bool useXmlDocSerializer = true;

            // Load XML document from the loaded stream.
            var doc = new XmlDocument();
            doc.Load(stream);

            Version version;

            // Get the file version.
            string versionAttr = doc.DocumentElement.GetAttribute(VersionAttributeName);
            if (string.IsNullOrEmpty(versionAttr) == true)
            {
                // The version is not specified in the root element,
                // assume it's version 0.0.0.0.
                version = new Version(0, 0, 0, 0);
            }
            else
            {
                version = new Version(versionAttr);
            }

            // Get the flag indicating whether to use XmlDocSerializer or not.
            string useXmlDocSerializerAttr =
                doc.DocumentElement.GetAttribute(UseXmlDocSerializerAttributeName);
            if (string.IsNullOrEmpty(useXmlDocSerializerAttr) == true ||
                bool.Parse(useXmlDocSerializerAttr) != true)
            {
                useXmlDocSerializer = false;

                Logger.Log(LogLevels.Warning, "The data model is serialized with XmlSerializer and can only be deserialized with it.");
                Logger.Log(LogLevels.Warning, "The deserialization can be more than 100 times faster if XmlDocSerializer is used.");
                Logger.Log(LogLevels.Warning, "Make sure you save the loaded data model again so it can be loaded faster next time.");
            }

            // Find the serializer for the version and the data model type to deserialize.
            int index = -1;
            for (int i = serializers.Count - 1; i >= 0; --i)
            {
                // Find the serializer for the version.
                var serializer = serializers[i];
                if (serializer.Version == version)
                {
                    index = i;
                    break;
                }

                // Down grade the type.
                type = serializer.FindPreviousType(type);
            }

            if (type == null || index < 0)
            {
                // Unable to find the serializer for the version.
                throw new IOException("Version mismatch.");
            }

            // Deserialize!
            DataModelBase dataModel = null;
            if (useXmlDocSerializer == true)
            {
                dataModel = serializers[index].Deserialize(type, doc);
            }
            else
            {
                // Create the reader so we can deserialize the XML document.
                var reader = new XmlNodeReader(doc.DocumentElement);
                dataModel = serializers[index].DeserializeWithXmlSerializer(type, reader);
            }

            if (dataModel == null)
            {
                // Failed loading data model with the serializer,
                // maybe the serializer is not for this file?
                // This could happen when the file has no version,
                // e.g. the file from the old effect maker.
                throw new IOException("Unable to find a serializer for this version of data model.");
            }

            // Advance to the next serializer.
            ++index;

            // Upgrade the data model until the latest version.
            for (; index < serializers.Count; ++index)
            {
                var serializer = serializers[index];
                dataModel = serializer.UpgradeDataModel(dataModel);
            }

            return dataModel;
        }

        /// <summary>
        /// Deserialize the specified data model type from the provided stream.
        /// The type should be of the latest version.
        /// </summary>
        /// <typeparam name="T">The data model type.</typeparam>
        /// <param name="stream">The stream.</param>
        /// <returns>The deserialized data model.</returns>
        public static T Deserialize<T>(Stream stream) where T : DataModelBase
        {
            return (T)Deserialize(stream, typeof(T));
        }

        /// <summary>
        /// Serialize the data model to the given stream.
        /// </summary>
        /// <typeparam name="T">The data model type.</typeparam>
        /// <param name="dataModel">The data model to serialize.</param>
        /// <param name="stream">The stream.</param>
        /// <returns>True on success.</returns>
        public static bool SerializeWithXmlSerializer<T>(T dataModel, Stream stream) where T : DataModelBase
        {
            // Get the latest serializer.
            var serializer = serializers[serializers.Count - 1];

            // Create the XmlDocument for serialization.
            XmlDocument doc = new XmlDocument();
            var nav = doc.CreateNavigator();

            // Serialize the data model.
            using (XmlWriter writer = nav.AppendChild())
            {
                // Serialize the data model to XmlDocument.
                serializer.SerializeWithXmlSerializer(dataModel, writer);
            }

            // Inject version to the root element.
            doc.DocumentElement.SetAttribute("Version", serializer.Version.ToString());

            // Save the XML document to the stream.
            doc.Save(stream);

            return true;
        }

        /// <summary>
        /// Deserialize the specified data model type from the provided stream.
        /// The type should be of the latest version.
        /// </summary>
        /// <typeparam name="T">The data model type.</typeparam>
        /// <param name="stream">The stream.</param>
        /// <returns>The deserialized data model.</returns>
        public static T DeserializeWithXmlSerializer<T>(Stream stream) where T : DataModelBase
        {
            // Load XML document from the loaded stream.
            var doc = new XmlDocument();
            doc.Load(stream);

            Version version;

            // Get the file version.
            string versionAttr = doc.DocumentElement.GetAttribute("Version");
            if (string.IsNullOrEmpty(versionAttr) == true)
            {
                // The version is not specified in the root element,
                // assume it's version 0.0.0.0.
                version = new Version(0, 0, 0, 0);
            }
            else
            {
                version = new Version(versionAttr);
            }

            // Find the serializer for the version and the data model type to deserialize.
            Type type = typeof(T);
            int index = -1;
            for (int i = serializers.Count - 1; i >= 0; --i)
            {
                // Find the serializer for the version.
                var serializer = serializers[i];
                if (serializer.Version == version)
                {
                    index = i;
                    break;
                }

                // Down grade the type.
                type = serializer.FindPreviousType(type);
            }

            if (type == null || index < 0)
            {
                // Unable to find the serializer for the version.
                throw new IOException("Unable to find a serializer for this version of data model.");
            }

            // Create the reader so we can deserialize the XML document.
            var reader = new XmlNodeReader(doc.DocumentElement);

            // Deserialize!
            DataModelBase dataModel = serializers[index].DeserializeWithXmlSerializer(type, reader);
            if (dataModel == null)
            {
                // Failed loading data model with the serializer,
                // maybe the serializer is not for this file?
                // This could happen when the file has no version,
                // e.g. the file from the old effect maker.
                throw new IOException("Unable to find a serializer for this version of data model.");
            }

            // Advance to the next serializer.
            ++index;

            // Upgrade the data model until the latest version.
            for (; index < serializers.Count; ++index)
            {
                var serializer = serializers[index];
                dataModel = serializer.UpgradeDataModel(dataModel);
            }

            return (T)dataModel;
        }
    }
}
