﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using EffectMaker.DataModel.DataModels;

using EffectMaker.Foundation.Collections.Generic;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.DataModel.Extensions
{
    /// <summary>
    /// Extensions for data models.
    /// </summary>
    public static class DataModelExtension
    {
        /// <summary>
        /// Static constructor.
        /// </summary>
        static DataModelExtension()
        {
            ShouldCopyInstanceForSetter = false;
        }

        /// <summary>
        /// Get or set the flag indicating whether to copy instance or only
        /// set reference in the set methods.
        /// </summary>
        public static bool ShouldCopyInstanceForSetter { get; set; }

        /// <summary>
        /// Helper method for a property getter which returns a data model.
        /// When the backing field is null, this method create a new instance
        /// of the data model to the backing field.
        /// </summary>
        /// <typeparam name="T">The type of the returning data model.</typeparam>
        /// <param name="self">The extending data model.</param>
        /// <param name="field">The backing field of the property.</param>
        /// <param name="allowNull">False to create instance to the backing field when it's null.</param>
        /// <returns>The data model stored in the backing field.</returns>
        public static T GetDataModelFromField<T>(
            this DataModelBase self,
            ref T field,
            bool allowNull) where T : DataModelBase
        {
            if (field == null && allowNull == false)
            {
                field = (T)Activator.CreateInstance(typeof(T));
                field.Parent = self;
                self.Children.Add(field);
            }

            return field;
        }

        /// <summary>
        /// Helper method for setting the data model to the given backing field.
        /// </summary>
        /// <typeparam name="T">The type of the data model.</typeparam>
        /// <param name="self">The extending data model.</param>
        /// <param name="field">The backing field.</param>
        /// <param name="value">The value to set.</param>
        public static void SetDataModelToField<T>(
            this DataModelBase self,
            ref T field,
            T value) where T : DataModelBase
        {
            if (object.ReferenceEquals(field, value) == true)
            {
                return;
            }

            if (field != null)
            {
                self.Children.Remove(field);
            }

            if (value == null)
            {
                field = null;
            }
            else
            {
                if (field == null ||
                    field.GetType() != value.GetType())
                {
                    if (ShouldCopyInstanceForSetter == true && value != null)
                    {
                        field = (T)value.Clone();
                    }
                    else
                    {
                        field = (T)value;
                    }

                    field.Parent = self;
                }
                else
                {
                    field.Set(value);
                }
            }

            if (field != null && self.Children.IndexOf(field) < 0)
            {
                self.Children.Add(field);
            }
        }

        /// <summary>
        /// Helper method for a property getter which returns an object.
        /// When the backing field is null, this method create a new instance
        /// of the object to the backing field.
        /// </summary>
        /// <typeparam name="T">The type of the returning object.</typeparam>
        /// <param name="self">The extending data model.</param>
        /// <param name="field">The backing field of the property.</param>
        /// <param name="allowNull">False to create instance to the backing field when it's null.</param>
        /// <returns>The object stored in the backing field.</returns>
        public static T GetObjectFromField<T>(
            this DataModelBase self,
            ref T field,
            bool allowNull)
        {
            if (field == null && allowNull == false)
            {
                field = (T)Activator.CreateInstance(typeof(T));
            }

            return field;
        }

        /// <summary>
        /// Helper method for setting the ISettable to the given backing field.
        /// </summary>
        /// <typeparam name="T">The type of the data model.</typeparam>
        /// <param name="self">The extending data model.</param>
        /// <param name="field">The backing field.</param>
        /// <param name="value">The value to set.</param>
        public static void SetISettableToField<T>(
            this DataModelBase self,
            ref T field,
            T value) where T : ISettable
        {
            if (object.ReferenceEquals(field, value) == true)
            {
                return;
            }

            if (value == null)
            {
                field = default(T);
            }
            else
            {
                if (field == null)
                {
                    field = (T)Activator.CreateInstance(typeof(T));
                }

                field.Set(value);
            }
        }

        /// <summary>
        /// Helper method for a property getter which returns a data model list.
        /// When the backing field is null, return an empty list.
        /// </summary>
        /// <typeparam name="T">The type of the returning data model.</typeparam>
        /// <param name="self">The extending data model.</param>
        /// <param name="field">The backing field of the property.</param>
        /// <returns>The data model list stored in the backing field.</returns>
        public static List<T> GetDataModelListFromField<T>(
            this DataModelBase self,
            ref List<T> field) where T : DataModelBase
        {
            if (field == null)
            {
                field = new List<T>();
            }

            return field;
        }

        /// <summary>
        /// Helper method for setting the data model list to the given backing field.
        /// </summary>
        /// <typeparam name="T">The type of the data model.</typeparam>
        /// <param name="self">The extending data model.</param>
        /// <param name="field">The backing field.</param>
        /// <param name="value">The value to set.</param>
        public static void SetDataModelListToField<T>(
            this DataModelBase self,
            ref List<T> field,
            List<T> value) where T : DataModelBase
        {
            if (object.ReferenceEquals(field, value) == true)
            {
                return;
            }

            // If the target list (the field) is not created yet, create it.
            if (field == null)
            {
                if (value != null)
                {
                    field = new List<T>(value.Count);
                }
                else
                {
                    return;
                }
            }

            // First make both lists have the same number of items.
            if (field.Count < value.Count)
            {
                field.AddRange(Enumerable.Repeat(default(T), value.Count - field.Count));
            }
            else if (field.Count > value.Count)
            {
                for (int i = field.Count - 1; i >= value.Count; --i)
                {
                    self.Children.Remove(field[i]);
                    field.RemoveAt(i);
                }
            }

            // Now copy the items.
            for (int i = 0; i < value.Count; ++i)
            {
                bool shouldAddToChildren = false;

                T newDataModel = value[i];
                T oldDataModel = field[i];
                if (newDataModel != null && oldDataModel == null)
                {
                    // The target item is null, set or clone the source to the target.
                    if (ShouldCopyInstanceForSetter == true)
                    {
                        field[i] = (T)newDataModel.Clone();
                    }
                    else
                    {
                        // In case newDataModel is already in the child list, remove it first.
                        self.Children.Remove(newDataModel);
                        field[i] = newDataModel;
                    }

                    // The new item should be added to the child list.
                    shouldAddToChildren = true;
                }
                else if (newDataModel == null)
                {
                    // The source item is null, just remove the item from the target list.
                    if (oldDataModel != null)
                    {
                        self.Children.Remove(oldDataModel);
                    }

                    field[i] = null;
                }
                else
                {
                    // Both source and target items aren't null, check the type to determine
                    // whether to copy (same type) or clone the source item (different types).
                    if (oldDataModel.GetType() == newDataModel.GetType())
                    {
                        if (ShouldCopyInstanceForSetter == true)
                        {
                            oldDataModel.Set(newDataModel);
                        }
                        else
                        {
                            // In case newDataModel is already in the child list, remove it first.
                            self.Children.Remove(newDataModel);
                            self.Children.Remove(oldDataModel);
                            field[i] = newDataModel;
                            shouldAddToChildren = true;
                        }
                    }
                    else
                    {
                        self.Children.Remove(oldDataModel);
                        if (ShouldCopyInstanceForSetter == true)
                        {
                            field[i] = (T)newDataModel.Clone();
                        }
                        else
                        {
                            // In case newDataModel is already in the child list, remove it first.
                            self.Children.Remove(newDataModel);
                            field[i] = newDataModel;
                        }

                        shouldAddToChildren = true;
                    }
                }

                if (shouldAddToChildren == true && field[i] != null)
                {
                    T currDataModel = field[i];
                    currDataModel.Parent = self;
                    if (self.Children.IndexOf(currDataModel) < 0)
                    {
                        self.Children.Add(currDataModel);
                    }
                }
            }
        }

        /// <summary>
        /// Helper method for a property getter which returns a data model array collection.
        /// When the backing field is null, return an empty list.
        /// </summary>
        /// <typeparam name="T">The type of the returning data model.</typeparam>
        /// <param name="self">The extending data model.</param>
        /// <param name="itemCount">The item count in the array collection.</param>
        /// <param name="field">The backing field of the property.</param>
        /// <returns>The data model list stored in the backing field.</returns>
        public static ArrayCollection<T> GetDataModelArrayCollectionFromField<T>(
            this DataModelBase self,
            int itemCount,
            ref ArrayCollection<T> field) where T : DataModelBase
        {
            if (field == null)
            {
                field = new ArrayCollection<T>(itemCount);
            }

            return field;
        }

        /// <summary>
        /// Helper method for setting the data model list to the given backing field.
        /// </summary>
        /// <typeparam name="T">The type of the data model.</typeparam>
        /// <param name="self">The extending data model.</param>
        /// <param name="itemCount">The item count in the array collection.</param>
        /// <param name="field">The backing field.</param>
        /// <param name="value">The value to set.</param>
        public static void SetDataModelArrayCollectionToField<T>(
            this DataModelBase self,
            int itemCount,
            ref ArrayCollection<T> field,
            ArrayCollection<T> value) where T : DataModelBase
        {
            if (object.ReferenceEquals(field, value) == true)
            {
                return;
            }

            if (field == null)
            {
                field = new ArrayCollection<T>(itemCount);
            }

            for (int i = 0; i < itemCount && i < value.Count; ++i)
            {
                bool shouldAddToChildren = false;

                T newDataModel = value[i];
                T oldDataModel = field[i];
                if (newDataModel != null && oldDataModel == null)
                {
                    // The target item is null, set or clone the source to the target.
                    if (ShouldCopyInstanceForSetter == true)
                    {
                        field[i] = (T)newDataModel.Clone();
                    }
                    else
                    {
                        // In case newDataModel is already in the child list, remove it first.
                        self.Children.Remove(newDataModel);
                        field[i] = newDataModel;
                    }

                    // The new item should be added to the child list.
                    shouldAddToChildren = true;
                }
                else if (newDataModel == null)
                {
                    // The source item is null, just remove the item from the target array.
                    if (oldDataModel != null)
                    {
                        self.Children.Remove(oldDataModel);
                    }

                    field[i] = null;
                }
                else
                {
                    // Both source and target items aren't null, check the type to determine
                    // whether to copy (same type) or clone the source item (different types).
                    if (oldDataModel.GetType() == newDataModel.GetType())
                    {
                        if (ShouldCopyInstanceForSetter == true)
                        {
                            oldDataModel.Set(newDataModel);
                        }
                        else
                        {
                            // In case newDataModel is already in the child list, remove it first.
                            self.Children.Remove(newDataModel);
                            self.Children.Remove(oldDataModel);
                            field[i] = newDataModel;
                            shouldAddToChildren = true;
                        }
                    }
                    else
                    {
                        self.Children.Remove(oldDataModel);
                        if (ShouldCopyInstanceForSetter == true)
                        {
                            field[i] = (T)newDataModel.Clone();
                        }
                        else
                        {
                            // In case newDataModel is already in the child list, remove it first.
                            self.Children.Remove(newDataModel);
                            field[i] = newDataModel;
                        }

                        shouldAddToChildren = true;
                    }
                }

                if (shouldAddToChildren == true && field[i] != null)
                {
                    T currDataModel = field[i];
                    currDataModel.Parent = self;
                    if (self.Children.IndexOf(currDataModel) < 0)
                    {
                        self.Children.Add(currDataModel);
                    }
                }
            }
        }

        /// <summary>
        /// Helper method for a property getter which enumerates data models.
        /// When the backing field is null, return an empty enumerable.
        /// </summary>
        /// <typeparam name="T">The type of the returning data model.</typeparam>
        /// <param name="self">The extending data model.</param>
        /// <param name="field">The backing field of the property.</param>
        /// <returns>The data model enumerable stored in the backing field.</returns>
        public static IEnumerable<T> GetDataModelEnumerableFromField<T>(
            this DataModelBase self,
            ref IEnumerable<T> field) where T : DataModelBase
        {
            if (field == null)
            {
                return Enumerable.Empty<T>();
            }
            else
            {
                return field;
            }
        }

        /// <summary>
        /// Helper method for a property getter which returns a list.
        /// When the backing field is null, return an empty list.
        /// </summary>
        /// <typeparam name="T">The type of the the list elements.</typeparam>
        /// <param name="self">The extending data model.</param>
        /// <param name="field">The backing field of the property.</param>
        /// <returns>The list stored in the backing field.</returns>
        public static List<T> GetValueListFromField<T>(
            this DataModelBase self,
            ref List<T> field)
        {
            if (field == null)
            {
                field = new List<T>();
            }

            return field;
        }

        /// <summary>
        /// Helper method for setting the list to the given backing field.
        /// </summary>
        /// <typeparam name="T">The type of the list elements.</typeparam>
        /// <param name="self">The extending data model.</param>
        /// <param name="field">The backing field.</param>
        /// <param name="value">The value to set.</param>
        public static void SetValueListToField<T>(
            this DataModelBase self,
            ref List<T> field,
            List<T> value)
        {
            if (object.ReferenceEquals(field, value) == true)
            {
                return;
            }

            if (field == null)
            {
                field = new List<T>();
            }

            field.Clear();
            field.AddRange(value);
        }

        /// <summary>
        /// Helper method for setting the list of ICloneable to the given backing field.
        /// </summary>
        /// <typeparam name="T">The type of the list elements.</typeparam>
        /// <param name="self">The extending data model.</param>
        /// <param name="field">The backing field.</param>
        /// <param name="value">The value to set.</param>
        public static void SetICloneableListToField<T>(
            this DataModelBase self,
            ref List<T> field,
            List<T> value) where T : ICloneable
        {
            if (object.ReferenceEquals(field, value) == true)
            {
                return;
            }

            if (field == null)
            {
                field = new List<T>();
            }

            field.Clear();

            T defaultValue = default(T);
            if (ShouldCopyInstanceForSetter == true)
            {
                field.AddRange(value.Select(
                    v => object.Equals(v, defaultValue) ? defaultValue : (T)v.Clone()));
            }
            else
            {
                field.AddRange(value.Cast<T>());
            }
        }
    }

    /// <summary>
    /// Helper class to temporarily set the copy instance flag on
    /// for the data model property setters.
    /// (the property setters only copy the reference by default)
    /// </summary>
    public class EnableDataModelCloneSetter : IDisposable
    {
        /// <summary>The original value of DataModelExtension.ShouldCopyInstanceForSetter flag.</summary>
        private bool originalFlag;

        /// <summary>
        /// Constructor.
        /// </summary>
        public EnableDataModelCloneSetter()
        {
            this.originalFlag = DataModelExtension.ShouldCopyInstanceForSetter;
            DataModelExtension.ShouldCopyInstanceForSetter = true;
        }

        /// <summary>
        /// Dispose this object.
        /// </summary>
        public void Dispose()
        {
            DataModelExtension.ShouldCopyInstanceForSetter = this.originalFlag;
        }
    }
}
