﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Threading;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.HookCommand;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Manager;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.Protocol;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Extensions;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.UILogic.Properties;
using EffectMaker.UILogic.ViewModels.IO;
using EffectMaker.UILogic.Commands;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Root view model.
    /// </summary>
    public class WorkspaceRootViewModel : HierarchyViewModel
    {
        /// <summary>
        /// Backing field for the SelectedNodeViewModel property.
        /// </summary>
        private ViewModelBase selectedNodeViewModel;

        /// <summary>
        /// ワークスペースビューモデルです。
        /// </summary>
        private WorkspaceViewModel workspaceViewModel;

        /// <summary>
        /// ビューアビューモデルです。
        /// </summary>
        private ViewerViewModel viewerViewModel;

        /// <summary>
        /// ワークスペースのストライプメニューで最後に追加されたサブノードのタイプ
        /// </summary>
        private SubNodeTypes lastAddNode = SubNodeTypes.RandomFe1;

        /// <summary>
        /// 複数選択したノードを保持するコレクションです。
        /// </summary>
        private readonly List<WorkspaceNodeViewModelBase> multiSelectedNodes = null;

        /// <summary>
        /// Initializes the WorkspaceRootViewModel instance.
        /// </summary>
        public WorkspaceRootViewModel()
            : base(null, new NullDataModel())
        {
            WorkspaceRootViewModel.Instance = this;

            this.Dialogs = new MainDialogs();

            this.Controller = new WorkspaceController(this);

            this.OnCloseAllExecutable = new AnonymousExecutable(this.OnCloseAll);
            this.OnCreateNewEmitterSetExecutable = new AnonymousExecutable(this.OnCreateNewEmitterSet);
            this.OnCreateNewEmitterExecutable = new AnonymousExecutable(this.OnCreateNewEmitter);
            this.OnCreateNewPreviewExecutable = new AnonymousExecutable(this.OnCreateNewPreview);
            this.OnRemoveNodeExecutable = new AnonymousExecutable(this.OnRemoveNode);
            this.OnFileOpenExecutable = new AnonymousExecutable(this.OnFileOpen);
            this.OnFileSaveExecutable = new AnonymousExecutable(this.OnFileSave);
            this.OnFileSaveAllExecutable = new AnonymousExecutable(this.OnFileSaveAll);
            this.OnFileSaveAllAsExecutable = new AnonymousExecutable(this.OnFileSaveAllAs);
            this.OnWorkspaceSaveExecutable = new AnonymousExecutable(this.OnWorkspaceSave);
            this.OnWorkspaceOpenExecutable = new AnonymousExecutable(this.OnWorkspaceOpen);
            this.OnLinkEmitterSetExecutable = new AnonymousExecutable(this.OnLinkEmitterSet);
            this.OnUnlinkEmitterSetExecutable = new AnonymousExecutable(this.OnUnlinkEmitterSet);
            this.OnUnlinkAllEmitterSetsExecutable = new AnonymousExecutable(this.OnUnlinkAllEmitterSets);

            this.multiSelectedNodes = new List<WorkspaceNodeViewModelBase>();
        }

        /// <summary>
        /// ワークスペースビューモデルが変更されたときに発生するイベントです。
        /// </summary>
        public event EventHandler<WorkspaceViewModel> WorkspaceViewModelChanged;

        /// <summary>
        /// ビューアビューモデルが変更されたときに発生するイベントです。
        /// </summary>
        public event EventHandler<ViewerViewModel> ViewerViewModelChanged;

        /// <summary>
        /// WorkspaceRootViewModelのインスタンスを取得します。
        /// </summary>
        public static WorkspaceRootViewModel Instance { get; private set; }

        /// <summary>
        /// コピペのステートアップデートをブロックします(テキストボックス対策)
        /// </summary>
        public bool BlockingUpdateCopyPasteState { get; set; }

        /// <summary>
        /// 終了処理中に立てるフラグ
        /// </summary>
        public bool IsExiting { get; set; }

        /// <summary>
        /// ワークスペースビューモデルを取得します。
        /// </summary>
        public WorkspaceViewModel WorkspaceViewModel
        {
            get { return this.workspaceViewModel; }

            private set
            {
                if (value != this.workspaceViewModel)
                {
                    this.workspaceViewModel = value;

                    this.WorkspaceViewModelChanged?.Invoke(this, this.workspaceViewModel);
                }
            }
        }

        /// <summary>
        /// ビューアビューモデルを取得または設定します。
        /// </summary>
        public ViewerViewModel ViewerViewModel
        {
            get
            {
                return this.viewerViewModel;
            }

            set
            {
                if (value != this.viewerViewModel)
                {
                    if (this.viewerViewModel != null)
                    {
                        this.Children.Remove(this.viewerViewModel);
                    }

                    this.viewerViewModel = value;
                    this.Children.Add(this.viewerViewModel);

                    this.ViewerViewModelChanged?.Invoke(this, this.viewerViewModel);
                }
            }
        }

        /// <summary>
        /// Gets the controller.
        /// The controller is a bridge between the UI and the implemenation
        /// of the code behind the buttons.
        /// </summary>
        public WorkspaceController Controller { get; private set; }

        /// <summary>
        /// Gets or sets the currently selected view model node.
        /// </summary>
        public object SelectedNodeViewModel
        {
            get
            {
                return this.selectedNodeViewModel;
            }

            set
            {
                if (this.selectedNodeViewModel != value)
                {
                    var argNode = value as WorkspaceNodeViewModelBase;
                    var curNode = this.selectedNodeViewModel as WorkspaceNodeViewModelBase;
                    if (argNode != null
                        && curNode != null
                        && argNode.IsMultiSelected
                        && argNode.GetType() == curNode.GetType())
                    {
                        // 複数選択状態の同種ノードをクリックした際は入れ替え
                        curNode.IsMultiSelected = true;
                        argNode.IsMultiSelected = false;
                    }
                    else
                    {
                        // それ以外の選択ノード切り替え時は強制的に複数選択を解除
                        this.ClearMultiSelection();
                    }

                    // 古いノードのイベントハンドラを削除
                    var owner = this.GetModifyFlagEventOwner(this.selectedNodeViewModel);

                    if (owner != null)
                    {
                        owner.ModificationFlagViewModel.PropertyChanged -= this.OnModifyFlagPropertyChanged;
                    }

                    this.SetValue(ref this.selectedNodeViewModel, value as ViewModelBase);

                    // Set active target to the command manager.
                    CommandManager.SetActiveTarget(value);

                    // カーブエディタウィンドウへの変更通知
                    if (this.Dialogs.OnNotifyEmitterToCurveEditorExecutable != null)
                    {
                        this.Dialogs.OnNotifyEmitterToCurveEditorExecutable.Execute(value);
                    }

                    this.UpdateUIStates();

                    // 新しいノードにイベントハンドラを追加
                    owner = this.GetModifyFlagEventOwner(this.selectedNodeViewModel);

                    if (owner != null)
                    {
                        owner.ModificationFlagViewModel.PropertyChanged += this.OnModifyFlagPropertyChanged;
                    }
                }
            }
        }

        /// <summary>
        /// Enumerates the available reserved shader items.
        /// (each item contains the name, the icon image, and the user data info)
        /// </summary>
        public IEnumerable<Tuple<string, Image, object>> AvailableReservedShaders
        {
            get
            {
                if (ReservedShaderUserDataManager.Definition == null ||
                    ReservedShaderUserDataManager.Definition.ReservedShaderDefinitions == null)
                {
                    yield break;
                }

                var culture = System.Threading.Thread.CurrentThread.CurrentUICulture;
                var cultureEnUs = System.Globalization.CultureInfo.CreateSpecificCulture("en-US");
                bool useEnglish = culture.Equals(cultureEnUs) == true;

                foreach (var def in ReservedShaderUserDataManager.Definition.ReservedShaderDefinitions)
                {
                    string name = def.Name;
                    if (useEnglish == true && string.IsNullOrEmpty(def.NameEn) == false)
                    {
                        name = def.NameEn;
                    }

                    Image icon = ReservedShaderUserDataManager.LoadReservedShaderIcon(def);

                    yield return Tuple.Create<string, Image, object>(name, icon, def.UserDataInfo);
                }
            }
        }

        /// <summary>
        /// 複数選択による同時編集対象として選択されているノードのコレクションを取得します。
        /// </summary>
        public List<WorkspaceNodeViewModelBase> MultiSelectedNodes
        {
            get { return this.multiSelectedNodes; }
        }

        #region Property - UIのステータスにバインドするプロパティ

        /// <summary>
        /// エミッタセットを新規作成可能か取得します。
        /// </summary>
        public bool CanCreateNewEmitterSet
        {
            get
            {
                // エミッタセットの数をチェック
                return this.WorkspaceViewModel.EmitterSetCount + 1 <= IOConstants.MaxEmitterSetCount;
            }
        }

        /// <summary>
        /// エミッタセットを読み込み可能か取得します。
        /// </summary>
        public bool CanLoadEmitterSet
        {
            get
            {
                // エミッタセットの数をチェック
                return this.WorkspaceViewModel.EmitterSetCount + 1 <= IOConstants.MaxEmitterSetCount;
            }
        }

        /// <summary>
        /// エミッタを新規作成可能か取得します。
        /// </summary>
        public bool CanCreateNewEmitter
        {
            get
            {
                // エミッタセットを選択中のときtrue
                if (this.selectedNodeViewModel is EmitterSetViewModel)
                {
                    var emitterSetViewModel = this.selectedNodeViewModel as EmitterSetViewModel;
                    return this.CanHaveEmitter(emitterSetViewModel);
                }

                // エミッタかを選択中のとき
                if (this.selectedNodeViewModel is EmitterViewModel)
                {
                    var emitterViewModel = this.selectedNodeViewModel as EmitterViewModel;
                    return this.CanHaveEmitter(emitterViewModel.Parent);
                }

                // プレビューを選択中のとき
                if (this.selectedNodeViewModel is PreviewViewModel)
                {
                    var previewViewModel = this.selectedNodeViewModel as PreviewViewModel;
                    return this.CanHaveEmitter(previewViewModel.Parent);
                }

                // それ以外はfalse
                return false;
            }
        }

        /// <summary>
        /// 子エミッタが新規作成可能か取得します。
        /// </summary>
        public bool CanCreateNewChildEmitter
        {
            get
            {
                // エミッタを選択中のときtrue
                if (this.selectedNodeViewModel is EmitterViewModel)
                {
                    var emitterViewModel = this.selectedNodeViewModel as EmitterViewModel;
                    return this.CanHaveEmitter(emitterViewModel);
                }

                // それ以外はfalse
                return false;
            }
        }

        /// <summary>
        /// プリセットからエミッタが新規作成可能か取得します。
        /// </summary>
        public bool CanCreateNewEmitterFromPreset
        {
            get
            {
                if (this.selectedNodeViewModel is EmitterSetViewModel ||
                    this.selectedNodeViewModel is EmitterViewModel)
                {
                    var node = this.selectedNodeViewModel as IHierarchyObject;
                    return this.CanHaveEmitter(node);
                }

                // それ以外はfalse
                return false;
            }
        }

        /// <summary>
        /// 選択しているノードに、新ランダムフィールドが追加可能か取得します.
        /// </summary>
        public bool CanCreateNewRandomField
        {
            get { return this.CanCreateNewField(this.CanHaveRandomField); }
        }

        /// <summary>
        /// 選択しているノードに、ランダムフィールドが追加可能か取得します.
        /// </summary>
        public bool CanCreateNewRandomFe1Field
        {
            get { return this.CanCreateNewField(this.CanHaveRandomFe1Field); }
        }

        /// <summary>
        /// 選択しているノードに、磁石フィールドが追加可能か取得します.
        /// </summary>
        public bool CanCreateNewMagnetField
        {
            get { return this.CanCreateNewField(this.CanHaveMagnetField); }
        }

        /// <summary>
        /// 選択しているノードに、スピンフィールドが追加可能か取得します.
        /// </summary>
        public bool CanCreateNewSpinField
        {
            get { return this.CanCreateNewField(this.CanHaveSpinField); }
        }

        /// <summary>
        /// 選択しているノードに、収束フィールドが追加可能か取得します.
        /// </summary>
        public bool CanCreateNewConvergeField
        {
            get { return this.CanCreateNewField(this.CanHaveConvergeField); }
        }

        /// <summary>
        /// 選択しているノードに、"位置に加算"フィールドが追加可能か取得します.
        /// </summary>
        public bool CanCreateNewAddLocationField
        {
            get { return this.CanCreateNewField(this.CanHaveAddLocationField); }
        }

        /// <summary>
        /// 選択しているノードに、シンプルコリジョンフィールドが追加可能か取得します.
        /// </summary>
        public bool CanCreateNewCollisionField
        {
            get { return this.CanCreateNewField(this.CanHaveCollisionField); }
        }

        /// <summary>
        /// 選択しているノードに、カールノイズフィールドが追加可能か取得します.
        /// </summary>
        public bool CanCreateNewCurlNoiseField
        {
            get { return this.CanCreateNewField(this.CanHaveCurlNoiseField); }
        }

        /// <summary>
        /// 選択しているノードに、カスタムフィールドが追加可能か取得します.
        /// </summary>
        public bool CanCreateNewCustomField
        {
            get { return this.CanCreateNewField(this.CanHaveCustomField); }
        }

        /// <summary>
        /// 選択しているノードに、カスタムアクションが追加可能か取得します.
        /// </summary>
        public bool CanCreateNewCustomAction
        {
            get
            {
                // エミッタを選択している場合
                if (this.selectedNodeViewModel is EmitterViewModel)
                {
                    var emitterViewModel = this.selectedNodeViewModel as EmitterViewModel;
                    return this.CanHaveCustomAction(emitterViewModel);
                }

                // 親ノードを調べる
                var hierarchyViewModel = this.selectedNodeViewModel as HierarchyViewModel;

                // 親を持っていなかった場合
                if (hierarchyViewModel == null || hierarchyViewModel.Parent == null)
                {
                    return false;
                }

                // 親がエミッタの場合
                if (hierarchyViewModel.Parent is EmitterViewModel)
                {
                    var emitterViewModel = hierarchyViewModel.Parent as EmitterViewModel;
                    return this.CanHaveCustomAction(emitterViewModel);
                }

                // それ以外はfalse
                return false;
            }
        }

        /// <summary>
        /// 選択しているノードに、エミッタ拡張パラメータが追加可能か取得します.
        /// </summary>
        public bool CanCreateNewEmitterExtParams
        {
            get
            {
                // エミッタを選択している場合
                if (this.selectedNodeViewModel is EmitterViewModel)
                {
                    var emitterViewModel = this.selectedNodeViewModel as EmitterViewModel;
                    return this.CanHaveEmitterExtParams(emitterViewModel);
                }

                // 親ノードを調べる
                var hierarchyViewModel = this.selectedNodeViewModel as HierarchyViewModel;

                // 親を持っていなかった場合
                if (hierarchyViewModel == null || hierarchyViewModel.Parent == null)
                {
                    return false;
                }

                // 親がエミッタの場合
                if (hierarchyViewModel.Parent is EmitterViewModel)
                {
                    var emitterViewModel = hierarchyViewModel.Parent as EmitterViewModel;
                    return this.CanHaveEmitterExtParams(emitterViewModel);
                }

                // それ以外はfalse
                return false;
            }
        }

        /// <summary>
        /// 選択しているノードに、エミッタプラグインが追加可能か取得します.
        /// </summary>
        public bool CanCreateNewReservedShader
        {
            get
            {
                // エミッタを選択している場合
                if (this.selectedNodeViewModel is EmitterViewModel)
                {
                    var emitterViewModel = this.selectedNodeViewModel as EmitterViewModel;
                    return this.CanHaveReservedShader(emitterViewModel);
                }

                // 親ノードを調べる
                var hierarchyViewModel = this.selectedNodeViewModel as HierarchyViewModel;

                // 親を持っていなかった場合
                if (hierarchyViewModel == null || hierarchyViewModel.Parent == null)
                {
                    return false;
                }

                // 親がエミッタの場合
                if (hierarchyViewModel.Parent is EmitterViewModel)
                {
                    var emitterViewModel = hierarchyViewModel.Parent as EmitterViewModel;
                    return this.CanHaveReservedShader(emitterViewModel);
                }

                // それ以外はfalse
                return false;
            }
        }

        /// <summary>
        /// 選択しているノードが階層化を解除できるか（＝子エミッタか）取得します。
        /// </summary>
        public bool CanUnparentEmitter
        {
            get
            {
                var emitter = this.selectedNodeViewModel as EmitterViewModel;
                return emitter != null && emitter.CanUnparentEmitter;
            }
        }

        /// <summary>
        /// プレビューを新規作成可能か取得します。
        /// </summary>
        public bool CanCreateNewPreview
        {
            get
            {
                // エミッタセット・エミッタ・プレビューを選択中で、32個未満のときtrue
                if (this.selectedNodeViewModel is EmitterSetViewModel ||
                    this.selectedNodeViewModel is EmitterViewModel ||
                    this.selectedNodeViewModel is PreviewViewModel)
                {
                    return this.CanAddPreview;
                }

                // それ以外はfalse
                return false;
            }
        }

        /// <summary>
        /// 選択されたノードが「閉じる」メニューで削除可能かを取得します。
        /// </summary>
        public bool CanCloseSelectedNode
        {
            get
            {
                // エミッタセットを選択していたときtrue
                var emitterSetViewModel = this.selectedNodeViewModel as EmitterSetViewModel;

                if (emitterSetViewModel != null)
                {
                    return true;
                }

                // プレビューを選択していたとき
                var previewViewModel = this.selectedNodeViewModel as PreviewViewModel;

                if (previewViewModel != null)
                {
                    // 親がエミッタセットでないときfalse
                    var parentEmitterSetViewModel = previewViewModel.Parent as EmitterSetViewModel;

                    if (parentEmitterSetViewModel == null)
                    {
                        return false;
                    }

                    // 親エミッタセットに子プレビューが2つ以上あるときtrue
                    // それ以外はfalse
                    int numPreviews = parentEmitterSetViewModel.Children.Count(item => item is PreviewViewModel);

                    return numPreviews >= 2 ? true : false;
                }

                return false;
            }
        }

        /// <summary>
        /// 選択されたノードが「削除」メニューで削除可能かを取得します。
        /// </summary>
        public bool CanDeleteSelectedNode
        {
            get
            {
                // エミッタを選択していたとき
                var emitterViewModel = this.selectedNodeViewModel as EmitterViewModel;
                if (emitterViewModel != null)
                {
                    // 親がエミッタセットでないときtrue
                    var parentEmitterSetViewModel = emitterViewModel.Parent as EmitterSetViewModel;
                    if (parentEmitterSetViewModel == null)
                    {
                        return true;
                    }

                    // 非アクティブのときtrue
                    if (emitterViewModel.EnableConvert == false)
                    {
                        return true;
                    }

                    // 親エミッタセットに子エミッタが2つ以上あるときtrue
                    // それ以外はfalse
                    int numEmitters = 0;
                    foreach (ViewModelBase item in parentEmitterSetViewModel.Children)
                    {
                        var emitter = item as EmitterViewModel;
                        if (emitter != null && emitter.EnableConvert)
                        {
                            ++numEmitters;
                        }
                    }

                    return numEmitters >= 2 ? true : false;
                }

                // フィールドを選択していたときtrue
                var fieldViewModel = this.selectedNodeViewModel as FieldViewModel;
                if (fieldViewModel != null)
                {
                    return true;
                }

                // エミッタプラグインを選択しているときtrue
                var reservedShaderViewModel = this.selectedNodeViewModel as ReservedShaderNodeViewModel;
                if (reservedShaderViewModel != null)
                {
                    return true;
                }

                // カスタムアクションを選択しているときtrue
                var customActionViewModel = this.selectedNodeViewModel as CustomActionViewModel;
                if (customActionViewModel != null)
                {
                    return true;
                }

                // エミッタ拡張パラメータを選択しているときtrue
                var emitterExtParamsViewModel = this.selectedNodeViewModel as EmitterExtParamsViewModel;
                if (emitterExtParamsViewModel != null)
                {
                    return true;
                }

                return false;
            }
        }

        /// <summary>
        /// 選択されたノードが「機能の有効 / 無効」メニューでON/OFF可能かを取得します。
        /// </summary>
        public bool CanSwitchConvertFlag
        {
            get
            {
                // エミッタを選択しているとき
                var emitterViewModel = this.selectedNodeViewModel as EmitterViewModel;
                if (emitterViewModel != null)
                {
                    // 親がエミッタセットでないときtrue
                    var parentEmitterSetViewModel = emitterViewModel.Parent as EmitterSetViewModel;
                    if (parentEmitterSetViewModel == null)
                    {
                        return true;
                    }

                    // アクティブでないときtrue
                    if (emitterViewModel.EnableConvert == false)
                    {
                        return true;
                    }

                    // 親エミッタセットにアクティブな子エミッタが2つ以上あるときtrue
                    // それ以外はfalse
                    int numEmitters = 0;
                    foreach (ViewModelBase item in parentEmitterSetViewModel.Children)
                    {
                        var emitter = item as EmitterViewModel;
                        if (emitter != null && emitter.EnableConvert)
                        {
                            ++numEmitters;
                        }
                    }

                    return numEmitters >= 2 ? true : false;
                }

                // フィールドを選択しているときtrue
                var fieldViewModel = this.selectedNodeViewModel as FieldViewModel;
                if (fieldViewModel != null)
                {
                    return true;
                }

                // カスタムアクションを選択しているときtrue
                var customActionViewModel = this.selectedNodeViewModel as CustomActionViewModel;
                if (customActionViewModel != null)
                {
                    return true;
                }

                // エミッタプラグインを選択しているときtrue
                var pluginViewModel = this.selectedNodeViewModel as ReservedShaderNodeViewModel;
                if (pluginViewModel != null)
                {
                    return true;
                }

                return false;
            }
        }

        /// <summary>
        /// 選択中のノードが削除可能か取得します(「閉じる」と「削除」の論理和)。
        /// </summary>
        public bool CanRemoveSelectedNode
        {
            get
            {
                return this.CanCloseSelectedNode | this.CanDeleteSelectedNode;
            }
        }

        /// <summary>
        /// 「全て閉じる」が実行可能かどうかを取得します。
        /// </summary>
        public bool CanCloseAll
        {
            get
            {
                var viewer = this.GetChild<ViewerViewModel>();

                // エミッタセットを開いているか？
                bool result = this.WorkspaceViewModel.GetChildren<EmitterSetViewModel>().Any();

                // モデルノードに何か読み込んでいるか？
                result |= viewer.GetChildren<ModelViewModel>().Any(m => m.DataModel.ModelBinaryId != -1);

                // 背景画像を何か設定しているか？
                result |= !string.IsNullOrEmpty(viewer.DataModel.ViewerBackgroundData.ImageFilePath);

                // PTCL転送を何かしているか？
                result |= !string.IsNullOrEmpty(viewer.DataModel.ViewerBasicData.PtclData.FilePath);

                // どれかに該当するなら「全て閉じる」を有効にする。
                return result;
            }
        }

        /// <summary>
        /// 削除可能なノードの有無を取得します。
        /// </summary>
        public bool CanRemoveAnyNode
        {
            get
            {
                // チャイルドに一つでもエミッタセットがあればtrue,
                // なければfalseを返す
                var emitterSet = this.WorkspaceViewModel.GetChild<EmitterSetViewModel>();
                if (emitterSet != null)
                {
                    return true;
                }
                else
                {
                    return false;
                }
            }
        }

        /// <summary>
        /// 選択中のノードが開き直せるか取得します。
        /// </summary>
        public bool CanReopenSelectedNode
        {
            get
            {
                var owner = this.GetViewModelToSave(
                    this.selectedNodeViewModel) as IModificationFlagOwner;

                // 保存可能なノードを選択していなかった
                if (owner == null)
                {
                    return false;
                }

                // 保存対象がエミッタセットのとき、ファイルパスが指定されているかチェック
                var emitterSetViewModel = owner as EmitterSetViewModel;

                if (emitterSetViewModel != null)
                {
                    if (string.IsNullOrEmpty(emitterSetViewModel.FilePath))
                    {
                        return false;
                    }

                    if (!File.Exists(Path.Combine(
                            emitterSetViewModel.FilePath,
                            emitterSetViewModel.DataModel.Name + IOConstants.EmitterSetFileExtension)))
                    {
                        return false;
                    }
                }

                // 保存対象がプレビューのとき、ファイルパスが指定されているかチェック
                var previewViewModel = owner as PreviewViewModel;

                if (previewViewModel != null)
                {
                    if (string.IsNullOrEmpty(previewViewModel.FilePath))
                    {
                        return false;
                    }

                    if (!File.Exists(Path.Combine(
                            previewViewModel.FilePath,
                            previewViewModel.DataModel.Name + IOConstants.PreviewFileExtension)))
                    {
                        return false;
                    }
                }

                // ノードに変更があった
                return owner.ModificationFlagViewModel.IsAnyValueModified;
            }
        }

        /// <summary>
        /// 選択中のノードが上書き保存可能か取得します。
        /// </summary>
        public bool CanSaveSelectedNode
        {
            get
            {
                IModificationFlagOwner owner = this.GetViewModelToSave(
                    this.selectedNodeViewModel) as IModificationFlagOwner;

                // 保存可能なノードを選択していなかった
                if (owner == null)
                {
                    return false;
                }

                // ノードに変更があった
                if (owner.ModificationFlagViewModel.IsAnyValueModified)
                {
                    return true;
                }

                // 保存対象がエミッタセットのとき、ファイルパスが指定されているかチェック
                var emitterSetViewModel = owner as EmitterSetViewModel;

                if (emitterSetViewModel != null)
                {
                    if (string.IsNullOrEmpty(emitterSetViewModel.FilePath))
                    {
                        return false;
                    }
                }

                // 保存対象がプレビューのとき、ファイルパスが指定されているかチェック
                PreviewViewModel previewViewModel = owner as PreviewViewModel;

                if (previewViewModel != null)
                {
                    if (string.IsNullOrEmpty(previewViewModel.FilePath))
                    {
                        return false;
                    }
                }

                return true;
            }
        }

        /// <summary>
        /// 選択中のノードが名前を付けて保存可能かを取得します。
        /// </summary>
        public bool CanSaveAsSelectedNode
        {
            get
            {
                IModificationFlagOwner owner = this.GetViewModelToSave(
                    this.selectedNodeViewModel) as IModificationFlagOwner;

                if (owner == null)
                {
                    return false;
                }

                return true;
            }
        }

        /// <summary>
        /// 上書き保存可能なノードの有無を取得します。
        /// </summary>
        public bool CanSaveAnyNode
        {
            get
            {
                // スターマークのついているエミッタセットが一つでもあればtrueを返す
                foreach (IHierarchyObject child in this.WorkspaceViewModel.Children)
                {
                    var emitterSet = child as EmitterSetViewModel;

                    if (emitterSet == null)
                    {
                        continue;
                    }

                    if (emitterSet.ModificationFlagViewModel.IsAnyValueModified)
                    {
                        return true;
                    }
                }

                return false;
            }
        }

        /// <summary>
        /// 選択中のノードが複製可能か取得します。
        /// </summary>
        public bool CanDuplicateSelectedNode
        {
            get
            {
                // エミッタセットを選択していたとき
                if (this.selectedNodeViewModel is EmitterSetViewModel)
                {
                    // エミッタセットの数をチェック
                    return this.WorkspaceViewModel.EmitterSetCount + 1 <= IOConstants.MaxEmitterSetCount;
                }
                else if (this.selectedNodeViewModel is EmitterViewModel)
                {
                    var emitter = this.selectedNodeViewModel as EmitterViewModel;

                    // エミッタの数をチェック
                    if (emitter.Parent is EmitterSetViewModel)
                    {
                        var emitterSet = emitter.Parent as EmitterSetViewModel;
                        return emitterSet.Emitters.Count() + 1 <= IOConstants.MaxEmitterCount;
                    }
                    else
                    {
                        var parentNode = emitter.Parent;

                        // 孫以降のエミッタも対応できるように、エミッタの一番上の親まで追っていく
                        while (parentNode is EmitterViewModel)
                        {
                           parentNode = parentNode.Parent;
                        }

                        // エミッタの一番上の親がエミッタセットだった場合
                        if (parentNode is EmitterSetViewModel)
                        {
                            var emitterSet = parentNode as EmitterSetViewModel;
                            return emitterSet.Emitters.Count() + 1 <= IOConstants.MaxEmitterCount;
                        }
                    }
                }
                else if (this.selectedNodeViewModel is PreviewViewModel)
                {
                    return this.CanAddPreview;
                }

                return false;
            }
        }

        /// <summary>
        /// 選択ノードをコピーできるか
        /// </summary>
        public bool CanCopySelectedNode
        {
            get
            {
                var viewModel = this.selectedNodeViewModel as WorkspaceNodeViewModelBase;

                if (viewModel == null)
                {
                    return false;
                }

                IExecutable exec = viewModel.NodeCopyExecutable;
                if (exec == null)
                {
                    return false;
                }

                return exec.CanExecute(null);
            }
        }

        /// <summary>
        /// 選択ノードをペーストできるか
        /// </summary>
        public bool CanPasteSelectedNode
        {
            get
            {
                var viewModel = this.selectedNodeViewModel as WorkspaceNodeViewModelBase;
                if (viewModel == null)
                {
                    return false;
                }

                IExecutable update = viewModel.EvaluateCopyPasteAvailabilityExecutable;
                IExecutable exec = viewModel.NodePasteExecutable;
                if (update == null || exec == null)
                {
                    return false;
                }

                update.Execute(null);
                return exec.CanExecute(null);
            }
        }

        /// <summary>
        /// Gets a value indicating whether can paste emitter.
        /// </summary>
        public bool CanPasteEmitter
        {
            get
            {
                bool canPasteEmitter = true;
                var esetVM = this.SelectedNodeViewModel as EmitterSetViewModel;
                canPasteEmitter &= esetVM != null;
                if (canPasteEmitter)
                {
                    canPasteEmitter &= esetVM.CanPasteEmitterNode;
                    canPasteEmitter &= esetVM.Emitters.Count() + 1 <= IOConstants.MaxEmitterCount;
                }

                return canPasteEmitter;
            }
        }

        /// <summary>
        /// 単純な貼り付けか、エミッタセットへのエミッタの貼り付けが可能かを取得します。
        /// </summary>
        public bool CanPasteNodeOrPasteEmitter
        {
            get
            {
                return this.CanPasteSelectedNode || this.CanPasteEmitter;
            }
        }

        /// <summary>
        /// 選択中のノードが名前を変更可能かを取得します。
        /// </summary>
        public bool CanRenameSelectedNode
        {
            get
            {
                var viewModel = this.selectedNodeViewModel as HierarchyViewModel;
                if (viewModel == null)
                {
                    return false;
                }

                return viewModel.CanSetName;
            }
        }

        /// <summary>
        /// 選択されたノードの種類に合わせて閉じるボタンのツールチップテキストを返すプロパティ
        /// </summary>
        public string RemoveButtonText
        {
            get
            {
                if (this.SelectedNodeViewModel is WorkspaceViewModel ||
                    this.SelectedNodeViewModel is EmitterSetViewModel ||
                    this.SelectedNodeViewModel is PreviewViewModel ||
                    this.SelectedNodeViewModel is ViewerViewModel ||
                    this.SelectedNodeViewModel is ModelViewModel)
                {
                    return Properties.Resources.ButtonHintClose;
                }

                return Properties.Resources.ButtonHintDelete;
            }
        }

        /// <summary>
        /// ワークスペースのストライプメニューで最後に追加されたサブノードのタイプ
        /// </summary>
        public SubNodeTypes LastAddNode
        {
            get { return this.lastAddNode; }
            set { this.lastAddNode = value; }
        }

        /// <summary>
        /// ストライプメニューのToolStripSplitButtonで、現在選択されているサブノードのタイプを追加できるか返す.
        /// </summary>
        public bool CanAddLastNode
        {
            get
            {
                switch (this.LastAddNode)
                {
                    case SubNodeTypes.RandomFe1:
                        return this.CanCreateNewRandomFe1Field;
                    case SubNodeTypes.Random:
                        return this.CanCreateNewRandomField;
                    case SubNodeTypes.Magnet:
                        return this.CanCreateNewMagnetField;
                    case SubNodeTypes.Spin:
                        return this.CanCreateNewSpinField;
                    case SubNodeTypes.Converge:
                        return this.CanCreateNewConvergeField;
                    case SubNodeTypes.AddLocation:
                        return this.CanCreateNewAddLocationField;
                    case SubNodeTypes.Collision:
                        return this.CanCreateNewCollisionField;
                    case SubNodeTypes.CurlNoise:
                        return this.CanCreateNewCurlNoiseField;
                    case SubNodeTypes.CustomField:
                        return this.CanCreateNewCustomField;
                    case SubNodeTypes.CustomAction:
                        return this.CanCreateNewCustomAction;
                    case SubNodeTypes.EmitterExtParams:
                        return this.CanCreateNewEmitterExtParams;
                    case SubNodeTypes.ReservedShader:
                        return this.CanCreateNewReservedShader;
                    default:
                        return true;
                }
            }
        }

        /// <summary>
        /// その他ノードを追加できるか
        /// </summary>
        public bool CanAddAnotherNode
        {
            get
            {
                var hierarchyViewModel = this.SelectedNodeViewModel as HierarchyViewModel;
                var emitterViewModel = hierarchyViewModel as EmitterViewModel;
                if (emitterViewModel == null && hierarchyViewModel != null)
                {
                    emitterViewModel = hierarchyViewModel.Parent as EmitterViewModel;
                }

                if (emitterViewModel != null)
                {
                    return this.CanHaveAddLocationField(emitterViewModel) ||
                        this.CanHaveCollisionField(emitterViewModel) ||
                        this.CanHaveConvergeField(emitterViewModel) ||
                        this.CanHaveCurlNoiseField(emitterViewModel) ||
                        this.CanHaveMagnetField(emitterViewModel) ||
                        this.CanHaveRandomFe1Field(emitterViewModel) ||
                        this.CanHaveRandomField(emitterViewModel) ||
                        this.CanHaveSpinField(emitterViewModel) ||
                        this.CanHaveCustomField(emitterViewModel) ||
                        this.CanHaveReservedShader(emitterViewModel) ||
                        this.CanHaveCustomAction(emitterViewModel) ||
                        this.CanHaveEmitterExtParams(emitterViewModel);
                }

                return false;
            }
        }

        /// <summary>
        /// プレビューが追加できる状態か
        /// </summary>
        public bool CanAddPreview
        {
            get
            {
                var nodeViewModel = this.selectedNodeViewModel as IHierarchyObject;
                if (nodeViewModel == null)
                {
                    return false;
                }

                var parentEmitterSet = nodeViewModel as EmitterSetViewModel ??
                                       nodeViewModel.FindNearestParentOfType<EmitterSetViewModel>();
                if (parentEmitterSet == null)
                {
                    return false;
                }

                return parentEmitterSet.GetChildren<PreviewViewModel>().Count() < 32;
            }
        }

        #endregion

        #region IExecutable - UIから呼び出すIExecutable

        /// <summary>
        /// 全てのエミッタセットを閉じるIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnCloseAllExecutable { get; set; }

        /// <summary>
        /// エミッタセットを新規作成するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnCreateNewEmitterSetExecutable { get; set; }

        /// <summary>
        /// エミッタを新規作成するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnCreateNewEmitterExecutable { get; set; }

        /// <summary>
        /// プレビューを新規作成するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnCreateNewPreviewExecutable { get; set; }

        /// <summary>
        /// ノードを削除するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnRemoveNodeExecutable { get; set; }

        /// <summary>
        /// ファイルを保存するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnFileSaveExecutable { get; set; }

        /// <summary>
        /// ファイルを全て上書き保存するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnFileSaveAllExecutable { get; set; }

        /// <summary>
        /// ファイルを全てエクスポートするIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnFileSaveAllAsExecutable { get; set; }

        /// <summary>
        /// ファイルを開くIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnFileOpenExecutable { get; set; }

        /// <summary>
        /// ワークスペースを保存するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnWorkspaceSaveExecutable { get; set; }

        /// <summary>
        /// ワークスペースを開くIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnWorkspaceOpenExecutable { get; set; }

        /// <summary>
        /// エミッタセットにリンクを作成するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnLinkEmitterSetExecutable { get; set; }

        /// <summary>
        /// エミッタセットのリンクを削除するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnUnlinkEmitterSetExecutable { get; set; }

        /// <summary>
        /// 全てのエミッタセットのリンクを削除するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnUnlinkAllEmitterSetsExecutable { get; set; }

        #endregion

        /// <summary>
        /// メインで使用するダイアログ
        /// </summary>
        public MainDialogs Dialogs { get; set; }

        /// <summary>
        /// フックコマンドを呼び出します。
        /// </summary>
        /// <param name="cmdType">コマンドタイプ</param>
        /// <param name="targetList">ファイルリスト</param>
        /// <returns>成功すればtrue.</returns>
        public static bool CallHookCommand(HookCommand cmdType, List<string> targetList)
        {
            string[] errMsg;
            int result = HookCommandExecuter.Execute(cmdType, targetList, out errMsg);
            if (result != 0)
            {
                var exe = WorkspaceRootViewModel.Instance.Dialogs.OnShowErrorDialogExecutable;
                if (exe != null)
                {
                    exe.Execute(errMsg);
                }

                Logger.Log("LogView", LogLevels.Error, errMsg[2]);
                return false;
            }

            // ファイルオープン前コマンドの処理後にFE1アップデータを実行
            if (cmdType == HookCommand.PreOpen && OptionStore.RuntimeOptions.IsCommandLineMode == false)
            {
                AncientEsetUpdater.Instance.PreOpenUpdate(targetList, AncientEsetUpdater.EsetType.FE1);
            }

            return true;
        }

        /// <summary>
        /// 複数選択しているノードを解除します。
        /// </summary>
        public void ClearMultiSelection()
        {
            var tmpList = this.MultiSelectedNodes.ToArray();
            foreach (var node in tmpList)
            {
                node.IsMultiSelected = false;
            }

            this.MultiSelectedNodes.Clear();
        }

        /// <summary>
        /// parentViewModelを親とするエミッタを追加することができるかチェックする.
        /// </summary>
        /// <param name="parentViewModel">追加したい場所の親エミッタ</param>
        /// <returns>追加可能ならtrueを、不可能ならfalseを返す.</returns>
        public bool CanHaveEmitter(IHierarchyObject parentViewModel)
        {
            // エミッタセットは１番上にあるので、常にエミッタを追加できる
            if (parentViewModel is EmitterSetViewModel)
            {
                EmitterSetViewModel emitterSet = parentViewModel as EmitterSetViewModel;
                return emitterSet.Emitters.Count() + 1 <= IOConstants.MaxEmitterCount;
            }

            // エミッタの場合は、どの階層にいるのか調べる
            if (parentViewModel is EmitterViewModel)
            {
                // 許容されるエミッタの階層数
                const int Threshold = 2;

                // viewModelがエミッタなので、初期状態で階層数を1にしておく
                int counter = 1;

                var viewModel = parentViewModel;
                while (counter < Threshold)
                {
                    var emitterViewModel = viewModel.Parent as EmitterViewModel;
                    if (emitterViewModel == null)
                    {
                        var emitterSetViewModel = viewModel.Parent as EmitterSetViewModel;

                        // 親がエミッタでもエミッタセットでもない場合はfalseを返す.
                        if (emitterSetViewModel == null)
                        {
                            return false;
                        }

                        // エミッタセットが持っているエミッタの数がIOConstans.MaxEmitterCount以上だと、新しいエミッタを追加できない。
                        if (emitterSetViewModel.Emitters.Count() >= IOConstants.MaxEmitterCount)
                        {
                            return false;
                        }

                        break;
                    }

                    // 親エミッタがnullだったらfalseを返す.
                    viewModel = viewModel.Parent;
                    if (viewModel == null)
                    {
                        return false;
                    }

                    counter++;
                }

                // エミッタの階層数がthrethold未満なら、エミッタを追加することを許可する.
                return counter < Threshold;
            }

            // エミッタでもエミッタセットでもない場合は、falseを返す.
            return false;
        }

        /// <summary>
        /// 与えられたViewModelで、新ランダムフィールドが作成可能か調べる.
        /// </summary>
        /// <param name="viewModel">チェックを行なうViewModel</param>
        /// <returns>viewModelが新ランダムフィールドを作成可能ならtrueを返す.</returns>
        public bool CanHaveRandomField(IHierarchyObject viewModel)
        {
            return this.NotExistSubNodeType(viewModel, typeof(RandomViewModel));
        }

        /// <summary>
        /// 与えられたViewModelで、ランダムフィールドが作成可能か調べる.
        /// </summary>
        /// <param name="viewModel">チェックを行なうViewModel</param>
        /// <returns>viewModelがランダムフィールドを作成可能ならtrueを返す.</returns>
        public bool CanHaveRandomFe1Field(IHierarchyObject viewModel)
        {
            return this.NotExistSubNodeType(viewModel, typeof(RandomFe1ViewModel));
        }

        /// <summary>
        /// 与えられたViewModelで、磁力フィールドが作成可能か調べる.
        /// </summary>
        /// <param name="viewModel">チェックを行なうViewModel</param>
        /// <returns>viewModelが磁力フィールドを作成可能ならtrueを返す.</returns>
        public bool CanHaveMagnetField(IHierarchyObject viewModel)
        {
            return this.NotExistSubNodeType(viewModel, typeof(MagnetViewModel));
        }

        /// <summary>
        /// 与えられたViewModelで、スピンフィールドが作成可能か調べる.
        /// </summary>
        /// <param name="viewModel">チェックを行なうViewModel</param>
        /// <returns>viewModelがスピンフィールドを作成可能ならtrueを返す.</returns>
        public bool CanHaveSpinField(IHierarchyObject viewModel)
        {
            return this.NotExistSubNodeType(viewModel, typeof(SpinViewModel));
        }

        /// <summary>
        /// 与えられたViewModelで、収束フィールドが作成可能か調べる.
        /// </summary>
        /// <param name="viewModel">チェックを行なうViewModel</param>
        /// <returns>viewModelが収束フィールドを作成可能ならtrueを返す.</returns>
        public bool CanHaveConvergeField(IHierarchyObject viewModel)
        {
            return this.NotExistSubNodeType(viewModel, typeof(ConvergeViewModel));
        }

        /// <summary>
        /// 与えられたViewModelで、"位置に加算"フィールドが作成可能か調べる.
        /// </summary>
        /// <param name="viewModel">チェックを行なうViewModel</param>
        /// <returns>viewModelが"位置に加算"フィールドを作成可能ならtrueを返す.</returns>
        public bool CanHaveAddLocationField(IHierarchyObject viewModel)
        {
            return this.NotExistSubNodeType(viewModel, typeof(AddLocationViewModel));
        }

        /// <summary>
        /// 与えられたViewModelで、シンプルコリジョンフィールドが作成可能か調べる.
        /// </summary>
        /// <param name="viewModel">チェックを行なうViewModel</param>
        /// <returns>viewModelがシンプルコリジョンフィールドを作成可能ならtrueを返す.</returns>
        public bool CanHaveCollisionField(IHierarchyObject viewModel)
        {
            return this.NotExistSubNodeType(viewModel, typeof(CollisionViewModel));
        }

        /// <summary>
        /// 与えられたViewModelで、カールノイズフィールドが作成可能か調べる.
        /// </summary>
        /// <param name="viewModel">チェックを行なうViewModel</param>
        /// <returns>viewModelがカールノイズフィールドを作成可能ならtrueを返す.</returns>
        public bool CanHaveCurlNoiseField(IHierarchyObject viewModel)
        {
            return this.NotExistSubNodeType(viewModel, typeof(CurlNoiseViewModel));
        }

        /// <summary>
        /// 与えられたViewModelで、カスタムフィールドが作成可能か調べる.
        /// </summary>
        /// <param name="viewModel">チェックを行なうViewModel</param>
        /// <returns>viewModelがカスタムフィールドを作成可能ならtrueを返す.</returns>
        public bool CanHaveCustomField(IHierarchyObject viewModel)
        {
            return this.NotExistSubNodeType(viewModel, typeof(CustomViewModel));
        }

        /// <summary>
        /// 与えられたViewModelで、カスタムアクションが作成可能か調べる.
        /// </summary>
        /// <param name="viewModel">チェックを行なうViewModel</param>
        /// <returns>viewModelがカスタムアクションを作成可能ならtrueを返す.</returns>
        public bool CanHaveCustomAction(IHierarchyObject viewModel)
        {
            return this.NotExistSubNodeType(viewModel, typeof(CustomActionViewModel));
        }

        /// <summary>
        /// 与えられたViewModelで、エミッタ拡張パラメータが作成可能か調べる.
        /// </summary>
        /// <param name="viewModel">チェックを行なうViewModel</param>
        /// <returns>viewModelがエミッタ拡張パラメータを作成可能ならtrueを返す.</returns>
        public bool CanHaveEmitterExtParams(IHierarchyObject viewModel)
        {
            return this.NotExistSubNodeType(viewModel, typeof(EmitterExtParamsViewModel));
        }

        /// <summary>
        /// 与えられたViewModelで、エミッタプラグイン作成可能か調べる.
        /// </summary>
        /// <param name="viewModel">チェックを行なうViewModel</param>
        /// <returns>viewModelがカスタムアクションを作成可能ならtrueを返す.</returns>
        public bool CanHaveReservedShader(IHierarchyObject viewModel)
        {
            return this.NotExistSubNodeType(viewModel, typeof(ReservedShaderNodeViewModel));
        }

        /// <summary>
        /// 指定されたメソッドを使用して、新しいフィールドが作成可能か返す.
        /// </summary>
        /// <param name="checkField">確認したいフィールドが作成可能か返すメソッド</param>
        /// <returns>作成可能ならtrue, 不可能ならfalse</returns>
        public bool CanCreateNewField(Func<IHierarchyObject, bool> checkField)
        {
            // エミッタを選択している場合
            if (this.selectedNodeViewModel is EmitterViewModel)
            {
                var emitterViewModel = this.selectedNodeViewModel as EmitterViewModel;
                return checkField(emitterViewModel);
            }

            // 親ノードを調べる
            var hierarchyViewModel = this.selectedNodeViewModel as HierarchyViewModel;

            // 親を持っていなかった場合
            if (hierarchyViewModel == null || hierarchyViewModel.Parent == null)
            {
                return false;
            }

            // 親がエミッタの場合
            if (hierarchyViewModel.Parent is EmitterViewModel)
            {
                var emitterViewModel = hierarchyViewModel.Parent as EmitterViewModel;
                return checkField(emitterViewModel);
            }

            // それ以外はfalse
            return false;
        }

        /// <summary>
        /// エミッタセットの数を取得します。
        /// </summary>
        /// <returns>エミッタセットの数を返します。</returns>
        public int GetNumEmitterSet()
        {
            int count = this.WorkspaceViewModel.Children.Where(
                item => item is EmitterSetViewModel).Count();

            return count;
        }

        #region Function - UIから呼び出すIExecutableの本体

        /// <summary>
        /// Creates a new empty workspace.
        /// </summary>
        public void CreateNewWorkspace()
        {
            // 現状のWorkspaceを破棄
            if (this.WorkspaceViewModel != null)
            {
                this.WorkspaceViewModel.PropertyChanged -= this.OnWorkspacePropertyChanged;
                this.WorkspaceViewModel.Dispose();
                this.WorkspaceViewModel = null;
            }

            foreach (var child in this.Children.OfType<WorkspaceNodeViewModelBase>())
            {
                child.Dispose();
            }

            this.Children.Clear();

            // Childへの登録をコンストラクタ内に移動
            var newWorkSpace = new WorkspaceData();
            this.WorkspaceViewModel = new WorkspaceViewModel(this, newWorkSpace);
            this.WorkspaceViewModel.PropertyChanged += this.OnWorkspacePropertyChanged;
        }

        /// <summary>
        /// エミッタセットを新規作成します。
        /// </summary>
        /// <param name="parameter">プリセットとして選択するインデックス</param>
        public void OnCreateNewEmitterSet(object parameter)
        {
            int index = parameter == null ? 0 : (int)parameter;
            this.WorkspaceViewModel.OnCreateNewEmitterSet(index);
        }

        /// <summary>
        /// エミッタを新規作成します。
        /// </summary>
        /// <param name="parameter">プリセットとして選択するインデックス</param>
        public void OnCreateNewEmitter(object parameter)
        {
            int index = parameter == null ? 0 : (int)parameter;
            this.WorkspaceViewModel.OnCreateNewEmitter(index);
        }

        /// <summary>
        /// 子エミッタを新規作成します。
        /// </summary>
        /// <param name="parameter">プリセットとして選択するインデックス</param>
        public void OnCreateNewChildEmitter(object parameter)
        {
            int index = parameter == null ? 0 : (int)parameter;
            this.WorkspaceViewModel.OnCreateNewChildEmitter(index);
        }

        /// <summary>
        /// プレビューを新規作成します。
        /// </summary>
        public void OnCreateNewPreview()
        {
            this.WorkspaceViewModel.OnCreateNewPreview();
        }

        /// <summary>
        /// 選択中のノードを削除します。
        /// </summary>
        public void OnRemoveNode()
        {
            this.Controller.Remove();
        }

        /// <summary>
        /// ファイルを開きます。
        /// parmeterがnullの場合は、ファイルを開くダイアログを表示します。
        /// </summary>
        /// <param name="parameter">ファイルパスの配列.</param>
        public void OnFileOpen(object parameter)
        {
            string[] fileNames = null;
            string[] parameters = parameter as string[];
            if (parameters != null)
            {
                fileNames = parameters;
            }
            else
            {
                // ファイルを開くダイアログを表示
                fileNames = this.Dialogs.ShowFileOpenDialog();
            }

            // ファイルが選択されなかったとき、読み込み処理をキャンセル
            if (fileNames == null || fileNames.Length == 0)
            {
                return;
            }

            // 描画パスダイログの表示情報をリセット
            WorkspaceRootViewModel.Instance.Dialogs.ResetInvalidDrawPathDialogInfo();

            // パスリストを作成
            var pathList = new SelectedFilePathList(fileNames);

            // 無効なファイルがあったとき、ログにメッセージを表示
            foreach (string path in pathList.InvalidFileList)
            {
                Logger.Log("LogView", LogLevels.Warning, Resources.WarningUnknownFormat, Path.GetFileName(path));
            }

            // ワークスペースは複数開けないので最初の1つだけ読み込む
            var workspacePath = pathList.WorkspaceList.FirstOrDefault();
            if (!string.IsNullOrEmpty(workspacePath))
            {
                this.OnWorkspaceOpen(workspacePath);
            }

            // ワークスペース読み込み後は他のファイルを追加する形で読み込みを継続する

            // プレビューと一緒に読み込んだエミッタセットのリスト
            var loadedEmitterSetList = new List<string>();

            // 選択された全てのプレビューファイルを開く
            foreach (string path in pathList.PreviewFileList)
            {
                // プレビューの読み込みを実行
                bool resCommand = CommandManager.Execute(new LoadPreviewCommand(this.WorkspaceViewModel, path, loadedEmitterSetList, null, null));

                if (!resCommand)
                {
                    continue;
                }
            }

            // プレビューと一緒に読み込んだエミッタセットを読み込みリストから削除
            foreach (string loadedPath in loadedEmitterSetList)
            {
                int findIndex = pathList.EmitterSetList.FindIndex(item => item == loadedPath);

                if (findIndex != -1)
                {
                    pathList.EmitterSetList.RemoveAt(findIndex);
                }
            }

            if (!CallHookCommand(HookCommand.PreOpen, pathList.EmitterSetList))
            {
                return;
            }

            // 選択された全てのエミッタセットファイルを開く
            ////this.WorkspaceViewModel.MultipleOpenMode = true;
            foreach (string path in pathList.EmitterSetList)
            {
                // エミッタセット数が最大値を超えていたら、エラーメッセージを表示して、読み込みを中止する.
                if (this.CanLoadEmitterSet == false)
                {
                    Logger.Log("LogView", LogLevels.Error, Resources.WarningEmitterSetMax);
                    break;
                }

                // エミッタセットの読み込みを実行
                Foundation.Command.CommandManager.Execute(new LoadEmitterSetCommand(this.WorkspaceViewModel, path));
            }

            ////this.WorkspaceViewModel.MultipleOpenMode = false;
        }

        /// <summary>
        /// ファイルを開き直します。
        /// </summary>
        /// <param name="parameter">使用しません</param>
        public void OnFileReopen(object parameter)
        {
            var esetVm = this.SelectedNodeViewModel as EmitterSetViewModel;
            if (esetVm == null)
            {
                return;
            }

            var esetData = esetVm.DataModel;
            if (esetData == null)
            {
                return;
            }

            if (this.Dialogs.OnShowOkCancelDialogExecutable != null)
            {
                var msg = new string[3];
                msg[0] = Resources.ConfirmReopenEset;
                msg[1] = Resources.ConfirmCaption;
                this.Dialogs.OnShowOkCancelDialogExecutable.Execute(msg);
                if (msg[2] == "Cancel")
                {
                    return;
                }
            }

            var filePath = Path.Combine(esetData.FilePath, esetData.Name + IOConstants.EmitterSetFileExtension);
            CommandManager.Execute(new ReopenEmitterSetCommand(esetVm, filePath));
        }

        /// <summary>
        /// ファイルを保存します。
        /// </summary>
        /// <param name="parameter">Custom parameter</param>
        public void OnFileSave(object parameter)
        {
            ViewModelBase saveViewModel = this.GetViewModelToSave(this.selectedNodeViewModel);

            if (saveViewModel == null)
            {
                Logger.Log(
                    LogLevels.Warning,
                    "MainForm.OnFileSaveAs : View model is null.");

                return;
            }

            // 描画パスダイログの表示情報をリセット
            this.Dialogs.ResetInvalidDrawPathDialogInfo();

            // エミッタセットを保存
            var emitterSetViewModel = saveViewModel as EmitterSetViewModel;

            if (emitterSetViewModel != null)
            {
                CommandManager.Execute(new SaveEmitterSetCommand(this.WorkspaceViewModel, emitterSetViewModel, (bool)parameter));
            }

            // プレビューを保存
            var previewViewModel = saveViewModel as PreviewViewModel;

            if (previewViewModel != null)
            {
                // ファイルを保存
                CommandManager.Execute(new SavePreviewCommand(this.WorkspaceViewModel, previewViewModel, (bool)parameter));
            }
        }

        /// <summary>
        /// 全てのファイルを上書き保存します。
        /// </summary>
        /// <param name="parameter">Custom parameter.</param>
        public void OnFileSaveAll(object parameter)
        {
            var targetList = new List<string>();
            var actionList = new List<Action>();

            // 描画パスダイログの表示情報をリセット
            this.Dialogs.ResetInvalidDrawPathDialogInfo();

            // スターマークのついているエミッタセットについてファイル情報と実処理を取得する
            foreach (var emitterSet in this.WorkspaceViewModel.Children.OfType<EmitterSetViewModel>())
            {
                if (emitterSet.ModificationFlagViewModel.IsAnyValueModified)
                {
                    if (string.IsNullOrEmpty(emitterSet.FilePath))
                    {
                        // 新規作成したエミッタセットを保存するかどうか確認
                        string fileName = emitterSet.FileName + IOConstants.EmitterSetFileExtension;
                        DialogResult result = this.Dialogs.ShowSaveModificationDialog(fileName);

                        if (result == DialogResult.No)
                        {
                            // 「いいえ」を選択したときは処理中のエミッタセットを保存リストに登録せず、
                            // 残りのエミッタセットの処理を引き続き行う
                            continue;
                        }
                        else if (result == DialogResult.Cancel)
                        {
                            // 「キャンセル」を選択したときは以降の保存処理をすべてキャンセルする
                            return;
                        }
                    }

                    if (!this.WorkspaceViewModel.GetListAndActionForSaveEmitterSet(
                        emitterSet, targetList, actionList))
                    {
                        return;
                    }
                }
            }

            using (new MessageDropSection())
            {
                if (!CallHookCommand(HookCommand.PreSave, targetList))
                {
                    return;
                }

                foreach (var action in actionList)
                {
                    action();
                }

                CallHookCommand(HookCommand.PostSave, targetList);

                // オプションでプレビューも保存対象に入っているなら保存処理を行う
                if (OptionStore.RootOptions.Basic.Preview.IncludeAllSave)
                {
                    // スターマークのついているエミッタセットについてファイル情報と実処理を取得する
                    var esets = this.WorkspaceViewModel.Children.OfType<EmitterSetViewModel>();
                    foreach (var preview in esets.SelectMany(eset => eset.Children.OfType<PreviewViewModel>()))
                    {
                        // TODO:プレビューのModificationFlagが機能していないので、暫定的に強制保存
                        ////if (preview.ModificationFlagViewModel.IsAnyValueModified)
                        {
                            if (string.IsNullOrEmpty(preview.FilePath))
                            {
                                // 新規作成したエミッタセットを保存するかどうか確認
                                string fileName = preview.FileName + IOConstants.PreviewFileExtension;
                                DialogResult result = this.Dialogs.ShowSaveModificationDialog(fileName);

                                if (result == DialogResult.No)
                                {
                                    // 「いいえ」を選択したときは処理中のプレビューを保存リストに登録せず、
                                    // 残りのプレビューの処理を引き続き行う
                                    continue;
                                }
                                else if (result == DialogResult.Cancel)
                                {
                                    // 「キャンセル」を選択したときは以降の保存処理をすべてキャンセルする
                                    return;
                                }
                            }

                            if (!this.WorkspaceViewModel.SavePreviewFile(preview, false))
                            {
                                return;
                            }
                        }
                    }
                }
            }

            // メニューの状態を更新
            this.UpdateUIStates();
        }

        /// <summary>
        /// 全てエクスポート
        /// </summary>
        /// <param name="parameter">使わないパラメータです</param>
        public void OnFileSaveAllAs(object parameter)
        {
            this.WorkspaceViewModel.ExportAll(null);
        }

        /// <summary>
        /// ワークスペースの保存
        /// </summary>
        /// <param name="parameter">名前をつけるならtrue,上書きベースならfalse.</param>
        public void OnWorkspaceSave(object parameter)
        {
            // 描画パスダイログの表示情報をリセット
            this.Dialogs.ResetInvalidDrawPathDialogInfo();

            var showDialog = (bool)parameter;
            this.WorkspaceViewModel.SaveWorkspaceFile(showDialog);
        }

        /// <summary>
        /// ワークスペースを開く
        /// </summary>
        /// <param name="parameter">開くファイル名.</param>
        public void OnWorkspaceOpen(object parameter)
        {
            var filePath = new string[1];
            filePath[0] = (string)parameter;
            if (string.IsNullOrEmpty(filePath[0]))
            {
                this.Dialogs.OnShowWorkspaceFileOpenDialogExecutable.Execute(filePath);
            }

            if (string.IsNullOrEmpty(filePath[0]))
            {
                return;
            }

            // 開いてるエミッタセットがあったら確認する
            if (this.WorkspaceViewModel.GetChildren<EmitterSetViewModel>().Any())
            {
                if (this.Dialogs.OnShowOkCancelDialogExecutable != null)
                {
                    var msg = new string[3];
                    msg[0] = Resources.WarningExistingEmitterSetOnLoadWorkspace;
                    msg[1] = Resources.WarningCaption;
                    this.Dialogs.OnShowOkCancelDialogExecutable.Execute(msg);
                    if (msg[2] == "Cancel")
                    {
                        msg[0] = Resources.WarningAbortLoadWorkspace;
                        msg[1] = Resources.WarningCaption;
                        this.Dialogs.OnShowWarningDialogExecutable.Execute(msg);
                        return;
                    }
                }
            }

            // 全閉じ
            if (!this.WorkspaceViewModel.OnCloseAll(false))
            {
                return;
            }

            // エミッタセットが全て閉じられていないときは処理をキャンセル
            var emitterSet = this.WorkspaceViewModel.GetChild<EmitterSetViewModel>();
            if (emitterSet != null)
            {
                return;
            }

            this.OnWorkspaceOpenCore(filePath[0]);

            this.WorkspaceViewModel.IsSelected = true;
        }

        /// <summary>
        /// The on work space open core.
        /// </summary>
        /// <param name="filePath">
        /// The file path.
        /// </param>
        public void OnWorkspaceOpenCore(string filePath)
        {
            // ノードの選択を外す
            this.SelectedNodeViewModel = null;

            // Workspaceを作り直す
            this.CreateNewWorkspace();

            // そしてようやくロード
            this.WorkspaceViewModel.LoadWorkspaceFile(filePath);

            this.UpdateCloseAllState();
        }

        /// <summary>
        /// エミッタセットにリンクを作成します。
        /// </summary>
        /// <param name="parameter">エミッタセット名</param>
        public void OnLinkEmitterSet(object parameter)
        {
            string esetName = parameter as string;
            Debug.Assert(string.IsNullOrEmpty(esetName) == false, "パラメータが不正");

            // ファイル名に一致するエミッタセットを取得
            EmitterSetViewModel eset = this.GetEmitterSet(esetName);
            if (eset == null)
            {
                return;
            }

            eset.Linked = true;
        }

        /// <summary>
        /// エミッタセットのリンクを削除します。
        /// </summary>
        /// <param name="parameter">エミッタセット名</param>
        public void OnUnlinkEmitterSet(object parameter)
        {
            string fileName = parameter as string;
            Debug.Assert(string.IsNullOrEmpty(fileName) == false, "パラメータが不正");

            // ファイル名に一致するエミッタセットを取得
            EmitterSetViewModel eset = this.GetEmitterSet(fileName);
            if (eset == null)
            {
                return;
            }

            eset.Linked = false;
        }

        /// <summary>
        /// 全てのエミッタセットのリンクを削除します。
        /// </summary>
        /// <param name="parameter">null</param>
        public void OnUnlinkAllEmitterSets(object parameter)
        {
            Debug.Assert(parameter == null, "パラメータが不正");

            // 全てのエミッタセットのリンクを削除
            foreach (IHierarchyObject item in this.WorkspaceViewModel.Children)
            {
                EmitterSetViewModel eset = item as EmitterSetViewModel;
                if (eset == null)
                {
                    continue;
                }

                eset.Linked = false;
            }
        }

        /// <summary>
        /// 全てのファイルを閉じます。
        /// </summary>
        /// <param name="parameter">
        /// Custom parameter.
        /// [0] : 処理されないファイルがあったときfalse
        /// [1] : 例外発生時に呼び出された場合true
        /// </param>
        public void OnCloseAll(object parameter)
        {
            Debug.Assert(parameter != null, "パラメータがnull");

            var parameters = (object[])parameter;

            parameters[0] = this.WorkspaceViewModel.OnCloseAll((bool)parameters[1]);

            // エミッタセットが全て閉じられていないか、EffectMaker終了時はここでキャンセル
            var emitterSet = this.WorkspaceViewModel.GetChild<EmitterSetViewModel>();
            if (emitterSet != null || this.IsExiting)
            {
                return;
            }

            // 描画パスダイログの表示情報をリセット
            this.Dialogs.ResetInvalidDrawPathDialogInfo();

            // Workspaceを作り直す
            this.CreateNewWorkspace();

            // ModelInfoManagerを初期化
            ModelInfoManager.Initialize(
                this.GetChild<ViewerViewModel>().DataModel,
                SynchronizationContext.Current);

            using (new ForceRefreshBinary())
            {
                // Send Viewer data to the viewer.
                ViewerMessageHelper.SendViewer(this.GetChild<ViewerViewModel>().DataModel);
            }

            this.WorkspaceViewModel.IsSelected = true;

            this.UpdateCloseAllState();
        }

        #endregion

        #region Function - プレビューノードの基本設定タブから呼び出す処理

        /// <summary>
        /// プレビューファイルを選択中のプレビューにロードします。
        /// </summary>
        public void OpenPreviewFile()
        {
            // 選択中のプレビューを取得
            var previewViewModel = this.selectedNodeViewModel as PreviewViewModel;

            if (previewViewModel == null)
            {
                return;
            }

            // ファイルを開くダイアログを表示
            string filePath = this.Dialogs.ShowPreviewFileOpenDialog();

            // ファイルが選択されなかったとき、読み込み処理をキャンセル
            if (filePath == null)
            {
                return;
            }

            // プレビューの読み込みを実行
            bool resCommand = CommandManager.Execute(new LoadPreviewCommand(this.WorkspaceViewModel, filePath, null, null, previewViewModel));
        }

        /// <summary>
        /// プレビューファイルを読み込んで選択中のエミッタセットに追加します。
        /// </summary>
        public void OpenNewPreviewFile()
        {
            EmitterSetViewModel emitterSetViewModel = null;

            // 選択中のエミッタセットを取得
            HierarchyViewModel selectedViewModel = this.selectedNodeViewModel as HierarchyViewModel;

            if (selectedViewModel != null)
            {
                emitterSetViewModel = ViewModelBase.GetParent<EmitterSetViewModel>(selectedViewModel);
            }

            if (emitterSetViewModel == null)
            {
                return;
            }

            // ファイルを開くダイアログを表示
            string filePath = this.Dialogs.ShowPreviewFileOpenDialog();

            // ファイルが選択されなかったとき、読み込み処理をキャンセル
            if (filePath == null)
            {
                return;
            }

            // プレビューの読み込みを実行
            bool resCommand = CommandManager.Execute(new LoadPreviewCommand(this.WorkspaceViewModel, filePath, null, emitterSetViewModel, null));
        }

        #endregion

        /// <summary>
        /// 一致する名前のエミッタセットを取得します。
        /// </summary>
        /// <param name="esetName">エミッタセットの名前</param>
        /// <returns>エミッタセットを返します。</returns>
        public EmitterSetViewModel GetEmitterSet(string esetName)
        {
            foreach (IHierarchyObject child in this.WorkspaceViewModel.Children)
            {
                var emitterSet = child as EmitterSetViewModel;

                if (emitterSet == null)
                {
                    continue;
                }

                if (string.Equals(emitterSet.DataModel.Name, esetName))
                {
                    return emitterSet;
                }
            }

            return null;
        }

        /// <summary>
        /// ビューモデルのデフォルト値を更新します。
        /// </summary>
        /// <param name="viewModel">ビューモデル</param>
        public void UpdateViewModelDescriptors(HierarchyViewModel viewModel)
        {
            // ビューモデルのデフォルト値を更新
            viewModel.UpdatePropertyDescriptors();

            // チャイルドのデフォルト値を更新
            foreach (var child in viewModel.Children.OfType<HierarchyViewModel>())
            {
                this.UpdateViewModelDescriptors(child);
            }
        }

        /// <summary>
        /// The show curve editor.
        /// </summary>
        public void ShowCurveEditor()
        {
            if (this.Dialogs.OnShowCurveEditorExecutable != null)
            {
                this.Dialogs.OnShowCurveEditorExecutable.Execute(null);
            }
        }

        /// <summary>
        /// カーブエディタにデータコンテキストを接続します
        /// </summary>
        /// <param name="dataContext">
        /// The data Context.
        /// </param>
        public void ConnectCurveEditor(object dataContext)
        {
            if (this.Dialogs.OnConnectCurveEditorExecutable != null)
            {
                this.Dialogs.OnConnectCurveEditorExecutable.Execute(dataContext);
            }
        }

        /// <summary>
        /// カーブエディタからデータコンテキストを切断します
        /// </summary>
        /// <param name="dataContext">
        /// The data Context.
        /// </param>
        public void DisconnectCurveEditor(object dataContext)
        {
            if (this.Dialogs.OnDisconnectCurveEditorExecutable != null)
            {
                this.Dialogs.OnDisconnectCurveEditorExecutable.Execute(dataContext);
            }
        }

        /// <summary>
        /// コピー＆ペースト関係のプロパティ変更を発行する
        /// </summary>
        public void RaiseCopyPastePropertyChanged()
        {
            if (this.BlockingUpdateCopyPasteState)
            {
                return;
            }

            this.OnPropertyChanged(() => this.CanCopySelectedNode);
            this.OnPropertyChanged(() => this.CanPasteSelectedNode);
            this.OnPropertyChanged(() => this.CanPasteNodeOrPasteEmitter);
        }

        /// <summary>
        /// UIの表示設定を更新します。
        /// </summary>
        public void UpdateUIStates()
        {
            // TODO:チェック結果をプロパティに保存する
            this.OnPropertyChanged(() => this.CanCreateNewEmitterSet);
            this.OnPropertyChanged(() => this.CanLoadEmitterSet);
            this.OnPropertyChanged(() => this.CanCreateNewEmitter);
            this.OnPropertyChanged(() => this.CanCreateNewChildEmitter);
            this.OnPropertyChanged(() => this.CanCreateNewEmitterFromPreset);
            this.OnPropertyChanged(() => this.CanCreateNewRandomField);
            this.OnPropertyChanged(() => this.CanCreateNewRandomFe1Field);
            this.OnPropertyChanged(() => this.CanCreateNewMagnetField);
            this.OnPropertyChanged(() => this.CanCreateNewSpinField);
            this.OnPropertyChanged(() => this.CanCreateNewConvergeField);
            this.OnPropertyChanged(() => this.CanCreateNewAddLocationField);
            this.OnPropertyChanged(() => this.CanCreateNewCollisionField);
            this.OnPropertyChanged(() => this.CanCreateNewCurlNoiseField);
            this.OnPropertyChanged(() => this.CanCreateNewCustomField);
            this.OnPropertyChanged(() => this.CanCreateNewCustomAction);
            this.OnPropertyChanged(() => this.CanCreateNewEmitterExtParams);
            this.OnPropertyChanged(() => this.CanCreateNewReservedShader);
            this.OnPropertyChanged(() => this.CanCreateNewPreview);
            this.OnPropertyChanged(() => this.CanCloseSelectedNode);
            this.OnPropertyChanged(() => this.CanDeleteSelectedNode);
            this.OnPropertyChanged(() => this.CanSwitchConvertFlag);
            this.OnPropertyChanged(() => this.CanRemoveSelectedNode);
            this.OnPropertyChanged(() => this.RemoveButtonText);
            this.OnPropertyChanged(() => this.CanSaveAsSelectedNode);
            this.OnPropertyChanged(() => this.CanSaveAnyNode);
            this.OnPropertyChanged(() => this.CanSaveSelectedNode);
            this.OnPropertyChanged(() => this.CanReopenSelectedNode);
            this.OnPropertyChanged(() => this.CanRenameSelectedNode);
            this.OnPropertyChanged(() => this.CanDuplicateSelectedNode);
            this.OnPropertyChanged(() => this.CanAddLastNode);
            this.OnPropertyChanged(() => this.CanAddAnotherNode);
            this.OnPropertyChanged(() => this.CanUnparentEmitter);

            this.UpdateCloseAllState();
            this.RaiseCopyPastePropertyChanged();
        }

        /// <summary>
        /// 「全て閉じる」に関するプロパティの変更を通知します。
        /// </summary>
        public void UpdateCloseAllState()
        {
            this.OnPropertyChanged(() => this.CanRemoveAnyNode);
            this.OnPropertyChanged(() => this.CanCloseAll);
        }

        /// <summary>
        /// ViewModelの親を辿ってカーブエディタウィンドウの状態を更新します。
        /// </summary>
        /// <param name="viewModel">ビューモデル</param>
        public void UpdateCurveEditorWindow(IHierarchyObject viewModel)
        {
            var emitterViewModel = viewModel;
            while (!(emitterViewModel is EmitterViewModel))
            {
                emitterViewModel = emitterViewModel.Parent;
                if (emitterViewModel == null)
                {
                    break;
                }
            }

            if (emitterViewModel != null)
            {
                ((EmitterViewModel)emitterViewModel).UpdateEmitterAnimation();
            }
        }

        /// <summary>
        /// viewModelに指定されたサブノードが存在するか調べる.
        /// </summary>
        /// <param name="viewModel">チェックするViewModel</param>
        /// <param name="subNodeType">フィールドのViewModelのクラス</param>
        /// <returns>viewModelがEmitterViewModelであり、subNodeTypeを持っていなかったらtrueを返す.</returns>
        private bool NotExistSubNodeType(IHierarchyObject viewModel, Type subNodeType)
        {
            if (viewModel is EmitterViewModel)
            {
                return viewModel.Children.Any(item => item.GetType() == subNodeType) == false;
            }

            return false;
        }

        /// <summary>
        /// 監視対象のModifyFlagのプロパティが変更されたときの処理を行います。
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnModifyFlagPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "IsAnyValueModified")
            {
                this.OnPropertyChanged(() => this.CanSaveSelectedNode);
                this.OnPropertyChanged(() => this.CanReopenSelectedNode);
            }
        }

        /// <summary>
        /// ModifyFlagの変更をチェックするビューモデルを取得します。
        /// </summary>
        /// <param name="viewModel">ビューモデル</param>
        /// <returns>ModifyFlagの変更をチェックするビューモデルを返します。</returns>
        private IModificationFlagOwner GetModifyFlagEventOwner(ViewModelBase viewModel)
        {
            IModificationFlagOwner owner = null;

            // エミッタセットノードのとき
            if (viewModel is EmitterSetViewModel)
            {
                owner = (IModificationFlagOwner)viewModel;
            }

            // エミッタノードのとき
            // 親のエミッタセットに変更があれば保存可能になるので、
            // 親のエミッタセットを探す
            if (this.SelectedNodeViewModel is EmitterViewModel)
            {
                var parentViewModel = viewModel as IHierarchyObject;

                while (parentViewModel != null)
                {
                    if (parentViewModel is EmitterSetViewModel)
                    {
                        break;
                    }
                    else
                    {
                        parentViewModel = parentViewModel.Parent;
                    }
                }

                owner = (IModificationFlagOwner)parentViewModel;
            }

            // プレビューノードのとき
            if (viewModel is PreviewViewModel)
            {
                owner = (IModificationFlagOwner)viewModel;
            }

            return owner;
        }

        /// <summary>
        /// 保存するビューモデルを取得します。
        /// </summary>
        /// <param name="viewModel">ビューモデル</param>
        /// <returns>保存するビューモデルを返します。</returns>
        private ViewModelBase GetViewModelToSave(ViewModelBase viewModel)
        {
            var hierarchyViewModel = viewModel as HierarchyViewModel;

            if (viewModel == null)
            {
                return null;
            }

            if (viewModel is IFileOwner)
            {
                return viewModel;
            }

            var parent = hierarchyViewModel.Parent as ViewModelBase;
            return this.GetViewModelToSave(parent);
        }

        /// <summary>
        /// WorkspaceViewModelのPropertyChangedイベントをハンドルします.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnWorkspacePropertyChanged(
            object sender,
            PropertyChangedEventArgs e)
        {
            if (e.PropertyName.Equals("Children"))
            {
                this.UpdateCloseAllState();
            }
        }
    }
}
