﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UILogic.Attributes;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the ViewerBackgroundColorViewModel.
    /// </summary>
    public class ViewerBackgroundGroupViewModel : ColorPickerOwnerViewModel<ViewerBackgroundData>, IModificationFlagOwner
    {
        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public ViewerBackgroundGroupViewModel(HierarchyViewModel parent, ViewerBackgroundData dataModel)
            : base(parent, dataModel)
        {
            this.OnReloadBackgroundImageFileExecutable = new AnonymousExecutable(this.OnReloadBackgroundImageFile);

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
            this.ModificationFlagViewModel.AddIgnoreProperties(this.IgnoreColorPickerPropertyNames);
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// 背景の画像ファイルパスを取得または設定します。
        /// </summary>
        [UseDataModelOriginalValue]
        public string ImageFilePath
        {
            get
            {
                return this.GetDataModelValue(() => this.ImageFilePath);
            }

            set
            {
                if (value != this.ImageFilePath)
                {
                    this.SetDataModelValue(value, () => this.ImageFilePath);

                    // ビューアノードをリロード
                    var viewer = GetParent<ViewerViewModel>(this);
                    ViewerMessageHelper.SendViewer(viewer.DataModel);
                }

                WorkspaceRootViewModel.Instance.UpdateCloseAllState();
            }
        }

        /// <summary>
        /// 背景画像ファイルをリロードするためのIExecutableを取得します。
        /// </summary>
        public IExecutable OnReloadBackgroundImageFileExecutable { get; private set; }

        /// <summary>
        /// 背景画像ファイルのリロード処理を行います。
        /// </summary>
        /// <param name="parameter">Custom parameter</param>
        private void OnReloadBackgroundImageFile(object parameter)
        {
            // ビューアノードをリロード
            var viewer = GetParent<ViewerViewModel>(this);
            ViewerMessageHelper.SendViewer(viewer.DataModel);
        }
    }
}
