﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Dynamic;
using System.Linq;
using System.Linq.Expressions;
using System.Reflection;
using System.Runtime.CompilerServices;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.DataModelProxies;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Disposables;
using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.Attributes;
using EffectMaker.UILogic.Commands;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Base class for the view models.
    /// </summary>
    public abstract class ViewModelBase : DynamicObject, INotifyPropertyChanged, IDisposable
    {
        /// <summary>These view model properties will not be added to the property info map.</summary>
        private static readonly HashSet<string> IgnoredViewModelProperties = new HashSet<string>()
        {
            "Parent",
            "Children",
            "DataModel",
            "Proxy",
            "IsSingleColumn",
            "ModificationFlagViewModel",
            "DynamicSelf",
        };

        /// <summary>
        /// DataModelProxy.
        /// </summary>
        private DataModelProxy proxy;

        /// <summary>
        /// Dictionary of the properties of this view model.
        /// </summary>
        private readonly Dictionary<string, PropertyData> viewModelPropInfoMap =
            new Dictionary<string, PropertyData>();

        /// <summary>
        /// プロパティ変更通知を抑制する名前リストです。
        /// </summary>
        private readonly List<string> suppressPropertyChangedNames;

        /// <summary>
        /// Initializes the ViewModelBase instance.
        /// </summary>
        /// <param name="dataModel">The encapsulated data model.</param>
        protected ViewModelBase(DataModelBase dataModel)
        {
            this.proxy = this.CreateDataModelProxy(dataModel);
            this.suppressPropertyChangedNames = new List<string>();
            this.OnValueChangedExecutable = new AnonymousExecutable(this.OnValueChanged);

            this.UpdateViewModelPropertyDescriptorMap();

            OptionStore.OptionChanged += this.OnOptionChanged;
        }

        /// <summary>
        /// PropertyChangedEventHandler.
        /// </summary>
        public event PropertyChangedEventHandler PropertyChanged;

        /// <summary>
        /// Gets the data model proxy.
        /// </summary>
        public dynamic Proxy
        {
            get { return this.proxy; }
        }

        /// <summary>
        /// 上下レイアウトにするかどうかのフラグを取得します。
        /// </summary>
        public bool IsSingleColumn
        {
            get { return OptionStore.RootOptions.Interface.UseSingleColumnLayout; }
        }

        /// <summary>
        /// リニア編集モードかどうか取得または設定します。
        /// </summary>
        public bool IsLinearMode
        {
            get { return OptionStore.ProjectConfig.LinearMode; }
        }

        /// <summary>
        /// Get an executable to run when an value changes.
        /// </summary>
        public IExecutable OnValueChangedExecutable { get; protected set; }

        /// <summary>
        /// プロパティ以外のバインディングを解決する関数を取得または設定します。
        /// </summary>
        protected Func<string, object> CustomBindingResolver { get; set; }

        /// <summary>
        /// 親のビューモデルを辿って一致する型の親を検索します。
        /// </summary>
        /// <typeparam name="T">型</typeparam>
        /// <param name="viewModel">ビューモデル</param>
        /// <returns>見つかったビューモデル</returns>
        public static T GetParent<T>(ViewModelBase viewModel)
            where T : class
        {
            IHierarchyObject node = viewModel as IHierarchyObject;
            T result = null;

            while (node != null && result == null)
            {
                result = node as T;
                node = node.Parent;
            }

            return result;
        }

        /// <summary>
        /// Disposes the instance.
        /// </summary>
        public virtual void Dispose()
        {
            OptionStore.OptionChanged -= this.OnOptionChanged;
            if (this.proxy != null)
            {
                this.proxy.Dispose();
                this.proxy = null;
            }
        }

        /// <summary>
        /// Try to get the requested property value from the data model.
        /// </summary>
        /// <param name="binder">The get member binder.</param>
        /// <param name="result">The property value.</param>
        /// <returns>True on success.</returns>
        public override bool TryGetMember(
            GetMemberBinder binder, out object result)
        {
            result = null;
            if (this.proxy == null || this.proxy.DataModel == null)
            {
                return false;
            }

            PropertyData propertyData = null;

            // プロパティ以外のオブジェクトに対するバインディング解決を行う
            if (this.CustomBindingResolver != null)
            {
                // 解決できた時だけtrueで処理終了、できなかったら通常の解決処理へスルーする
                result = this.CustomBindingResolver(binder.Name);
                if (result != null)
                {
                    return true;
                }
            }

            // Try to get the descriptor of the requested property.
            var effectMakerBinder = binder as EffectMakerGetMemberBinder;
            if (effectMakerBinder != null &&
                effectMakerBinder.ShouldGetNonDynamicMember == true &&
                this.viewModelPropInfoMap.TryGetValue(binder.Name, out propertyData) == true)
            {
                result = propertyData.Info.GetValue(this, null);
                return true;
            }
            else if (this.proxy != null &&
                     this.proxy.TryGetMember(binder, out result) == true)
            {
                return true;
            }
            else
            {
                Logger.Log(
                    LogLevels.Error,
                    "ViewModelBase.TryGetMember : Getting non-existing member '{0}' from view model '{1}'",
                    binder.Name,
                    this.GetType().Name);
                return false;
            }
        }

        /// <summary>
        /// Try to set value to the property from the data model.
        /// </summary>
        /// <param name="binder">The set member binder.</param>
        /// <param name="value">The value to set.</param>
        /// <returns>True on success.</returns>
        public override bool TrySetMember(
            SetMemberBinder binder, object value)
        {
            if (this.proxy == null || this.proxy.DataModel == null)
            {
                return false;
            }

            // Is the binder our binder? There are some extra parameters that we can use.
            var effectMakerBinder = binder as EffectMakerSetMemberBinder;

            // Get the original value.
            var getMemberBinder = new EffectMakerGetMemberBinder(binder.Name);
            if (effectMakerBinder != null && effectMakerBinder.ShouldSetNonDynamicMember == true)
            {
                getMemberBinder.ShouldGetNonDynamicMember = true;
            }

            object origValue;
            if (this.TryGetMember(getMemberBinder, out origValue) == false)
            {
                Debug.Assert(false, string.Format("ViewModelBase.TrySetMember : Failed getting member '{0}' from view model '{1}' for modification checking.", binder.Name, this.GetType().Name));
                return false;
            }

            // 値のセットでバイナリ転送が発生したことを通知するフラグです。
            bool sendBinaryRequested = false;

            // Clone the original value if possible.
            if (origValue is ICloneable)
            {
                origValue = (origValue as ICloneable).Clone();
            }

            // Check if the new value is different from the current value.
            if (object.Equals(origValue, value) == false)
            {
                PropertyData propertyData = null;
                if (WorkspaceRootViewModel.Instance.MultiSelectedNodes.Any() &&
                    !MultiNodeEditUtil.DisableMultiNodeEdit)
                {
                    // 複数ノード編集時にリロードが発生しそうになったら通信を遮断してそれを通知します。
                    // その場合のリロードはSetPropertyCommand内で行っています。
                    using (var adjuster = new MultiNodeEditUtil.ReloadDetector(() =>
                        this.SetValueCore(binder, effectMakerBinder, value, out propertyData)))
                    {
                        if (adjuster.Result == false)
                        {
                            return false;
                        }

                        sendBinaryRequested = adjuster.SendBinaryRequested;
                    }
                }
                else if (this.SetValueCore(binder, effectMakerBinder, value, out propertyData) == false)
                {
                    return false;
                }

                // Store the original value of the property.
                if (propertyData != null &&
                    propertyData.UseDataModelPropertyOriginalValue == false &&
                    propertyData.IsOriginalValueSet == false)
                {
                    propertyData.OriginalValue = origValue;
                }

                // ...and raise an event to tell the value changed.
                if (this.suppressPropertyChangedNames.Contains(binder.Name) == false)
                {
                    this.OnPropertyChanged(binder.Name);
                }
            }

            // Should we issue command?
            if (effectMakerBinder != null &&
                effectMakerBinder.ShouldIssueCommand == true)
            {
                // Figure out which original value to use.
                if (effectMakerBinder.IsOriginalValueSet == true)
                {
                    if (effectMakerBinder.OriginalValue is ICloneable)
                    {
                        origValue = ((ICloneable)effectMakerBinder.OriginalValue).Clone();
                    }
                    else
                    {
                        origValue = effectMakerBinder.OriginalValue;
                    }
                }

                if (object.Equals(origValue, value) == false)
                {
                    // Issue and execute the command.
                    CommandManager.Execute(new SetPropertyCommand(this, binder.Name, origValue, value, sendBinaryRequested));
                }
            }

            return true;
        }

        /// <summary>
        /// Provides a list of available members in the data model.
        /// This method is used for debug purpose and called only by Visual Studio.
        /// </summary>
        /// <returns>Returns a list of the public properties contained in the data model.</returns>
        public override IEnumerable<string> GetDynamicMemberNames()
        {
            if (this.proxy == null || this.proxy.DataModel == null)
            {
                return new string[0];
            }

            return this.Proxy.GetDynamicMemberNames();
        }

        /// <summary>
        /// Get the original value of the specified property.
        /// </summary>
        /// <param name="propertyName">The name of the property.</param>
        /// <returns>The original value.</returns>
        public object GetOriginalValue(string propertyName)
        {
            object origValue;

            // Get the original value.
            PropertyData propertyData = null;
            if (this.viewModelPropInfoMap.TryGetValue(propertyName, out propertyData) == true &&
                propertyData.UseDataModelPropertyOriginalValue == false)
            {
                origValue = propertyData.OriginalValue;
            }
            else
            {
                string dataModelPropertyName = propertyName;
                if (propertyData != null &&
                    propertyData.UseDataModelPropertyOriginalValue == true &&
                    string.IsNullOrEmpty(propertyData.DataModelPropertyNameForOriginalValue) == false)
                {
                    dataModelPropertyName = propertyData.DataModelPropertyNameForOriginalValue;
                }

                if (this.proxy == null ||
                    this.proxy.GetPropertyOriginalValue(dataModelPropertyName, out origValue) == false)
                {
                    return null;
                }
            }

            return origValue;
        }

        /// <summary>
        /// Get the default value of the specified property.
        /// </summary>
        /// <param name="propertyName">The property name.</param>
        /// <returns>The default value.</returns>
        public object GetDefaultValue(string propertyName)
        {
            if (this.proxy == null || this.proxy.DataModel == null)
            {
                return null;
            }

            object defaultValue;

            // Create the binder for TryGetMember method.
            var binder = new EffectMakerGetMemberBinder(propertyName, true);

            // This will temporarily switch the data model to the default data model,
            // so we can get the default value from it.
            using (var block = new DataModelProxy.SwitchDefaultDataModelBlock(this.proxy))
            {
                if (this.TryGetMember(binder, out defaultValue) == false)
                {
                    return null;
                }
            }

            return defaultValue;
        }

        /// <summary>
        /// データロード時から、プロパティが変更されたかチェックする.
        /// </summary>
        /// <param name="propertyName">プロパティ名.</param>
        /// <returns>変更された場合、trueを返す.</returns>
        public bool IsPropertyChanged(string propertyName)
        {
            object origValue;
            object currValue;

            // Get the original value.
            PropertyData propertyData = null;
            if (this.viewModelPropInfoMap.TryGetValue(propertyName, out propertyData) == true &&
                propertyData.UseDataModelPropertyOriginalValue == false)
            {
                if (propertyData.IsOriginalValueSet == false)
                {
                    // The original value has not been set to the property data,
                    // which means the property has never been set through TrySetMember,
                    // thus the property is not changed.
                    return false;
                }

                origValue = propertyData.OriginalValue;
            }
            else
            {
                string dataModelPropertyName = propertyName;
                if (propertyData != null &&
                    propertyData.UseDataModelPropertyOriginalValue == true &&
                    string.IsNullOrEmpty(propertyData.DataModelPropertyNameForOriginalValue) == false)
                {
                    dataModelPropertyName = propertyData.DataModelPropertyNameForOriginalValue;
                }

                if (this.proxy == null ||
                    this.proxy.GetPropertyOriginalValue(dataModelPropertyName, out origValue) == false)
                {
                    return false;
                }
            }

            // Get the current value.
            var binder = new EffectMakerGetMemberBinder(propertyName, true);
            if (this.TryGetMember(binder, out currValue) == false)
            {
                return false;
            }

            // Compare the value and see if it's modified.
            return this.CheckPropertyValueModification(propertyName, origValue, currValue);
        }

        /// <summary>
        /// 値をデフォルト値に戻します.
        /// </summary>
        public virtual void ResetToDefaultValues()
        {
            if (this.proxy == null || this.proxy.DataModel == null)
            {
                return;
            }

            // デフォルト値のデータモデルを自身のデータモデルにセット
            DataModelBase myDataModel = this.proxy.DataModel;
            using (var block = new DataModelProxy.SwitchDefaultDataModelBlock(this.proxy))
            {
                DataModelBase defaultDataModel = this.proxy.DataModel;
                myDataModel.Set(defaultDataModel);
            }
        }

        /// <summary>
        /// 指定したプロパティ名の値がデフォルト値と異なるかチェックする.
        /// </summary>
        /// <param name="propertyName">プロパティ名.</param>
        /// <returns>デフォルト値と異なる場合、trueを返す.</returns>
        public bool IsPropertyDefaultValue(string propertyName)
        {
            if (this.proxy == null || this.proxy.DataModel == null)
            {
                return true;
            }

            object currentValue, defaultValue;

            // Create the binder for TryGetMember method.
            var binder = new EffectMakerGetMemberBinder(propertyName, true);

            // Get value normally through TryGetMember.
            if (this.TryGetMember(binder, out currentValue) == false)
            {
                return true;
            }

            // This will temporarily switch the data model to the default data model,
            // so we can get the default value from it.
            using (var block = new DataModelProxy.SwitchDefaultDataModelBlock(this.proxy))
            {
                if (this.TryGetMember(binder, out defaultValue) == false)
                {
                    return true;
                }
            }

            // Compare the value and see if it's NOT modified.
            return !this.CheckPropertyValueModification(propertyName, defaultValue, currentValue);
        }

        /// <summary>
        /// Resend PropertyChanged notification for all properties.
        /// This is required when the data model changes independently from the view model.
        /// </summary>
        public virtual void FirePropertyChanges()
        {
            foreach (string propertyName in this.viewModelPropInfoMap.Keys)
            {
                if (propertyName == "IsExpanded" ||
                    propertyName == "IsSelected" ||
                    propertyName == "IsMultiSelected")
                {
                    continue;
                }

                this.OnPropertyChanged(propertyName);
            }

            foreach (string propertyName in this.proxy.GetDynamicMemberNames())
            {
                this.OnPropertyChanged(propertyName);
            }
        }

        /// <summary>
        /// プロパティのデフォルト値を現在の値に更新します。
        /// </summary>
        public void UpdatePropertyDescriptors()
        {
            if (this.proxy != null)
            {
                this.proxy.UpdateDataModelPropertyDescriptorMap();
            }

            this.UpdateViewModelPropertyDescriptorMap();
            foreach (var propertyName in this.EnumerateViewModelPropertyNames().AsParallel())
            {
                this.UpdateOriginalValue(propertyName);
            }
        }

        /// <summary>
        /// DataModelProxyの更新
        /// </summary>
        /// <param name="dataModel">データモデル.</param>
        public void UpdateDataModelProxy(DataModelBase dataModel)
        {
            this.proxy = this.CreateDataModelProxy(dataModel);

            this.UpdatePropertyDescriptors();
        }

        /// <summary>
        /// Helper method for getting property value from data model.
        /// Usage:
        /// <code>return this.GetDataModelValue(() => this.PatternDivU);</code>
        /// </summary>
        /// <typeparam name="T">The type of the property.</typeparam>
        /// <param name="propertyExp">The view model property.</param>
        /// <returns>The data model property value.</returns>
        public T GetDataModelValue<T>(Expression<Func<T>> propertyExp)
        {
            string propertyName = (propertyExp.Body as MemberExpression).Member.Name;

            return this.GetDataModelValue<T>(propertyName);
        }

        /// <summary>
        /// Helper method for getting property value from data model.
        /// Usage:
        /// <code>return this.GetDataModelValue("PatternDivU");</code>
        /// </summary>
        /// <typeparam name="T">The type of the property.</typeparam>
        /// <param name="propertyName">The property name.</param>
        /// <returns>The data model property value.</returns>
        public T GetDataModelValue<T>(string propertyName)
        {
            if (this.proxy == null)
            {
                return default(T);
            }

            var binder = new EffectMakerGetMemberBinder(propertyName);

            object value;
            this.proxy.TryGetMember(binder, out value);

            return (T)value;
        }

        /// <summary>
        /// Helper method for setting value to data model property.
        /// Usage:
        /// <code>this.SetDataModelValue(value, () => this.PatternDivU);</code>
        /// </summary>
        /// <typeparam name="T">The type of the property.</typeparam>
        /// <param name="value">The value to set.</param>
        /// <param name="propertyExp">The view model property.</param>
        /// <returns>True on success.</returns>
        public bool SetDataModelValue<T>(T value, Expression<Func<T>> propertyExp)
        {
            return this.SetDataModelValue<T>(value, NameOf(propertyExp));
        }

        /// <summary>
        /// Helper method for setting value to data model property.
        /// Usage:
        /// <code>this.SetDataModelValue(value, "PatternDivU");</code>
        /// </summary>
        /// <typeparam name="T">The type of the property.</typeparam>
        /// <param name="value">The value to set.</param>
        /// <param name="propertyName">The property name.</param>
        /// <returns>True on success.</returns>
        public bool SetDataModelValue<T>(T value, string propertyName)
        {
            var binder = new EffectMakerSetMemberBinder(propertyName, false, true);
            return this.TrySetMember(binder, value);
        }

        /// <summary>
        /// 式木からメンバ名文字列を得ます。
        /// C#6.0ではnameofに置き換え可能です。
        /// </summary>
        /// <typeparam name="T">式木型</typeparam>
        /// <param name="propertyExp">式木</param>
        /// <returns>メンバ名文字列</returns>
        protected static string NameOf<T>(Expression<Func<T>> propertyExp)
        {
            Debug.Assert(propertyExp != null, "プロパティの指定が不正");

            var body = propertyExp.Body as MemberExpression;
            Debug.Assert(body != null, "プロパティの指定が不正");

            return body.Member.Name;
        }

        /// <summary>
        /// 式木で示された名前とプロパティ変更イベントの対象が一致するかを判定します。
        /// </summary>
        /// <typeparam name="T">プロパティの型</typeparam>
        /// <param name="e">プロパティ変更イベント引数</param>
        /// <param name="propertyExp">プロパティの式木</param>
        /// <returns>一致すればtrue,そうでなければfalse.</returns>
        protected static bool IsRaisedProperty<T>(PropertyChangedEventArgs e, Expression<Func<T>> propertyExp)
        {
            return e.PropertyName == NameOf(propertyExp);
        }

        /// <summary>
        /// Check if the property value has been modified.
        /// This method is and should only be used for comparing property values
        /// with their original value and default value.
        /// </summary>
        /// <param name="propertyName">The property name.</param>
        /// <param name="origValue">The original/default value.</param>
        /// <param name="currValue">The current value.</param>
        /// <returns>True if the value has been modified.</returns>
        protected virtual bool CheckPropertyValueModification(
            string propertyName,
            object origValue,
            object currValue)
        {
            // Check if the value is a sequence of Guid.
            // If it is, the property is probably a data model collection, in which
            // the elements are the Guid of the data models.
            if (origValue is IEnumerable<Guid>)
            {
                var origGuidList = (IEnumerable<Guid>)origValue;
                var currGuidList = DataModelProxy.ExtractDataModelGuids(currValue) as IEnumerable<Guid>;
                if (currGuidList == null)
                {
                    // origがnot nullでcurrがnullなら、それは変化があったことになるのでtrue.
                    return true;
                }

                return !currGuidList.UnorderedSequenceEqual(origGuidList);
            }
            else
            {
                // Compare the value and see if it changed.
                return !object.Equals(DataModelProxy.ExtractDataModelGuids(currValue), origValue);
            }
        }

        /// <summary>
        /// Manually sets the current value of the specified property as the original value.
        /// </summary>
        /// <param name="propertyName">The name of the property.</param>
        protected void UpdateOriginalValue([CallerMemberName]string propertyName = null)
        {
            // Get the original value.
            var getMemberBinder = new EffectMakerGetMemberBinder(propertyName, true);

            object origValue;
            if (this.TryGetMember(getMemberBinder, out origValue) == false)
            {
                Debug.Assert(false, string.Format("ViewModelBase.UpdateOriginalValue : Failed getting member '{0}' from view model '{1}' for modification checking.", propertyName, this.GetType().Name));
                return;
            }

            // Check if the property belongs to the view model.
            PropertyData propertyData = null;
            if (this.viewModelPropInfoMap.TryGetValue(propertyName, out propertyData) == false)
            {
                // Not the view model's property, no need to process it.
                return;
            }

            // Store the original value of the property.
            if (propertyData != null &&
                propertyData.UseDataModelPropertyOriginalValue == false &&
                propertyData.IsOriginalValueSet == false)
            {
                // Clone the original value if possible.
                if (origValue is ICloneable)
                {
                    origValue = (origValue as ICloneable).Clone();
                }

                propertyData.OriginalValue = origValue;
            }
        }

        /// <summary>
        /// Helper method to set value to a property in the data model proxy.
        /// </summary>
        /// <param name="value">The value to set.</param>
        /// <param name="issueCommand">True to issue command.</param>
        /// <param name="propertyName">The name of the property.</param>
        /// <returns>True on success.</returns>
        protected bool SetProxyProperty(
            object value,
            bool issueCommand = true,
            [CallerMemberName]string propertyName = null)
        {
            var binder = new EffectMakerSetMemberBinder(propertyName, false, issueCommand);
            return this.TrySetMember(binder, value);
        }

        /// <summary>
        /// Create a data model proxy.
        /// This method is called in the constructor.
        /// If you need a specific type of data model proxy,
        /// override this method and return the desired data model proxy.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <returns>The created data model proxy.</returns>
        protected virtual DataModelProxy CreateDataModelProxy(DataModelBase dataModel)
        {
            return new DataModelProxy(dataModel);
        }

        /// <summary>
        /// オプションを変更したときの処理を行います.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The event arguments.</param>
        protected virtual void OnOptionChanged(object sender, EventArgs e)
        {
            this.OnPropertyChanged(() => this.IsSingleColumn);
            this.OnPropertyChanged(() => this.IsLinearMode);
        }

        /// <summary>
        /// Attributes a value to a field.
        /// Raises a PropertyChanged event if the field value changed.
        /// </summary>
        /// <typeparam name="T">Type of the property.</typeparam>
        /// <param name="field">The field that stores the property value.</param>
        /// <param name="value">The new value to set.</param>
        /// <param name="propertyName">The name of the property to set.</param>
        /// <returns>Returns true if the value changed, false otherwise.</returns>
        protected bool SetValue<T>(
            ref T field,
            T value,
            [CallerMemberName]string propertyName = null)
        {
            if (EqualityComparer<T>.Default.Equals(field, value))
            {
                return false;
            }

            var settable = field as ISettable;
            if (settable != null)
            {
                settable.Set(value);
            }
            else
            {
                field = value;
            }

            this.OnPropertyChanged(propertyName);

            return true;
        }

        /// <summary>
        /// Method to be called when the value of a property changes.
        /// </summary>
        /// <param name="propertyName">The name of the property that changed.</param>
        protected virtual void OnPropertyChanged([CallerMemberName]string propertyName = null)
        {
            var handler = this.PropertyChanged;
            if (handler != null)
            {
                handler(this, new PropertyChangedEventArgs(propertyName));
            }
        }

        /// <summary>
        /// Method to be called when the value of a property changes.
        /// </summary>
        /// <typeparam name="T">プロパティ型</typeparam>
        /// <param name="propertyExp">式木によるプロパティ指定.</param>
        protected virtual void OnPropertyChanged<T>(Expression<Func<T>> propertyExp)
        {
            this.OnPropertyChanged(NameOf(propertyExp));
        }

        /// <summary>
        /// Enumerate property names of the view model.
        /// </summary>
        /// <returns>The property names.</returns>
        protected IEnumerable<string> EnumerateViewModelPropertyNames()
        {
            return this.viewModelPropInfoMap.Keys;
        }

        /// <summary>
        /// Update the property descriptor map for the this view model.
        /// </summary>
        protected virtual void UpdateViewModelPropertyDescriptorMap()
        {
            this.viewModelPropInfoMap.Clear();

            Type myType = this.GetType();

            var properties = myType.GetProperties();

            foreach (PropertyInfo propertyInfo in properties.Distinct(
                new PropertyEqualityComparer()))
            {
                if (IgnoredViewModelProperties.Contains(propertyInfo.Name) == true)
                {
                    continue;
                }

                this.viewModelPropInfoMap.Add(
                    propertyInfo.Name,
                    new PropertyData(propertyInfo));
            }
        }

        /// <summary>
        /// Called when value changes.
        /// </summary>
        /// <param name="parameter">Custom parameter provided to the executable.
        /// Should be of type ValueChangedExEventArgs.</param>
        protected void OnValueChanged(object parameter)
        {
            var e = parameter as ValueChangedExEventArgs;
            if (e == null || e.IsChanging)
            {
                return;
            }

            string propertyName = e.CustomParameter as string;
            if (propertyName == null)
            {
                return;
            }

            var binder = new EffectMakerSetMemberBinder(propertyName, true, e.IssueCommand)
            {
                OriginalValue = e.OldValue
            };

            // OnValueChangedが呼ばれる時はUI側は同値であるので、打ち返しを防ぐためにイベント発生を抑制する
            // ※UISliderのボタン・テキストボックス操作時にModificationFlagが更新されない不具合があり、無効化中
            ////this.suppressPropertyChangedNames.Add(binder.Name);
            this.TrySetMember(binder, e.NewValue);
            ////this.suppressPropertyChangedNames.Remove(binder.Name);
        }

        /// <summary>
        /// 値のセット処理のコアです。
        /// </summary>
        /// <param name="binder">バインダー</param>
        /// <param name="effectMakerBinder">拡張バインダー</param>
        /// <param name="value">設定値</param>
        /// <param name="propertyData"></param>
        /// <returns>セットできたらtrue,できなかったらfalse.</returns>
        private bool SetValueCore(
            SetMemberBinder binder,
            EffectMakerSetMemberBinder effectMakerBinder,
            object value,
            out PropertyData propertyData)
        {
            // Set the new value...
            if (this.viewModelPropInfoMap.TryGetValue(binder.Name, out propertyData) == true &&
                effectMakerBinder != null &&
                effectMakerBinder.ShouldSetNonDynamicMember == true)
            {
                // Try to convert the value to correct type.
                var canAssign = TypeConversionUtility.TryConvert(
                    value.GetType(),
                    propertyData.Info.PropertyType,
                    ref value);

                if (canAssign == false)
                {
                    Logger.Log(
                        LogLevels.Error,
                        "ViewModelBase.TrySetMember : Failed setting incompatible value type to member '{0}' of view model '{1}'",
                        binder.Name,
                        this.GetType().Name);
                    return false;
                }

                propertyData.Info.SetValue(this, value, null);
            }
            else if (this.proxy.TrySetMember(binder, value) == false)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// プロパティ情報と、ロードしたデータを格納するクラス.
        /// </summary>
        protected class PropertyData
        {
            /// <summary>The original value of the property.</summary>
            private object originalValue = null;

            /// <summary>
            /// Initializes the PropertyData instance.
            /// </summary>
            /// <param name="propertyInfo">The PropertyInfo instance.</param>
            public PropertyData(PropertyInfo propertyInfo)
            {
                this.Info = propertyInfo;

                var useDataModelOriginalValueAttr =
                    propertyInfo.GetCustomAttribute<UseDataModelOriginalValueAttribute>();

                if (useDataModelOriginalValueAttr != null)
                {
                    this.UseDataModelPropertyOriginalValue = true;

                    string propName = useDataModelOriginalValueAttr.DataModelPropertyName;
                    this.DataModelPropertyNameForOriginalValue =
                        string.IsNullOrEmpty(propName) ? propertyInfo.Name : propName;
                }
                else
                {
                    this.UseDataModelPropertyOriginalValue = false;
                    this.DataModelPropertyNameForOriginalValue = string.Empty;
                }
            }

            /// <summary>プロパティ情報.</summary>
            public PropertyInfo Info { get; private set; }

            /// <summary>ロード時点の値.</summary>
            public object OriginalValue
            {
                get
                {
                    return this.originalValue;
                }

                set
                {
                    this.originalValue = value;
                    this.IsOriginalValueSet = true;
                }
            }

            /// <summary>
            /// Get the flag indicating whether to use the data model property
            /// to acquire the original value.
            /// </summary>
            public bool UseDataModelPropertyOriginalValue { get; private set; }

            /// <summary>
            /// The name of the data model property to get the original value from.
            /// </summary>
            public string DataModelPropertyNameForOriginalValue { get; private set; }

            /// <summary>
            /// Get the flag indicating whether the original value is set.
            /// </summary>
            public bool IsOriginalValueSet { get; private set; }
        }
    }

    /// <summary>
    /// Base class for the view models.
    /// </summary>
    /// <typeparam name="DataModelType">
    /// The type of the data model the view model accepts.
    /// </typeparam>
    public abstract class ViewModelBase<DataModelType> : ViewModelBase
        where DataModelType : DataModelBase
    {
        /// <summary>
        /// Initializes the ViewModeBase instance.
        /// </summary>
        /// <param name="dataModel">The data model to encapsulate.</param>
        protected ViewModelBase(DataModelType dataModel)
            : base(dataModel)
        {
        }
    }

    /// <summary>
    /// プロパティ名の比較.
    /// </summary>
    public class PropertyEqualityComparer : IEqualityComparer<PropertyInfo>
    {
        #region IEqualityComparer<PropertyInfo> Members

        /// <summary>
        /// 比較する.
        /// </summary>
        /// <param name="x">比較対象左辺値.</param>
        /// <param name="y">比較対象右辺値.</param>
        /// <returns>同名の場合、trueを返す.</returns>
        public bool Equals(PropertyInfo x, PropertyInfo y)
        {
            if (x == null || y == null)
            {
                return false;
            }

            return x.Name == y.Name;
        }

        /// <summary>
        /// ハッシュコードの取得.
        /// </summary>
        /// <param name="obj">取得対象のオブジェクト.</param>
        /// <returns>ハッシュコード.</returns>
        public int GetHashCode(PropertyInfo obj)
        {
            if (obj == null)
            {
                return 0;
            }

            return obj.Name.GetHashCode();
        }

        #endregion
    }
}
