﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Dynamic;
using System.IO;
using System.Linq;
using System.Xml.Linq;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Manager;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// The view model for the reserved shader user data models.
    /// </summary>
    public class ReservedShaderUserDataViewModel : UserGroupViewModel<UserDataBase>, IModificationFlagOwner
    {
        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        /// <param name="index">The index of the user data.</param>
        public ReservedShaderUserDataViewModel(
            HierarchyViewModel parent,
            UserDataBase dataModel,
            int index)
            : base(parent, dataModel)
        {
            this.Index = index;
        }

        /// <summary>
        /// Get the index of the reserved shader user data.
        /// (not the callback ID, but the index in the list.)
        /// </summary>
        public int Index { get; private set; }

        /// <summary>
        /// Get the XAML file path for the UI definitions.
        /// </summary>
        public string XamlFilePath
        {
            get
            {
                UserDataInfo info =
                    ReservedShaderUserDataManager.FindReservedShaderUserDataInfo(this.Index);
                if (info == null)
                {
                    return string.Empty;
                }

                return info.XamlFilePath;
            }
        }

        /// <summary>
        /// 型名をエミッタプラグイン固有の型名に置き換えてエクスポートします。
        /// </summary>
        /// <returns>XML化したデータ</returns>
        public override string ExportValuesAsXml()
        {
            Type type = this.DataModel.GetType();

            var root = new XElement(
                "root",
                new XAttribute("type", type.FullName));

            try
            {
                this.ExportValues(new XmlElementContainer(root));
            }
            catch
            {
                // TODO: report the error
                return null;
            }

            return root.ToString();
        }

        /// <summary>
        /// Set values based on a XML content.
        /// </summary>
        /// <param name="xmlContent">The XML content to set values from.</param>
        /// <returns>Returns true if import succeeded, false otherwise.</returns>
        public override bool ImportValuesFromXml(string xmlContent)
        {
            XElement root = null;

            try
            {
                root = XElement.Parse(xmlContent);
            }
            catch
            {
                // TODO: report the error
                return false;
            }

            Type type = this.DataModel.GetType();

            if ((string)root.Attribute("type") != type.FullName)
            {
                // TODO: report the error ?
                return false;
            }

            try
            {
                this.ImportValues(root.Elements());
            }
            catch
            {
                // TODO: report the error
                return false;
            }

            return true;
        }

        /// <summary>
        /// 同じエミッタプラグインかどうかまでチェックした上で、ペーストの可否を判定します。
        /// </summary>
        /// <param name="xmlContent">The XML content to set values from.</param>
        /// <returns>Returns true if can import, false otherwise.</returns>
        public override bool CanImportValuesFromXml(string xmlContent)
        {
            XElement root = null;

            try
            {
                root = XElement.Parse(xmlContent);
            }
            catch
            {
                return false;
            }

            Type type = this.DataModel.GetType();

            if ((string)root.Attribute("type") != type.FullName)
            {
                return false;
            }

            return true;
        }
    }
}
