﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using EffectMaker.DataModel.DataModels;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UILogic.Commands;
using EffectMaker.BusinessLogic.Options;

using Clipboard = System.Windows.Clipboard;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Common class for property pages.
    /// </summary>
    public abstract class PropertyPageViewModel : HierarchyViewModel
    {
        /// <summary>The constant name used to identify the clipboard data.</summary>
        private const string PropertyPageCopyDataClipboardName = "EffectMakerPropertyPageCopyData";

        /// <summary>
        /// Initializes the PropertyPageViewModel instance.
        /// </summary>
        /// <param name="parent">Parent view model.</param>
        /// <param name="dataModel">Data model to represent.</param>
        protected PropertyPageViewModel(HierarchyViewModel parent, DataModelBase dataModel)
            : base(parent, dataModel)
        {
            this.CopyPropertyPageExecutable = new AnonymousExecutable(this.OnCopyPropertyPage);
            this.PastePropertyPageExecutable = new AnonymousExecutable(
                this.OnPastePropertyPage,
                this.CanPastePropertyPage);

            this.ResetDefaultValueExecutable = new AnonymousExecutable(this.OnResetDefaultValues);
        }

        /// <summary>
        /// Gets the Copy executable instance.
        /// </summary>
        public IExecutable CopyPropertyPageExecutable { get; private set; }

        /// <summary>
        /// Gets the Paste executable instance.
        /// </summary>
        public IExecutable PastePropertyPageExecutable { get; private set; }

        /// <summary>
        /// Gets the executable for resetting the page data to its default values.
        /// </summary>
        public IExecutable ResetDefaultValueExecutable { get; private set; }

        /// <summary>
        /// Called when the Copy executable is run.
        /// </summary>
        protected virtual void OnCopyPropertyPage()
        {
            Clipboard.SetData(PropertyPageCopyDataClipboardName, this.ExportValuesAsXml());
        }

        /// <summary>
        /// Called when the Paste executable is run.
        /// </summary>
        /// <param name="parameter">The executable parameters.</param>
        protected virtual void OnPastePropertyPage(object parameter)
        {
            var clipboardData = Clipboard.GetData(PropertyPageCopyDataClipboardName) as string;
            if (clipboardData == null)
            {
                return;
            }

            var args = new object[]
            {
                this,
                this.ExportValuesAsXml(),
                clipboardData,
            };

            CommandManager.Execute(new PastePropertyPageCommand(this, this.ExportValuesAsXml(), clipboardData));
        }

        /// <summary>
        /// Check if the text contents in the clipboard can be pasted to this property page.
        /// </summary>
        /// <param name="parameter">The executable parameters.</param>
        /// <returns>True if the text contents in the clipboard can be pasted.</returns>
        protected virtual bool CanPastePropertyPage(object parameter)
        {
            // EffectConverter.exeから実行したときはThreadStateExceptionになるためスキップする
            if (OptionStore.RuntimeOptions.IsCommandLineMode)
            {
                return false;
            }

            if (Clipboard.ContainsData(PropertyPageCopyDataClipboardName) == false)
            {
                return false;
            }

            string clipboardData = Clipboard.GetData(PropertyPageCopyDataClipboardName) as string;

            if (clipboardData == null)
            {
                return false;
            }

            return this.CanImportValuesFromXml(clipboardData);
        }

        /// <summary>
        /// Called to reset the page data to its default values.
        /// </summary>
        /// <param name="parameter">The executable parameters.</param>
        protected virtual void OnResetDefaultValues(object parameter)
        {
            CommandManager.Execute(new ResetDefaultValueCommand(this));
        }
    }

    /// <summary>
    /// ジェネリック版のPropertyPageViewModelです。
    /// </summary>
    /// <typeparam name="T">データモデルの型</typeparam>
    public abstract class PropertyPageViewModel<T> : PropertyPageViewModel where T : DataModelBase
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="parent">親ビューモデル</param>
        /// <param name="dataModel">関連付けるデータモデル</param>
        protected PropertyPageViewModel(HierarchyViewModel parent, T dataModel)
            : base(parent, dataModel)
        {
        }

        /// <summary>
        /// 関連付けられたデータモデルを取得します。
        /// </summary>
        public T DataModel
        {
            get { return this.Proxy.DataModel; }
        }
    }
}
