﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Collections.Generic;
using System.IO;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.Attributes;
using EffectMaker.UILogic.Commands;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// ModelBasicVisibilityAnimationData に対応するビューモデルです。
    /// </summary>
    public class ModelBasicVisibilityAnimationViewModel : PropertyGroupViewModel<ModelBasicVisibilityAnimationData>, IModificationFlagOwner
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="parent">親ビューモデル</param>
        /// <param name="dataModel">データモデル</param>
        public ModelBasicVisibilityAnimationViewModel(HierarchyViewModel parent, ModelBasicVisibilityAnimationData dataModel)
            : base(parent, dataModel)
        {
            this.AnimationFileItems = new List<KeyValuePair<string, object>>
            {
                new KeyValuePair<string, object>(Properties.Resources.ModelBasicAnimationDefault, string.Empty)
            };

            this.ReloadAnimationFolderExecutable = new AnonymousExecutable(this.ReloadAnimationFolder);
            this.ClearAnimationFolderExecutable = new AnonymousExecutable(this.ClearAnimationFolder);

            // ModificationFlagViewModel を初期化
            // 初期化中に変更イベントが発生するのを防ぐため、必ずコンストラクタの最後で行ってください。
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
        }

        /// <summary>
        /// アニメーションファイルの選択項目を取得します。
        /// </summary>
        public List<KeyValuePair<string, object>> AnimationFileItems { get; private set; }

        /// <summary>
        /// アニメーションフォルダをリロードするための IExecutable を取得します。
        /// </summary>
        public IExecutable ReloadAnimationFolderExecutable { get; private set; }

        /// <summary>
        /// アニメーションフォルダをクリアするための IExecutable を取得します。
        /// </summary>
        public IExecutable ClearAnimationFolderExecutable { get; private set; }

        /// <summary>
        /// アニメーションフォルダパスを取得または設定します。
        /// </summary>
        [UseDataModelOriginalValue]
        public string AnimationFolderPath
        {
            get { return this.GetDataModelValue(() => this.AnimationFolderPath); }

            set
            {
                if (this.SetDataModelValue(value, () => this.AnimationFolderPath))
                {
                    this.UpdateAnimationFileItems(value);
                }
            }
        }

        /// <summary>
        /// アニメーションファイルパスを取得または設定します。
        /// </summary>
        [UseDataModelOriginalValue]
        public string AnimationFilePath
        {
            get { return this.GetDataModelValue(() => this.AnimationFilePath); }

            set
            {
                if (this.SetDataModelValue(value, () => this.AnimationFilePath))
                {
                    this.SendBinary();
                }
            }
        }

        /// <summary>
        /// ModificationFlagViewModel を取得します。
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// 全てのプロパティについての PropertyChanged イベントを発生させます。
        /// ビューモデルを通さずにデータモデルの値を変更したときに使用します。
        /// </summary>
        public override void FirePropertyChanges()
        {
            base.FirePropertyChanges();

            string animationFilePath = this.AnimationFilePath;
            this.ReloadAnimationFolder(null);
            this.AnimationFilePath = animationFilePath;
        }

        /// <summary>
        /// 子ビューモデルを更新します。
        /// </summary>
        public override void UpdateChildViewModels()
        {
            base.UpdateChildViewModels();

            string animationFilePath = this.AnimationFilePath;
            this.ReloadAnimationFolder(null);

            var binder = new EffectMakerSetMemberBinder(nameof(this.AnimationFilePath), false, false);
            if (this.TrySetMember(binder, animationFilePath))
            {
                this.SendBinary();
            }
        }

        /// <summary>
        /// ビューアにモデルバイナリを送信します。
        /// </summary>
        private void SendBinary()
        {
            var modelViewModel = (ModelViewModel)this.Parent.Parent;
            modelViewModel.SendModelInfo();
        }

        /// <summary>
        /// アニメーションファイルの選択項目を更新します。
        /// </summary>
        /// <param name="animationFolderPath">アニメーションフォルダパス</param>
        private void UpdateAnimationFileItems(string animationFolderPath)
        {
            const string filter = "*.fvbb";
            var binder = new EffectMakerSetMemberBinder(nameof(this.AnimationFilePath), false, false);

            // アニメーションファイルの選択項目をリセット
            // インスタンスを変更しないと UI が更新されないため List を new する
            this.AnimationFileItems = new List<KeyValuePair<string, object>>
            {
                new KeyValuePair<string, object>(Properties.Resources.ModelBasicAnimationDefault, string.Empty)
            };

            // 指定されたアニメーションフォルダが有効かチェック
            if (string.IsNullOrEmpty(animationFolderPath) || Directory.Exists(animationFolderPath) == false)
            {
                this.OnPropertyChanged(nameof(this.AnimationFileItems));
                this.TrySetMember(binder, string.Empty);
                return;
            }

            bool found = false;
            DirectoryInfo animationDirInfo = new DirectoryInfo(animationFolderPath);

            // アニメーションファイルリストを取得
            foreach (DirectoryInfo dirInfo in PathUtility.EnumerateSubFolders(animationDirInfo))
            {
                foreach (FileInfo fileInfo in dirInfo.EnumerateFiles(filter))
                {
                    this.AnimationFileItems.Add(new KeyValuePair<string, object>(fileInfo.Name, fileInfo.FullName));

                    if (fileInfo.FullName == this.AnimationFilePath)
                    {
                        found = true;
                    }
                }
            }

            this.OnPropertyChanged(nameof(this.AnimationFileItems));
            this.TrySetMember(binder, found ? this.AnimationFilePath : string.Empty);

            this.SendBinary();
        }

        /// <summary>
        /// アニメーションフォルダのリロード処理を行います。
        /// </summary>
        /// <param name="parameter">パラメータ</param>
        private void ReloadAnimationFolder(object parameter)
        {
            this.UpdateAnimationFileItems(this.AnimationFolderPath);
        }

        /// <summary>
        /// アニメーションフォルダのクリア処理を行います。
        /// </summary>
        /// <param name="parameter">パラメータ</param>
        private void ClearAnimationFolder(object parameter)
        {
            using (new CommandCombiner())
            {
                this.AnimationFilePath = string.Empty;
                this.AnimationFolderPath = string.Empty;
            }
        }
    }
}
