﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace EffectMaker.UILogic.ViewModels.Extenders
{
    /// <summary>
    /// Extends functionalities for view models who owns property pages.
    /// E.g. workspace nodes and property pages who has sub pages.
    /// </summary>
    internal class PropertyPageOwnerViewModelExtender
    {
        /// <summary>
        /// A dictionary that maps the selected page index of the extended view model types.
        /// With this map, all extended view models of the same type selects the property
        /// page of the same index.
        /// </summary>
        private static readonly Dictionary<Type, int> ViewModelTypeSelectedPageMap =
            new Dictionary<Type, int>();

        /// <summary>The extended view model.</summary>
        private HierarchyViewModel extendedViewModel;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="extendedViewModel">The extended view model.</param>
        public PropertyPageOwnerViewModelExtender(HierarchyViewModel extendedViewModel)
        {
            if (extendedViewModel == null || (extendedViewModel is IPropertyPageOwner) == false)
            {
                throw new ArgumentException("The extended view model must be an IPropertyPageOwner and cannot be null.");
            }

            this.extendedViewModel = extendedViewModel;
        }

        /// <summary>
        /// Get the selected property page.
        /// </summary>
        public virtual HierarchyViewModel SelectedPropertyPage
        {
            get
            {
                var extendedViewModelType = this.extendedViewModel.GetType();
                var extendedPageOwner = (IPropertyPageOwner)this.extendedViewModel;

                // Find the selected property page view model index.
                int index;
                if (ViewModelTypeSelectedPageMap.TryGetValue(extendedViewModelType, out index) == false)
                {
                    // The selected index of the extended view model type has not
                    // been set, return the default child property page.
                    if (extendedPageOwner.DefaultPropertyPageViewModel == null)
                    {
                        return null;
                    }

                    index = this.IndexOfPropertyPage(
                        extendedPageOwner.DefaultPropertyPageViewModel as PropertyPageViewModel);

                    ViewModelTypeSelectedPageMap[extendedViewModelType] = index;
                }

                // Find the property page view model with the index.
                HierarchyViewModel selectedPage = this.FindChildPropertyPageByIndex(index);
                if (selectedPage == null)
                {
                    selectedPage = extendedPageOwner.DefaultPropertyPageViewModel;
                }

                return selectedPage;
            }

            set
            {
                var extendedViewModelType = this.extendedViewModel.GetType();
                var extendedPageOwner = (IPropertyPageOwner)this.extendedViewModel;

                if (value != null)
                {
                    // The property page view model might not be the direct child of
                    // this view model, find the correct page view model through its
                    // ascendants.
                    HierarchyViewModel directChild = null;
                    HierarchyViewModel vm = value;
                    while (vm != null)
                    {
                        if (this.extendedViewModel.Children.Contains(vm) == true)
                        {
                            directChild = vm;
                            break;
                        }

                        vm = vm.Parent as HierarchyViewModel;
                    }

                    var pageVM = directChild as PropertyPageViewModel;
                    if (pageVM != null)
                    {
                        ViewModelTypeSelectedPageMap[extendedViewModelType] = this.IndexOfPropertyPage(pageVM);
                    }
                    else
                    {
                        // Unable to find the child property page, just use the default page.
                        ViewModelTypeSelectedPageMap[extendedViewModelType] = this.IndexOfPropertyPage(
                            (PropertyPageViewModel)extendedPageOwner.DefaultPropertyPageViewModel);
                    }

                    // The assigned property page is a child of the selected page,
                    // ask the selected page to select its sub page.
                    if (value != directChild &&
                        directChild is IPropertyPageOwner)
                    {
                        ((IPropertyPageOwner)directChild).SelectedPropertyPage = value;
                    }
                }
                else if (extendedPageOwner.DefaultPropertyPageViewModel != null)
                {
                    // The view model is null, just use the default page.
                    ViewModelTypeSelectedPageMap[extendedViewModelType] = this.IndexOfPropertyPage(
                        (PropertyPageViewModel)extendedPageOwner.DefaultPropertyPageViewModel);
                }
            }
        }

        /// <summary>
        /// Find the index of the specified property page
        /// from the children of the extended view model.
        /// </summary>
        /// <param name="vm">The property page view model.</param>
        /// <returns>The index of the property page view model.</returns>
        private int IndexOfPropertyPage(PropertyPageViewModel vm)
        {
            int index = 0;
            foreach (var child in this.extendedViewModel.Children)
            {
                if (child is PropertyPageViewModel)
                {
                    if (child == vm)
                    {
                        return index;
                    }

                    ++index;
                }
            }

            return -1;
        }

        /// <summary>
        /// Find the child property page view model of the extended view model
        /// at the specified index.
        /// </summary>
        /// <param name="index">The index.</param>
        /// <returns>The property page view model.</returns>
        private PropertyPageViewModel FindChildPropertyPageByIndex(int index)
        {
            int currIndex = 0;
            foreach (var child in this.extendedViewModel.Children)
            {
                if (child is PropertyPageViewModel)
                {
                    if (currIndex == index)
                    {
                        return (PropertyPageViewModel)child;
                    }

                    ++currIndex;
                }
            }

            return null;
        }
    }
}
