﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Texture;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.Attributes;
using EffectMaker.UILogic.Properties;
using EffectMaker.UILogic.ViewModels.IO;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterTextureFileViewModel.
    /// </summary>
    public class EmitterTextureFileViewModel : PropertyGroupViewModel<EmitterTextureFileData>, IModificationFlagOwner
    {
        /// <summary>
        /// ガンマ補正するか
        /// </summary>
        private const bool IsEnableGammaCorrection = false;

        /// <summary>
        /// コピペの対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignoreCopyPropertyNames = new string[]
        {
            "EnableUvChannel",
            "PatternAnimType",
            "RandomPatternTableCount",
            "UvChannelItems",
        };

        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignoreModifyPropertyNames = new string[]
        {
            "WarningMessages",
            "EnableUvChannel",
            "FilePathBgColor",
            "UvChannelItems",
        };

        /// <summary>
        /// 親エミッタセット
        /// </summary>
        private EmitterSetViewModel emitterSetViewModel = null;

        /// <summary>
        /// UVチャンネルの有効/無効
        /// </summary>
        private bool enableUvChannel;

        /// <summary>
        /// 再読み込み要求
        /// </summary>
        private bool requestReload;

        /// <summary>
        /// ファイルモニタイベントをロックするフラグ
        /// </summary>
        private bool monitorEventLock = false;

        /// <summary>
        /// 警告ウィンドウを表示するかどうか
        /// </summary>
        private bool onWarningWindow = true;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterTextureFileViewModel(
            HierarchyViewModel parent, EmitterTextureFileData dataModel)
            : base(parent, dataModel)
        {
            this.UvChannelItems = new KeyValuePair<string, object>[]
            {
                new KeyValuePair<string, object>(string.Format(Resources.EmitterTextureFileUvChannelName, 0), 0),
                new KeyValuePair<string, object>(string.Format(Resources.EmitterTextureFileUvChannelName, 1), 1),
            };

            this.WrapModeItems = new KeyValuePair<string, object>[]
            {
                new KeyValuePair<string, object>(Properties.Resources.EmitterWrapTypeMirror, 0),
                new KeyValuePair<string, object>(Properties.Resources.EmitterWrapTypeRepeat, 1),
                new KeyValuePair<string, object>(Properties.Resources.EmitterWrapTypeClamp, 2),
            };

            this.RepeatCountItems = new KeyValuePair<string, object>[]
            {
                new KeyValuePair<string, object>(Properties.Resources.EmitterRepeatCount1x1, 0),
                new KeyValuePair<string, object>(Properties.Resources.EmitterRepeatCount2x1, 1),
                new KeyValuePair<string, object>(Properties.Resources.EmitterRepeatCount1x2, 2),
                new KeyValuePair<string, object>(Properties.Resources.EmitterRepeatCount2x2, 3),
            };

            this.FilterModeItems = new KeyValuePair<string, object>[]
            {
                new KeyValuePair<string, object>(Properties.Resources.EmitterFilterModeLinear, 0),
                new KeyValuePair<string, object>(Properties.Resources.EmitterFilterModeNear, 1),
            };

            // エミッタだけ削除されたときに親エミッタセットがnullになるので先に取得しておく
            // エミッタ削除後にテクスチャファイル変更イベントを処理するときに必要
            this.emitterSetViewModel = GetParent<EmitterSetViewModel>(this);
            Debug.Assert(this.emitterSetViewModel != null, "親エミッタセットがnull");

            // 警告メッセージを初期化
            this.WarningMessages = new List<Tuple<string, Color, Font>>();
            this.UpdateWarningMessages(false);

            this.OnReloadTextureFileExecutable = new AnonymousExecutable(this.OnReloadTextureFile);

            // UVチャンネルの情報を初期化
            this.UpdateUvChannelItems();
            this.UpdateEnableUvChannel();

            // パターン分割のUIControlSetに関する設定を行なう.
            // UIControlSetのModificationFlagTargetをPatternDivUVにして、
            // PatternDivUVをPatternDivUとPatternDivVの親に設定する.
            var patternProperties =
                EnumerableUtility.Enumerate("PatternDivU", "PatternDivV");

            // ラップのUIControlSetに関する設定を行なう.
            // UIControlSetのModificationFlagTargetをWrapTypeUVにして、
            // WrapTypeUVをWrapTypeUとWrapTypeVの親に設定する.
            var wrapProperties =
                EnumerableUtility.Enumerate("WrapTypeU", "WrapTypeV");

            // ランダム反転のUIControlSetに関する設定を行なう.
            // UIControlSetのModificationFlagTargetをRandomInvUVにして、
            // RandomInvUVをRandomInvUとRandomInvVの親に設定する.
            var randomInvProperties =
                EnumerableUtility.Enumerate("RandomInvU", "RandomInvV");

            this.AddIgnoreCopyProperties(this.ignoreCopyPropertyNames);

            BusinessLogic.Manager.TextureManager.Instance.FileReloaded += this.OnFileReloaded;

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            var multipleModificationFlagsViewModel = new MultipleModificationFlagsViewModel(this);

            multipleModificationFlagsViewModel.SetPropertyDictionary("PatternDivUV", patternProperties);
            multipleModificationFlagsViewModel.SetPropertyDictionary("WrapTypeUV", wrapProperties);
            multipleModificationFlagsViewModel.SetPropertyDictionary("RandomInvUV", randomInvProperties);

            multipleModificationFlagsViewModel.AddIgnoreProperties(this.ignoreModifyPropertyNames);
            this.ModificationFlagViewModel = multipleModificationFlagsViewModel;
        }

        /// <summary>
        /// テクスチャファイルをリロードするためのIExecutableを取得します。
        /// </summary>
        public IExecutable OnReloadTextureFileExecutable { get; private set; }

        /// <summary>
        /// UVチャンネルの項目を取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> UvChannelItems { get; private set; }

        /// <summary>
        /// ラップモードの項目を取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> WrapModeItems { get; private set; }

        /// <summary>
        /// 繰り返し数の項目を取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> RepeatCountItems { get; private set; }

        /// <summary>
        /// フィルタモードの項目を取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> FilterModeItems { get; private set; }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// ファイルパスを取得または設定します.
        /// </summary>
        [UseDataModelOriginalValue]
        public string FilePath
        {
            get
            {
                return this.GetDataModelValue<string>(() => this.FilePath);
            }

            set
            {
                if (!string.IsNullOrEmpty(this.FilePath) && Path.IsPathRooted(this.FilePath))
                {
                    this.emitterSetViewModel.FileMonitor.UnregisterMonitor(this.FilePath);
                    this.emitterSetViewModel.TextureFileChanged -= this.OnTextureFileChanged;
                }

                if (!string.IsNullOrEmpty(value) && Path.IsPathRooted(value))
                {
                    this.emitterSetViewModel.FileMonitor.RegisterMonitorPreCopyPath(value, null, this.OnMonitorEvent);
                    this.emitterSetViewModel.TextureFileChanged += this.OnTextureFileChanged;
                }

                // 同じパスが渡された時は処理をスキップ(無用なスターマークの原因)
                if (value == this.FilePath)
                {
                    return;
                }

                if (this.SetDataModelValue(value, () => this.FilePath))
                {
                    this.UpdateWarningMessages(true);
                }

                if (this.onWarningWindow == true)
                {
                    this.CheckLinearMode();
                    this.CheckLocatable();
                }
            }
        }

        /// <summary>
        /// RGBを2乗して取得するか否かを取得または設定します。
        /// </summary>
        [UseDataModelOriginalValue]
        public bool IsSquareRGB
        {
            get
            {
                return this.GetDataModelValue(() => this.IsSquareRGB);
            }

            set
            {
                this.SetDataModelValue(value, () => this.IsSquareRGB);
                this.UpdateWarningMessages(true);
                if (this.onWarningWindow)
                {
                    this.CheckLinearMode();
                }
            }
        }

        /// <summary>
        /// ファイルパス入力テキストボックスの背景色です.
        /// </summary>
        public Color FilePathBgColor { get; set; }

        /// <summary>
        /// UVチャンネルの有効/無効を取得または設定します。
        /// </summary>
        public bool EnableUvChannel
        {
            get { return this.enableUvChannel; }
            set { this.SetValue(ref this.enableUvChannel, value); }
        }

        /// <summary>
        /// 再読み込み要求
        /// </summary>
        public bool RequestReload
        {
            get { return this.requestReload; }
            set { this.SetValue(ref this.requestReload, value); }
        }

        /// <summary>
        /// エミッタのタイプを設定または取得します。
        /// </summary>
        public int PatternAnimType
        {
            get
            {
                Debug.Assert(this.Parent is EmitterTextureViewModel, "Parent is not EmitterTextureViewModel.");
                return ((EmitterTextureViewModel)this.Parent).EmitterTexturePatternAnimationViewModel.PatternAnimType;
            }

            set
            {
                Debug.Assert(this.Parent is EmitterTextureViewModel, "Parent is not EmitterTextureViewModel.");
                ((EmitterTextureViewModel)this.Parent).EmitterTexturePatternAnimationViewModel.PatternAnimType = value;
            }
        }

        /// <summary>
        /// ランダム設定時の使用パターンテーブル数です。
        /// </summary>
        public int RandomPatternTableCount
        {
            get
            {
                Debug.Assert(this.Parent is EmitterTextureViewModel, "Parent is not EmitterTextureViewModel.");
                return ((EmitterTextureViewModel)this.Parent).EmitterTexturePatternAnimationViewModel.RandomPatternTableCountForViewModel;
            }

            set
            {
                Debug.Assert(this.Parent is EmitterTextureViewModel, "Parent is not EmitterTextureViewModel.");
                ((EmitterTextureViewModel)this.Parent).EmitterTexturePatternAnimationViewModel.RandomPatternTableCountForViewModel = value;
            }
        }

        /// <summary>
        /// テクスチャファイルに関する警告メッセージです.
        /// </summary>
        public List<Tuple<string, Color, Font>> WarningMessages { get; set; }

        /// <summary>
        /// Disposes the instance.
        /// </summary>
        public override void Dispose()
        {
            BusinessLogic.Manager.TextureManager.Instance.FileReloaded -= this.OnFileReloaded;
            if (!string.IsNullOrEmpty(this.FilePath))
            {
                this.emitterSetViewModel.FileMonitor.UnregisterMonitor(this.FilePath);
                this.emitterSetViewModel.TextureFileChanged -= this.OnTextureFileChanged;
            }

            base.Dispose();
        }

        /// <summary>
        /// Resend PropertyChanged notification for all properties.
        /// This is required when the data model changes independently from the view model.
        /// </summary>
        public override void FirePropertyChanges()
        {
            base.FirePropertyChanges();

            this.UpdateWarningMessages(true);

            // タブのコピペ時のみ、警告ウィンドウを出す
            if (ExportableViewModel.IsPasting == true)
            {
                this.CheckLinearMode();
                this.CheckLocatable();
            }
        }

        /// <summary>
        /// UVチャンネルの有効/無効を更新します。
        /// EmitterParticleShapeViewModelでパーティクル形状を変えたときに処理を行います。
        /// </summary>
        public void UpdateEnableUvChannel()
        {
            var emitter = GetParent<EmitterViewModel>(this);
            Debug.Assert(emitter != null, "emitter != null");

            this.EnableUvChannel = emitter.EmitterParticleViewModel.EmitterParticleShapeViewModel.ShapeType == 1;  // 1: Primitive
        }

        /// <summary>
        /// UVチャンネルの項目を更新します。
        /// EmitterParticleShapeViewModelでプリミティブ形状を変えたときや
        /// プリミティブをリロードしたときに処理を行います。
        /// </summary>
        public void UpdateUvChannelItems()
        {
            var emitter = GetParent<EmitterViewModel>(this);
            Debug.Assert(emitter != null, "emitter != null");
            string primitiveFilePath = emitter.EmitterParticleViewModel.EmitterParticleShapeViewModel.PrimitiveFilePath;

            // モデルデータを取得
            Foundation.Model.ModelData modelData = null;
            if (string.IsNullOrEmpty(primitiveFilePath) == false && File.Exists(primitiveFilePath))
            {
                EffectMaker.BusinessLogic.Manager.PrimitiveManager.Instance.LoadModelWithData(primitiveFilePath, true, out modelData);
            }

            // モデルが未指定のときはデフォルトの項目を作成
            if (modelData == null)
            {
                this.UvChannelItems = new KeyValuePair<string, object>[]
                {
                    new KeyValuePair<string, object>(string.Format(Resources.EmitterTextureFileUvChannelName, 0), 0),
                    new KeyValuePair<string, object>(string.Format(Resources.EmitterTextureFileUvChannelName, 1), 1),
                };

                this.OnPropertyChanged(() => this.UvChannelItems);
                return;
            }

            // モデルのUV名から項目を作成
            string channelName0 = modelData.TexCoordName0;
            channelName0 = string.IsNullOrEmpty(channelName0) ? string.Format(Resources.EmitterTextureFileUvChannelName, "0") : channelName0;

            string channelName1 = modelData.TexCoordName1;
            channelName1 = string.IsNullOrEmpty(channelName1) ? string.Format(Resources.EmitterTextureFileUvChannelName, "1") : channelName1;

            this.UvChannelItems = new KeyValuePair<string, object>[]
            {
                new KeyValuePair<string, object>(channelName0, 0),
                new KeyValuePair<string, object>(channelName1, 1),
            };

            this.OnPropertyChanged(() => this.UvChannelItems);
        }

        /// <summary>
        /// 警告ウィンドウを出すか指定して、FilePathを更新する
        /// </summary>
        /// <param name="filePath">テクスチャファイルのパス</param>
        /// <param name="warning">trueなら警告ウィンドウを表示する.falseなら警告ウィンドウを表示しない.</param>
        public void SetFilePathWithWarningWindow(string filePath, bool warning)
        {
            // warningがtrueだと警告ウィンドウを表示する.
            // warningがfalseだと警告ウィンドウを出さない。
            this.onWarningWindow = warning;

            // FilePathを更新する.
            this.FilePath = filePath;

            // 警告ウィンドウがでる状態に更新しておく.
            this.onWarningWindow = true;
        }

        /// <summary>
        /// Update child view models with the current data model.
        /// This method is usually called when data model is modified, thus some child
        /// view models might need to be created or removed according to the data model.
        /// </summary>
        public override void UpdateChildViewModels()
        {
            base.UpdateChildViewModels();
            this.UpdateWarningMessages(true);
        }

        /// <summary>
        /// テクスチャと編集モードの不一致が例外に該当するかどうかをチェックします。
        /// </summary>
        /// <param name="isLinearTexture">テクスチャのリニアモード</param>
        /// <returns>例外に該当するならtrue,そうでなければfalse.</returns>
        public bool CheckExceptionLinearWarning(bool isLinearTexture)
        {
            bool isLinearMode = OptionStore.ProjectConfig.LinearMode;

            // RGBの二乗取得の設定によっては警告を抑制する
            if (isLinearMode && !isLinearTexture && this.DataModel.IsSquareRGB)
            {
                return true;
            }

            // コンバイナの設定によってエミッタアイコンの差し替えを抑制する
            var emitterVm = this.FindNearestParentOfType<EmitterViewModel>();
            if (emitterVm != null &&
                emitterVm.EmitterCombinerViewModel != null &&
                emitterVm.EmitterTextureGroupViewModel != null)
            {
                // パーティクルシェーダタイプが歪みか屈折である
                var flag = emitterVm.EmitterCombinerViewModel.EmitterCombinerShaderViewModel.GetDataModelValue<int>("ShaderType") != 0;

                // テクスチャ0である
                flag &= emitterVm.EmitterTextureGroupViewModel.Texture0.EmitterTextureFileViewModel == this;

                // テクスチャが非リニアで、編集モードがリニアである
                flag &= !isLinearTexture;
                flag &= isLinearMode;

                // 以上の条件を満たす時は、テクスチャと編集モードが一致していなくても無効にはしない
                return flag;
            }

            return false;
        }

        /// <summary>
        /// 警告メッセージを更新します.
        /// </summary>
        /// <param name="noticeParentNode">親エミッタに通知するかどうかのフラグ</param>
        public void UpdateWarningMessages(bool noticeParentNode)
        {
            this.WarningMessages.Clear();
            bool unreachable = false, invalid = false;

            bool reachable, locatable;
            AssetsManager.GetAssetStatus(this.FilePath, this.emitterSetViewModel.FilePath, BusinessLogic.AssetResourceTypes.Texture, out reachable, out locatable);

            // ファイルパスがないとき
            if (reachable == false)
            {
                this.WarningMessages.Add(new Tuple<string, Color, Font>(
                    Resources.EmitterTextureFileWarningNotFound,
                    Color.Red,
                    null));

                this.FilePathBgColor = Color.LightPink;
                unreachable = true;
            }
            else
            {
                this.FilePathBgColor = Color.White;
            }

            // ロケートできないとき
            if (locatable == false)
            {
                this.WarningMessages.Add(new Tuple<string, Color, Font>(
                    Resources.EmitterTextureFileWarningUnreachable,
                    Color.SlateBlue,
                    null));

                // unreachableのフラグは立てない
            }

            // ファイルがあるとき
            if (reachable)
            {
                // テクスチャを取得
                var textureManager = EffectMaker.BusinessLogic.Manager.TextureManager.Instance;
                LoadTextureResult result = textureManager.LoadTexture(this.FilePath, true);

                if (result.ResultCode == LoadTextureResultCode.Success)
                {
                    bool isLinearMode = OptionStore.ProjectConfig.LinearMode;
                    bool isLinearTexture = result.TextureData.Linear.Any(x => x);

                    // テクスチャのリニアモードがオプションと違ったとき
                    if (isLinearTexture != isLinearMode)
                    {
                        this.FilePathBgColor = Color.FromArgb(176, 196, 243);

                        // 例外に該当するケースでなければ警告対象とする
                        if (!this.CheckExceptionLinearWarning(isLinearTexture))
                        {
                            if (isLinearTexture)
                            {
                                this.WarningMessages.Add(new Tuple<string, Color, Font>(
                                    Resources.EmitterTextureFileWarningLinear,
                                    Color.SlateBlue,
                                    null));
                            }
                            else
                            {
                                this.WarningMessages.Add(new Tuple<string, Color, Font>(
                                    Resources.EmitterTextureFileWarningNonLinear,
                                    Color.SlateBlue,
                                    null));
                            }

                            if (OptionStore.RootOptions.Basic.Emitters.ShowWarningIconLinearTexture)
                            {
                                invalid = true;
                            }
                        }
                    }

                    // snorm テクスチャは色表現がビューアと異なるので注意を表示する
                    if (result.TextureData.PixelFormat.IsSnorm())
                    {
                        this.WarningMessages.Add(new Tuple<string, Color, Font>(
                            string.Format(Properties.Resources.EmitterTextureFileNoticeSnorm, "snorm"),
                            Color.SlateBlue,
                            null));
                    }
                }
                else if (result.ResultCode == LoadTextureResultCode.InvalidFileVersion)
                {
                    this.WarningMessages.Add(new Tuple<string, Color, Font>(
                        Resources.EmitterTextureFileWarningInvalidFileVersion,
                        Color.Red,
                        null));

                    this.FilePathBgColor = Color.LightPink;
                }
            }

            // ノードアイコンを更新するため、エミッタのアセットステータスをアップデート
            if (noticeParentNode)
            {
                var emitterViewModel = ViewModelBase.GetParent<EmitterViewModel>((ViewModelBase)this.Parent);
                emitterViewModel.UpdateAssetStatus(unreachable, invalid);
            }

            this.OnPropertyChanged(() => this.WarningMessages);
            this.OnPropertyChanged(() => this.FilePathBgColor);
        }

        /// <summary>
        /// オプションを変更したときの処理を行います.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The event arguments.</param>
        protected override void OnOptionChanged(object sender, EventArgs e)
        {
            base.OnOptionChanged(sender, e);

            // リニアモードを変えたときのため警告メッセージを更新する
            this.UpdateWarningMessages(true);
        }

        /// <summary>
        /// ファイル監視システムのイベントがあった場合の処理
        /// </summary>
        /// <param name="path">監視パス</param>
        /// <param name="userData">ユーザーデータ(未使用)</param>
        /// <param name="watcherChangeTypes">変更タイプ</param>
        private void OnMonitorEvent(string path, object userData, WatcherChangeTypes watcherChangeTypes)
        {
            // テクスチャの自動リロードがOffになっていたら、何もしない
            if (OptionStore.RootOptions.Basic.Textures.AutoReload == false)
            {
                return;
            }

            if (this.monitorEventLock)
            {
                return;
            }

            // ファイル監視一時停止
            this.emitterSetViewModel.FileMonitor.UnregisterMonitor(this.FilePath);

            this.monitorEventLock = true;

            switch (watcherChangeTypes)
            {
                case WatcherChangeTypes.Changed:
                case WatcherChangeTypes.Created:
                    this.OnReloadTextureFile("FilePath");
                    break;
                case WatcherChangeTypes.Renamed:
                    if (File.Exists(path))
                    {
                        this.OnReloadTextureFile("FilePath");
                    }
                    else
                    {
                        this.UpdateWarningMessages(false);
                    }

                    break;
                case WatcherChangeTypes.Deleted:
                    this.UpdateWarningMessages(false);
                    break;
            }

            this.monitorEventLock = false;

            // ファイル監視復帰
            this.emitterSetViewModel.FileMonitor.RegisterMonitorPreCopyPath(this.FilePath, null, this.OnMonitorEvent);
            this.emitterSetViewModel.TriggerTextureFileChanged(this, path);
        }

        /// <summary>
        /// テクスチャファイルに変更があったとき、リロード処理を行った後の処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="path">テクスチャファイルパス</param>
        private void OnTextureFileChanged(object sender, string path)
        {
            if (sender == this || path != this.FilePath)
            {
                return;
            }

            this.UpdateWarningMessages(false);
        }

        /// <summary>
        /// テクスチャファイルのリロード処理を行います。
        /// </summary>
        /// <param name="parameter">Custom parameter</param>
        private void OnReloadTextureFile(object parameter)
        {
            Debug.Assert(parameter is string, "文字列である必要がある");

            var binder = new EffectMaker.Foundation.Dynamic.EffectMakerGetMemberBinder((string)parameter);

            object path = null;
            this.TryGetMember(binder, out path);

            Debug.Assert(path != null, "path != null");

            var pathString = path as string;
            Debug.Assert(string.IsNullOrEmpty(pathString) == false, "string.IsNullOrEmpty(path) == false");

            // リロード要求
            BusinessLogic.Manager.TextureManager.Instance.Reload(pathString);

            // 警告メッセージを更新
            this.UpdateWarningMessages(true);

            this.CheckLinearMode();
        }

        /// <summary>
        /// ファイルリロード後イベント
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void OnFileReloaded(object sender, FileReloadedEventArgs e)
        {
            // View側にリロード要求を発行する
            this.RequestReload = true;
            this.RequestReload = false;

            // 重複を避けるためバイナリコンバートの呼び出しはEmitterTextureGroupViewModelで処理する
        }

        /// <summary>
        /// テクスチャがエミッタセットから探索可能かどうかメッセージを出します.
        /// </summary>
        private void CheckLocatable()
        {
            // ファイルパスが指定されていないとき終了
            if (string.IsNullOrEmpty(this.FilePath))
            {
                return;
            }

            // ファイルパスが相対パスのとき終了
            if (Path.IsPathRooted(this.FilePath) == false)
            {
                return;
            }

            bool reachable, locatable;
            AssetsManager.GetAssetStatus(this.FilePath, this.emitterSetViewModel.FilePath, BusinessLogic.AssetResourceTypes.Texture, out reachable, out locatable);

            // エミッタセットからテクスチャを探索できないとき
            if (locatable == false)
            {
                WorkspaceRootViewModel.Instance.Dialogs.ShowWarningOnLoadTexture(string.Format(Resources.WarningTextureLocatable, Path.GetFileName(this.FilePath)));
            }
        }

        /// <summary>
        /// テクスチャをチェックしてリニアモードに不整合があったときメッセージを出します.
        /// </summary>
        private void CheckLinearMode()
        {
            // ファイルパスが指定されていないとき終了
            if (string.IsNullOrEmpty(this.FilePath))
            {
                return;
            }

            // ファイルパスが相対パスのとき終了
            if (Path.IsPathRooted(this.FilePath) == false)
            {
                return;
            }

            // テクスチャを取得
            var textureManager = EffectMaker.BusinessLogic.Manager.TextureManager.Instance;
            LoadTextureResult result = textureManager.LoadTexture(this.FilePath, true);

            if (result.ResultCode == LoadTextureResultCode.Success)
            {
                bool isLinearMode = OptionStore.ProjectConfig.LinearMode;
                bool isLinearTexture = result.TextureData.Linear.Any(x => x);

                // テクスチャのリニアモードがオプションと違ったときはダイアログを表示するが、
                // コンバイナの設定によっては抑制する
                if (isLinearTexture != isLinearMode && !this.CheckExceptionLinearWarning(isLinearTexture))
                {
                    WorkspaceRootViewModel.Instance.Dialogs.ShowTextureTypeMismatch(isLinearTexture
                        ? Resources.WarningTextureLinear
                        : Resources.WarningTextureNonlinear);
                }
            }
        }
    }
}
