﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Model.Types;
using EffectMaker.UILogic.Attributes;
using EffectMaker.UILogic.Properties;
using EffectMaker.UILogic.ViewModels.IO;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterParticleShapeViewModel.
    /// </summary>
    public class EmitterParticleShapeViewModel : PropertyGroupViewModel<EmitterParticleShapeData>, IModificationFlagOwner
    {
        /// <summary>
        /// コピペの対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignoreCopyPropertyNames = new string[]
        {
            "SamplePrimitiveFilePath",
            "CameraOffsetVisibility",
            "ConnectPtclScaleToZOffsetVisibility",
        };

        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignoreModifyPropertyNames = new string[]
        {
            "WarningMessages",
            "PrimitiveFilePathBgColor",
            "PrimitiveUvChannelNames",
            "CameraOffsetVisibility",
            "ConnectPtclScaleToZOffsetVisibility",
        };

        /// <summary>
        /// プロパティパネルで表示する頂点データの名前
        /// </summary>
        private readonly string[] vertexNames = new string[]
        {
            "pos",
            "normal",
            "tangent",
            "uv0",
            "uv1",
            "color",
        };

        /// <summary>
        /// 親エミッタセット
        /// </summary>
        private EmitterSetViewModel emitterSetViewModel = null;

        /// <summary>
        /// ファイルモニタイベントをロックするフラグ
        /// </summary>
        private bool monitorEventLock = false;

        /// <summary>
        /// 警告ウィンドウを表示するかどうか
        /// </summary>
        private bool onWarningWindow = true;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterParticleShapeViewModel(
            HierarchyViewModel parent, EmitterParticleShapeData dataModel)
            : base(parent, dataModel)
        {
            this.ParticleShapeItems = new KeyValuePair<string, object>[]
            {
                new KeyValuePair<string, object>(Properties.Resources.EmitterParticleShapeTypeParticle, 0),
                new KeyValuePair<string, object>(Properties.Resources.EmitterParticleShapeTypePrimitive, 1),
            };

            this.ParticleTypeItems = new KeyValuePair<string, object>[]
            {
                new KeyValuePair<string, object>(Properties.Resources.EmitterParticleTypeBillboard, 0),
                new KeyValuePair<string, object>(Properties.Resources.EmitterParticleTypeBillboardEx, 1),
                new KeyValuePair<string, object>(Properties.Resources.EmitterParticleTypeBillboardYParallel, 7),
                new KeyValuePair<string, object>(Properties.Resources.EmitterParticleTypeBillboardY, 2),
                new KeyValuePair<string, object>(Properties.Resources.EmitterParticleTypePlaneXY, 3),
                new KeyValuePair<string, object>(Properties.Resources.EmitterParticleTypePlaneXZ, 4),
                new KeyValuePair<string, object>(Properties.Resources.EmitterParticleTypeDirectionalY, 5),
                new KeyValuePair<string, object>(Properties.Resources.EmitterParticleTypeDirectionalPolygon, 6),

                // MEMO: 未使用機能につきUI封印中。新規機能を追加する場合は index = 10 から割り当てると無難。
                //new KeyValuePair<string, object>(Properties.Resources.EmitterParticleTypeWithScaleZ, 8),
                //new KeyValuePair<string, object>(Properties.Resources.EmitterParticleTypeWithScaleZDirectional, 9),
            };

            this.CameraOffsetProcessTypeItems = new KeyValuePair<string, object>[]
            {
                new KeyValuePair<string, object>(Properties.Resources.EmitterCameraOffsetTypeDisable, 2),
                new KeyValuePair<string, object>(Properties.Resources.EmitterCameraOffsetTypeDepth, 0),
                new KeyValuePair<string, object>(Properties.Resources.EmitterCameraOffsetTypeView, 1),
                new KeyValuePair<string, object>(Properties.Resources.EmitterCameraOffsetTypeFixedSize, 3),
            };

            // エミッタだけ削除されたときに親エミッタセットがnullになるので先に取得しておく
            // エミッタ削除後にプリミティブファイル変更イベントを処理するときに必要
            this.emitterSetViewModel = GetParent<EmitterSetViewModel>(this);
            Debug.Assert(this.emitterSetViewModel != null, "親エミッタセットがnull");

            // 警告メッセージを初期化
            this.WarningMessages = new List<Tuple<string, Color, Font>>();
            this.UpdateWarningMessages(false);

            // テクスチャタブのプリミティブUV情報を更新
            this.UpdatePrimitiveInfo();
            this.UpdateEnableUvChannel();

            this.OnReloadPrimitiveFileExecutable = new AnonymousExecutable(this.OnReloadPrimitiveFile);

            // コピペ時に含めないプロパティを登録する.
            this.AddIgnoreCopyProperties(this.ignoreCopyPropertyNames);

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
            this.ModificationFlagViewModel.AddIgnoreProperties(this.ignoreModifyPropertyNames);
        }

        /// <summary>
        /// パーティクル形状の項目を取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> ParticleShapeItems { get; private set; }

        /// <summary>
        /// パーティクルタイプの項目を取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> ParticleTypeItems { get; private set; }

        /// <summary>
        /// カメラオフセットタイプの項目を取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> CameraOffsetProcessTypeItems { get; private set; }

        /// <summary>
        /// プリミティブファイルをリロードするためのIExecutableを取得します。
        /// </summary>
        public IExecutable OnReloadPrimitiveFileExecutable { get; private set; }

        /// <summary>
        /// プリミティブファイルパスを取得または設定します。
        /// </summary>
        public string PrimitiveFilePath
        {
            get
            {
                return this.GetDataModelValue<string>(() => this.PrimitiveFilePath);
            }

            set
            {
                if (!string.IsNullOrEmpty(this.PrimitiveFilePath) && Path.IsPathRooted(this.PrimitiveFilePath))
                {
                    this.emitterSetViewModel.FileMonitor.UnregisterMonitor(this.PrimitiveFilePath);
                    this.emitterSetViewModel.PrimitiveFileChanged -= this.OnPrimitiveFileChanged;
                }

                if (!string.IsNullOrEmpty(value) && Path.IsPathRooted(value))
                {
                    this.emitterSetViewModel.FileMonitor.RegisterMonitorPreCopyPath(value, null, this.OnMonitorEvent);
                    this.emitterSetViewModel.PrimitiveFileChanged += this.OnPrimitiveFileChanged;
                }

                // 同じパスが渡された時は処理をスキップ(無用なスターマークの原因)
                if (value == this.PrimitiveFilePath)
                {
                    return;
                }

                if (this.SetDataModelValue(value, () => this.PrimitiveFilePath))
                {
                    this.UpdateWarningMessages();
                    this.UpdatePrimitiveInfo();
                }

                if (this.onWarningWindow == true)
                {
                    this.CheckLocatable();
                }
            }
        }

        /// <summary>
        /// サンプルプリミティブがあるディレクトリを返す
        /// </summary>
        public string SamplePrimitiveFilePath
        {
            get
            {
                return Path.Combine(IOConstants.ExecutableFolderPath, "Primitives");
            }

            set
            {
                this.onWarningWindow = true;
                this.PrimitiveFilePath = value;
            }
        }

        /// <summary>
        /// パーティクル形状を取得または設定します。
        /// </summary>
        [UseDataModelOriginalValue]
        public int ShapeType
        {
            get
            {
                return this.GetDataModelValue(() => this.ShapeType);
            }

            set
            {
                this.SetDataModelValue(value, () => this.ShapeType);

                // テクスチャタブのUVチャンネル有効/無効を更新
                {
                    var emitter = GetParent<EmitterViewModel>(this);
                    Debug.Assert(emitter != null, "emitter != null");

                    // EmitterTextureGroupViewModelよりもこちらが先に作成されるのでnullチェックしておく
                    if (emitter.EmitterTextureGroupViewModel != null)
                    {
                        bool enabled = this.ShapeType == 1;  // 1: Primitive
                        emitter.EmitterTextureGroupViewModel.Texture0.EmitterTextureFileViewModel.EnableUvChannel = enabled;
                        emitter.EmitterTextureGroupViewModel.Texture1.EmitterTextureFileViewModel.EnableUvChannel = enabled;
                        emitter.EmitterTextureGroupViewModel.Texture2.EmitterTextureFileViewModel.EnableUvChannel = enabled;
                    }
                }
            }
        }

        /// <summary>
        /// プリミティブファイルパス入力テキストボックスの背景色です.
        /// </summary>
        public Color PrimitiveFilePathBgColor { get; set; }

        /// <summary>
        /// プリミティブファイルに関する警告メッセージです.
        /// </summary>
        public List<Tuple<string, Color, Font>> WarningMessages { get; set; }

        /// <summary>
        /// カメラオフセットタイプ
        /// </summary>
        [UseDataModelOriginalValue]
        public int CameraOffsetProcessType
        {
            get
            {
                return this.GetDataModelValue(() => this.CameraOffsetProcessType);
            }

            set
            {
                this.SetDataModelValue(value, () => this.CameraOffsetProcessType);
                this.OnPropertyChanged(() => this.CameraOffsetVisibility);
                this.OnPropertyChanged(() => this.ConnectPtclScaleToZOffsetVisibility);
            }
        }

        /// <summary>
        /// カメラオフセット値の表示非表示
        /// </summary>
        public bool CameraOffsetVisibility
        {
            get { return this.CameraOffsetProcessType != 2; }
        }

        /// <summary>
        /// ZオフセットとPtclスケールの連動の表示非表示
        /// </summary>
        public bool ConnectPtclScaleToZOffsetVisibility
        {
            get { return this.CameraOffsetProcessType == 0 || this.CameraOffsetProcessType == 3; }
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// プリミティブの情報を表示します。
        /// </summary>
        public string PrimitiveProperty { get; private set; }

        /// <summary>
        /// Disposer
        /// </summary>
        public override void Dispose()
        {
            if (!string.IsNullOrEmpty(this.PrimitiveFilePath))
            {
                this.emitterSetViewModel.FileMonitor.UnregisterMonitor(this.PrimitiveFilePath);
                this.emitterSetViewModel.PrimitiveFileChanged -= this.OnPrimitiveFileChanged;
            }

            base.Dispose();
        }

        /// <summary>
        /// Update child view models with the current data model.
        /// This method is usually called when data model is modified, thus some child
        /// view models might need to be created or removed according to the data model.
        /// </summary>
        public override void UpdateChildViewModels()
        {
            base.UpdateChildViewModels();
            this.UpdateWarningMessages(true);
        }

        /// <summary>
        /// Resend PropertyChanged notification for all properties.
        /// This is required when the data model changes independently from the view model.
        /// </summary>
        public override void FirePropertyChanges()
        {
            base.FirePropertyChanges();

            this.UpdateWarningMessages();

            // タブのコピペ時の処理
            if (ExportableViewModel.IsPasting == true)
            {
                // タブのコピペ時は、ここで警告ウィンドウを出す.
                this.CheckLocatable();

                // テクスチャタブのUV情報を更新
                this.UpdatePrimitiveInfo();
                this.UpdateEnableUvChannel();
            }
        }

        /// <summary>
        /// 警告ウィンドウを出すか指定して、PrimitiveFilePathを更新する.
        /// </summary>
        /// <param name="filePath">プリミティブファイルのパス</param>
        /// <param name="warning">trueなら警告ウィンドウを表示する.falseなら警告ウィンドウを表示しない.</param>
        public void SetPrimitiveFilePathWithWarningWindow(string filePath, bool warning)
        {
            // warningがtrueだと警告ウィンドウを表示する.
            // warningがfalseだと警告ウィンドウを出さない。
            this.onWarningWindow = warning;

            // FilePathを更新する.
            this.PrimitiveFilePath = filePath;

            // 警告ウィンドウがでる状態に更新しておく.
            this.onWarningWindow = true;
        }

        /// <summary>
        /// ファイル監視システムのイベントがあった場合の処理
        /// </summary>
        /// <param name="path">監視パス</param>
        /// <param name="userData">ユーザーデータ(未使用)</param>
        /// <param name="watcherChangeTypes">変更タイプ</param>
        private void OnMonitorEvent(string path, object userData, WatcherChangeTypes watcherChangeTypes)
        {
            // プリミティブの自動リロードがOffになっていたら、何もしない
            if (OptionStore.RootOptions.Basic.Primitives.AutoReload == false)
            {
                return;
            }

            if (this.monitorEventLock)
            {
                return;
            }

            this.emitterSetViewModel.FileMonitor.UnregisterMonitor(this.PrimitiveFilePath);

            this.monitorEventLock = true;

            switch (watcherChangeTypes)
            {
                case WatcherChangeTypes.Changed:
                case WatcherChangeTypes.Created:
                    this.OnReloadPrimitiveFile("PrimitiveFilePath");
                    break;
                case WatcherChangeTypes.Renamed:
                    if (File.Exists(path))
                    {
                        this.OnReloadPrimitiveFile("PrimitiveFilePath");
                    }
                    else
                    {
                        this.UpdateWarningMessages(false);
                        this.UpdatePrimitiveInfo();
                    }

                    break;
                case WatcherChangeTypes.Deleted:
                    this.UpdateWarningMessages(false);
                    this.UpdatePrimitiveInfo();
                    break;
            }

            this.monitorEventLock = false;

            this.emitterSetViewModel.FileMonitor.RegisterMonitorPreCopyPath(this.PrimitiveFilePath, null, this.OnMonitorEvent);
            this.emitterSetViewModel.TriggerPrimitiveFileChanged(this, path);
        }

        /// <summary>
        /// プリミティブファイルに変更があったとき、リロード処理を行った後の処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="path">プリミティブファイルパス</param>
        private void OnPrimitiveFileChanged(object sender, string path)
        {
            if (sender == this || path != this.PrimitiveFilePath)
            {
                return;
            }

            this.UpdateWarningMessages(false);
            this.UpdatePrimitiveInfo();
        }

        /// <summary>
        /// プリミティブファイルのリロード処理を行います。
        /// </summary>
        /// <param name="parameter">Custom parameter</param>
        private void OnReloadPrimitiveFile(object parameter)
        {
            Debug.Assert(parameter is string, "文字列である必要がある");

            var binder = new EffectMaker.Foundation.Dynamic.EffectMakerGetMemberBinder((string)parameter);
            object path = null;
            this.TryGetMember(binder, out path);

            Debug.Assert(path != null, "path != null");

            var pathString = path as string;
            Debug.Assert(string.IsNullOrEmpty(pathString) == false, "string.IsNullOrEmpty(path) == false");

            // リロード要求.
            BusinessLogic.Manager.PrimitiveManager.Instance.Reload(pathString);

            // 警告メッセージを更新
            this.UpdateWarningMessages();
            this.CheckLocatable();

            // UVチャンネル名を更新
            this.UpdatePrimitiveInfo();
        }

        /// <summary>
        /// 警告メッセージを更新します.
        /// </summary>
        /// <param name="notice">親エミッタに通知するかどうかのフラグ</param>
        private void UpdateWarningMessages(bool notice = true)
        {
            this.WarningMessages.Clear();

            bool reachable, locatable;
            AssetsManager.GetAssetStatus(this.PrimitiveFilePath, this.emitterSetViewModel.FilePath, BusinessLogic.AssetResourceTypes.Primitive, out reachable, out locatable);

            // ファイルパスがないとき
            if (reachable == false)
            {
                this.WarningMessages.Add(new Tuple<string, Color, Font>(
                    Resources.EmitterPrimitiveWarningNotFound,
                    Color.Red,
                    null));

                this.PrimitiveFilePathBgColor = Color.LightPink;
            }
            else
            {
                this.PrimitiveFilePathBgColor = Color.White;
            }

            // ロケートできないとき
            if (locatable == false)
            {
                this.WarningMessages.Add(new Tuple<string, Color, Font>(
                    Resources.EmitterPrimitiveWarningUnreachable,
                    Color.SlateBlue,
                    null));

                // unreachableのフラグは立てない
            }

            // ファイルがあるとき
            if (reachable)
            {
                // モデルデータを取得
                Foundation.Model.ModelData modelData = null;
                var primitiveManager = EffectMaker.BusinessLogic.Manager.PrimitiveManager.Instance;
                LoadModelResults result = primitiveManager.LoadModelWithData(this.PrimitiveFilePath, true, out modelData);

                if (result == LoadModelResults.InvalidFileVersion)
                {
                    this.WarningMessages.Add(new Tuple<string, Color, Font>(
                        Resources.EmitterParticleShapeWarningInvalidFileVersion,
                        Color.Red,
                        null));

                    this.PrimitiveFilePathBgColor = Color.LightPink;
                }
                else if (result == LoadModelResults.Success)
                {
                    // uv0, uv1共に存在した場合
                    if ( modelData.TexCoord0.Count > 0 && modelData.TexCoord1.Count > 0)
                    {
                        // uv0とuv1でフォーマットが異なった場合、警告を表示しておく
                        if (modelData.TexCoord0.Type != modelData.TexCoord1.Type ||
                            modelData.TexCoord0.QuantizeType != modelData.TexCoord1.QuantizeType)
                        {
                            this.WarningMessages.Add(new Tuple<string, Color, Font>(
                                Resources.EmitterParticleShapeWarningInvalidUvFormatCombination,
                                Color.Red,
                                null));
                            this.PrimitiveFilePathBgColor = Color.LightPink;
                        }
                    }

                    string invalidVertexNames = string.Empty;

                    // 頂点データのname属性が正しいかチェックする
                    Action<Foundation.Model.VertexElement, bool, int> addInvalidVertexName = (vertexElement, isInValidVertexName, vertexNamesIndex) =>
                    {
                        if ( vertexElement.Count > 0 && isInValidVertexName)
                        {
                            if (string.IsNullOrEmpty(invalidVertexNames) == false)
                            {
                                invalidVertexNames += ", ";
                            }
                            invalidVertexNames += this.vertexNames[vertexNamesIndex];
                        }
                    };

                    addInvalidVertexName(modelData.Position, modelData.PositionName != "_p0", 0);
                    addInvalidVertexName(modelData.Normal, modelData.NormalName != "_n0", 1);
                    addInvalidVertexName(modelData.Tangent, modelData.TangentName != "_t0", 2);
                    // uv0とuv1は、name属性がアセットによって変化する可能性があるので、チェック項目から外しています。
//                    addInvalidVertexName(modelData.TexCoord0, modelData.TexCoordName0 != "_u0", 3);
//                    addInvalidVertexName(modelData.TexCoord1, modelData.TexCoordName1 != "_u1", 4);
                    addInvalidVertexName(modelData.Color, modelData.ColorName != "_c0", 5);

                    // name属性が不正だった場合は、警告を表示する
                    if (string.IsNullOrEmpty(invalidVertexNames) == false)
                    {
                        this.WarningMessages.Add(new Tuple<string, Color, Font>(
                            Resources.EmitterParticleShapeWarningInvalidVertexName + "( " + invalidVertexNames + " )",
                            Color.Red,
                            null));
                        this.PrimitiveFilePathBgColor = Color.LightPink;
                    }
                }
            }

            // ノードアイコンを更新するため、エミッタのアセットステータスをアップデート
            if (notice)
            {
                EmitterViewModel emitterViewModel = ViewModelBase.GetParent<EmitterViewModel>((ViewModelBase)this.Parent);
                emitterViewModel.UpdateAssetStatus(!reachable, false);
            }

            this.OnPropertyChanged(() => this.WarningMessages);
            this.OnPropertyChanged(() => this.PrimitiveFilePathBgColor);
        }

        /// <summary>
        /// プリミティブの情報を更新します。
        /// </summary>
        private void UpdatePrimitiveInfo()
        {
            // モデルデータを取得
            Foundation.Model.ModelData modelData = null;
            if (string.IsNullOrEmpty(this.PrimitiveFilePath) == false && File.Exists(this.PrimitiveFilePath))
            {
                EffectMaker.BusinessLogic.Manager.PrimitiveManager.Instance.LoadModelWithData(this.PrimitiveFilePath, true, out modelData);
            }

            // UVセットの数をチェック
            if (modelData != null && modelData.NumberOfTexCoords > 2)
            {
                Logger.Log("LogView", LogLevels.Warning, Resources.WarningLogPrimitiveUvSetOver);
            }

            // プリミティブ情報を更新
            {
                if (modelData == null)
                {
                    this.PrimitiveProperty = Properties.Resources.EmitterParticleShapeCommentPrimitiveNotLoaded;
                }
                else
                {
                    // 頂点数
                    this.PrimitiveProperty = Properties.Resources.EmitterParticleShapeCommentNumberOfVertex +
                        " " + modelData.Position.Count + Environment.NewLine;

                    // 内包するアセットの一覧
                    bool firstTime = true;
                    this.PrimitiveProperty += Properties.Resources.EmitterParticleShapeCommentAttribute + " ";
                    Action<Foundation.Model.VertexElement, int> addVertexElement = (vertexElement, vertexNamesIndex) =>
                    {
                        if (firstTime == false)
                        {
                            this.PrimitiveProperty += ", ";
                        }
                        if ( vertexElement.Count > 0)
                        {
                            this.PrimitiveProperty += this.vertexNames[vertexNamesIndex];
                            firstTime = false;
                        }
                    };

                    addVertexElement(modelData.Position, 0);
                    addVertexElement(modelData.Normal, 1);
                    addVertexElement(modelData.Tangent, 2);
                    addVertexElement(modelData.TexCoord0, 3);
                    addVertexElement(modelData.TexCoord1, 4);
                    addVertexElement(modelData.Color, 5);

                    this.PrimitiveProperty += Environment.NewLine;

                    // uvチャネルの名前
                    string channelName0 = modelData.TexCoordName0;
                    channelName0 = string.IsNullOrEmpty(channelName0) ? "-" : channelName0;

                    string channelName1 = modelData.TexCoordName1;
                    channelName1 = string.IsNullOrEmpty(channelName1) ? "-" : channelName1;

                    this.PrimitiveProperty += string.Format("{0} ( {1} , {2} )", Properties.Resources.EmitterParticleShapeCommentUvChannelNames, channelName0, channelName1) + Environment.NewLine;

                    // LOD数
                    this.PrimitiveProperty += Properties.Resources.EmitterParticleShapeCommentNumberOfLod + " " + modelData.LodCount;
                }

                this.OnPropertyChanged(() => this.PrimitiveProperty);
            }

            // テクスチャタブのUVチャンネルを更新
            {
                var emitter = GetParent<EmitterViewModel>(this);
                Debug.Assert(emitter != null, "emitter != null");

                // EmitterTextureGroupViewModelよりもこちらが先に作成されるのでnullチェックしておく
                if (emitter.EmitterTextureGroupViewModel != null)
                {
                    emitter.EmitterTextureGroupViewModel.Texture0.EmitterTextureFileViewModel.UpdateUvChannelItems();
                    emitter.EmitterTextureGroupViewModel.Texture1.EmitterTextureFileViewModel.UpdateUvChannelItems();
                    emitter.EmitterTextureGroupViewModel.Texture2.EmitterTextureFileViewModel.UpdateUvChannelItems();
                }
            }
        }

        /// <summary>
        /// テクスチャタブのUVチャンネルの有効/無効を更新します。
        /// </summary>
        private void UpdateEnableUvChannel()
        {
            var emitter = GetParent<EmitterViewModel>(this);
            Debug.Assert(emitter != null, "emitter != null");

            // EmitterTextureGroupViewModelよりもこちらが先に作成されるのでnullチェックしておく
            if (emitter.EmitterTextureGroupViewModel != null)
            {
                emitter.EmitterTextureGroupViewModel.Texture0.EmitterTextureFileViewModel.UpdateEnableUvChannel();
                emitter.EmitterTextureGroupViewModel.Texture1.EmitterTextureFileViewModel.UpdateEnableUvChannel();
                emitter.EmitterTextureGroupViewModel.Texture2.EmitterTextureFileViewModel.UpdateEnableUvChannel();
            }
        }

        /// <summary>
        /// プリミティブがエミッタセットから探索可能かどうかメッセージを出します.
        /// </summary>
        private void CheckLocatable()
        {
            // ファイルパスが指定されていないとき終了
            if (string.IsNullOrEmpty(this.PrimitiveFilePath))
            {
                return;
            }

            // ファイルパスが相対パスのとき終了
            if (Path.IsPathRooted(this.PrimitiveFilePath) == false)
            {
                return;
            }

            bool reachable, locatable;
            AssetsManager.GetAssetStatus(this.PrimitiveFilePath, this.emitterSetViewModel.FilePath, BusinessLogic.AssetResourceTypes.Primitive, out reachable, out locatable);

            // エミッタセットからプリミティブを探索できないとき
            if (locatable == false)
            {
                DialogResult result = WorkspaceRootViewModel.Instance.Dialogs.ShowWarningOnLoadPrimitive(string.Format(Resources.WarningPrimitiveLocatable, Path.GetFileName(this.PrimitiveFilePath)));
            }
        }
    }
}
