﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.Attributes;
using EffectMaker.UILogic.Commands;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterParticleFluctuationViewModel.
    /// </summary>
    public class EmitterParticleFluctuationViewModel : PropertyGroupViewModel<EmitterParticleFluctuationData>, IModificationFlagOwner
    {
        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignorePropertyNames = new string[]
        {
            "WaveTypeVisibility",
            "SettingVisibility",
            "ScaleYVisibility",
        };

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterParticleFluctuationViewModel(
            HierarchyViewModel parent, EmitterParticleFluctuationData dataModel)
            : base(parent, dataModel)
        {
            this.WaveTypeItems = new KeyValuePair<string, object>[]
            {
                new KeyValuePair<string, object>(Properties.Resources.EmitterWaveTypeSinCurve, 1 << 3),
                new KeyValuePair<string, object>(Properties.Resources.EmitterWaveTypeSawTooth, 1 << 4),
                new KeyValuePair<string, object>(Properties.Resources.EmitterWaveTypeSquare, 1 << 5),
            };

            var applyParamsProperties = EnumerableUtility.Enumerate(
                ViewModelBase.NameOf(() => dataModel.ApplyAlpha),
                ViewModelBase.NameOf(() => dataModel.ApplyScale));

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            var mmfvm = new MultipleModificationFlagsViewModel(this);
            mmfvm.SetPropertyDictionary("ApplyParams", applyParamsProperties);
            this.ModificationFlagViewModel = mmfvm;

            this.ModificationFlagViewModel.AddIgnoreProperties(this.ignorePropertyNames);
        }

        /// <summary>
        /// 波形タイプの項目を取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> WaveTypeItems { get; private set; }

        /// <summary>
        /// 振幅X
        /// </summary>
        public float AmplitudeX
        {
            get
            {
                return this.Amplitude.X;
            }

            set
            {
                var amplitude = new Vector2f(value, this.Amplitude.Y);
                var binder = new EffectMaker.Foundation.Dynamic.EffectMakerSetMemberBinder(
                    "Amplitude", false, false);
                this.TrySetMember(binder, amplitude);
                this.OnPropertyChanged();
            }
        }

        /// <summary>
        /// 振幅Y
        /// </summary>
        public float AmplitudeY
        {
            get
            {
                return this.Amplitude.Y;
            }

            set
            {
                var amplitude = new Vector2f(this.Amplitude.X, value);
                var binder = new EffectMaker.Foundation.Dynamic.EffectMakerSetMemberBinder(
                    "Amplitude", false, false);
                this.TrySetMember(binder, amplitude);
                this.OnPropertyChanged();
            }
        }

        /// <summary>
        /// 周期X
        /// </summary>
        public int FrequencyX
        {
            get
            {
                return this.Frequency.X;
            }

            set
            {
                var frequency = new Vector2i(value, this.Frequency.Y);
                var binder = new EffectMaker.Foundation.Dynamic.EffectMakerSetMemberBinder(
                    "Frequency", false, false);
                this.TrySetMember(binder, frequency);
                this.OnPropertyChanged();
            }
        }

        /// <summary>
        /// 周期Y
        /// </summary>
        public int FrequencyY
        {
            get
            {
                return this.Frequency.Y;
            }

            set
            {
                var frequency = new Vector2i(this.Frequency.X, value);
                var binder = new EffectMaker.Foundation.Dynamic.EffectMakerSetMemberBinder(
                    "Frequency", false, false);
                this.TrySetMember(binder, frequency);
                this.OnPropertyChanged();
            }
        }

        /// <summary>
        /// 初期位相X
        /// </summary>
        public int InitPhaseX
        {
            get
            {
                return this.InitPhase.X;
            }

            set
            {
                var initPhase = new Vector2i(value, this.InitPhase.Y);
                var binder = new EffectMaker.Foundation.Dynamic.EffectMakerSetMemberBinder(
                    "InitPhase", false, false);
                this.TrySetMember(binder, initPhase);
                this.OnPropertyChanged();
            }
        }

        /// <summary>
        /// 初期位相Y
        /// </summary>
        public int InitPhaseY
        {
            get
            {
                return this.InitPhase.Y;
            }

            set
            {
                var initPhase = new Vector2i(this.InitPhase.X, value);
                var binder = new EffectMaker.Foundation.Dynamic.EffectMakerSetMemberBinder(
                    "InitPhase", false, false);
                this.TrySetMember(binder, initPhase);
                this.OnPropertyChanged();
            }
        }

        /// <summary>
        /// 位相ランダムX
        /// </summary>
        public bool PhaseRandomX
        {
            get
            {
                return this.PhaseRandom.X != 0;
            }

            set
            {
                int internalValue = value ? 1 : 0;
                if (this.PhaseRandom.X == internalValue)
                {
                    return;
                }

                var phaseRandom = new Vector2i(internalValue, this.PhaseRandom.Y);
                var binder = new EffectMaker.Foundation.Dynamic.EffectMakerSetMemberBinder(
                    "PhaseRandom", false, false);
                this.TrySetMember(binder, phaseRandom);
                this.OnPropertyChanged();
            }
        }

        /// <summary>
        /// 位相ランダムY
        /// </summary>
        public bool PhaseRandomY
        {
            get
            {
                return this.PhaseRandom.Y != 0;
            }

            set
            {
                int internalValue = value ? 1 : 0;
                if (this.PhaseRandom.Y == internalValue)
                {
                    return;
                }

                var phaseRandom = new Vector2i(this.PhaseRandom.X, internalValue);
                var binder = new EffectMaker.Foundation.Dynamic.EffectMakerSetMemberBinder(
                    "PhaseRandom", false, false);
                this.TrySetMember(binder, phaseRandom);
                this.OnPropertyChanged();
            }
        }

        /// <summary>
        /// アルファに適用.
        /// </summary>
        [UseDataModelOriginalValue]
        public bool ApplyAlpha
        {
            get
            {
                return this.GetDataModelValue(() => this.ApplyAlpha);
            }

            set
            {
                this.SetDataModelValue(value, () => this.ApplyAlpha);

                // プロパティ変更通知.
                this.OnPropertyChanged(() => this.SettingVisibility);
            }
        }

        /// <summary>
        /// スケールに適用.
        /// </summary>
        [UseDataModelOriginalValue]
        public bool ApplyScale
        {
            get
            {
                return this.GetDataModelValue(() => this.ApplyScale);
            }

            set
            {
                this.SetDataModelValue(value, () => this.ApplyScale);

                // プロパティ変更通知.
                this.OnPropertyChanged(() => this.SettingVisibility);
                this.OnPropertyChanged(() => this.ScaleYVisibility);
            }
        }

        /// <summary>
        /// スケールを個別に設定.
        /// </summary>
        [UseDataModelOriginalValue]
        public bool ApplyScaleY
        {
            get
            {
                return this.GetDataModelValue(() => this.ApplyScaleY);
            }

            set
            {
                this.SetDataModelValue(value, () => this.ApplyScaleY);

                // プロパティ変更通知.
                this.OnPropertyChanged(() => this.ScaleYVisibility);
            }
        }

        /// <summary>
        /// 設定表示の表示状態.
        /// </summary>
        public bool SettingVisibility
        {
            get { return this.ApplyScale || this.ApplyAlpha; }
        }

        /// <summary>
        /// スケールを個別に設定の表示状態.
        /// </summary>
        public bool ScaleYVisibility
        {
            get { return this.ApplyScale && this.ApplyScaleY; }
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// 振幅.
        /// </summary>
        private Vector2f Amplitude
        {
            get { return this.GetDataModelValue(() => this.Amplitude); }
        }

        /// <summary>
        /// 周期.
        /// </summary>
        private Vector2i Frequency
        {
            get { return this.GetDataModelValue(() => this.Frequency); }
        }

        /// <summary>
        /// 初期位相.
        /// </summary>
        private Vector2i InitPhase
        {
            get { return this.GetDataModelValue(() => this.InitPhase); }
        }

        /// <summary>
        /// 位相ランダム.
        /// </summary>
        private Vector2i PhaseRandom
        {
            get { return this.GetDataModelValue(() => this.PhaseRandom); }
        }
    }
}
